/*
 * $Header: /var/cvs/ins/libs/remotetea/src/org/acplt/oncrpc/apps/jrpcgen/jrpcgen.java,v 1.1.1.1 2001/11/02 01:55:30 mbalazin Exp $
 *
 * Copyright (c) 1999, 2000
 * Lehrstuhl fuer Prozessleittechnik (PLT), RWTH Aachen
 * D-52064 Aachen, Germany.
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program (see the file COPYING.LIB for more
 * details); if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

//
// Personal note: this class probably suffers from a flashback on
// procedural programming ... but where do we need to be today?
//

package org.acplt.oncrpc.apps.jrpcgen;

import org.acplt.oncrpc.apps.jrpcgen.cup_runtime.Symbol;

import java.io.*;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Date;
import java.text.SimpleDateFormat;

/**
 * The class <code>jrpcgen</code> implements a Java-based rpcgen RPC protocol
 * compiler. jrpcgen is a Java-based tool that generates source code of Java
 * classes to implement an RPC protocol. The input to jrpcgen is a language
 * similiar to C (but more probably much more similiar to FORTRAN) known as
 * the RPC language (Remote Procedure Call Language).
 *
 * @version $Revision: 1.1.1.1 $ $Date: 2001/11/02 01:55:30 $ $State: Exp $ $Locker:  $
 * @author Harald Albrecht
 */
public class jrpcgen {

    /**
     * Print the help message describing the available command line options.
     */
    public static void printHelp() {
        System.out.println("Usage: jrpcgen [-options] x-file");
        System.out.println();
        System.out.println("where options include:");
        System.out.println("  -c <classname>  specify class name of client proxy stub");
        System.out.println("  -d <dir>        specify directory where to place generated source code files");
        System.out.println("  -p <package>    specify package name for generated source code files");
        System.out.println("  -s <classname>  specify class name of server proxy stub");
        System.out.println("  -nobackup       do not make backups of old source code files");
        System.out.println("  -noclient       do not create client proxy stub");
        System.out.println("  -noserver       do not create server proxy stub");
        System.out.println("  -parseonly      parse x-file only but do not create source code files");
        System.out.println("  -verbose        enable verbose output about what jrpcgen is doing");
        System.out.println("  -version        print jrpcgen version and exit");
        System.out.println("  -debug          enables printing of diagnostic messages");
        System.out.println("  -? -help        print this help message and exit");
        System.out.println("  --              end options");
        System.out.println();
    }

    /**
     * Current version of jrpcgen.
     */
    public static final String VERSION = "0.93.0";

    /**
     * String containing date/time when a jrpcgen run was started. This string
     * is used in the headers of the generated source code files.
     */
    public static final String startDate =
        (new SimpleDateFormat()).format(new Date());

    /**
     * Contains all global identifiers for type, structure and union specifiers
     * as well as for constants and enumeration members. This static attribute
     * is directly manipulated by parser.
     */
    public static Hashtable globalIdentifiers = new Hashtable();

    /**
     * Disable automatic backup of old source code files, if <code>true</code>.
     */
    public static boolean noBackups = false;

    /**
     * Holds information about the remote program defined in the jrpcgen
     * x-file.
     */
    public static Vector programInfos = null;

    /**
     * Enable diagnostic messages when parsing the x-file.
     */
    public static boolean debug = false;

    /**
     * Verbosity flag. If <code>true</code>, then jrpcgen will report about
     * the steps it is taking when generating all the source code files.
     */
    public static boolean verbose = false;

    /**
     * Parse x-file only but do not create source code files if set to
     * <code>true</code>.
     */
    public static boolean parseOnly = false;

    /**
     * Destination directory where to place the generated files.
     */
    public static File destinationDir = new File(".");

    /**
     * Current FileWriter object receiving generated source code.
     */
    public static Writer currentFileWriter = null;

    /**
     * Current PrintWriter object sitting on top of the
     * {@link #currentFileWriter} object receiving generated source code.
     */
    public static PrintWriter currentPrintWriter = null;

    /**
     * Full name of the current source code file.
     */
    public static String currentFilename = null;

    /**
     * Specifies package name for generated source code, if not
     * <code>null</code>. If <code>null</code>, then no package statement
     * is emitted.
     */
    public static String packageName = null;

    /**
     * Name of class containing global constants. It is derived from the
     * filename with the extension (".x") and path removed.
     */
    public static String baseClassname = null;

    /**
     * Do not generate source code for the client proxy stub if
     * <code>true</code>.
     */
    public static boolean noClient = false;

    /**
     * Do not generate source code for the server proxy stub if
     * <code>true</code>.
     */
    public static boolean noServer = false;

    /**
     * Name of class containing the ONC/RPC server stubs.
     */
    public static String serverClass = null;

    /**
     * Name of class containing the ONC/RPC client stubs.
     */
    public static String clientClass = null;

    /**
     * Creates a new source code file for a Java class based on its class
     * name. If an old version of the source file exists, it is renamed first.
     * The backup will have the same name as the original file with "~"
     * appended.
     *
     * @param classname Name of Java class to generate. Must not contain
     *   a file extension -- especially ".java" is invalid. When the source
     *   code file is created, ".java" is appended automatically.
     *
     * @return PrintWriter to send source code to.
     */
    public static PrintWriter createJavaSourceFile(String classname) {
        String filename = classname + ".java";
        File file = new File(destinationDir, filename);
        //
        // If an old file of the same name already exists, then rename it
        // before creating the new file.
        //
        if ( file.exists() && !noBackups ) {
            if ( !file.isFile() ) {
                //
                // If the file to be created already exists and is not a
                // regular file, then bail out with an error.
                //
                System.err.println("error: source file \"" + filename
                                   + "\"already exists and is not a regular file");
                System.exit(1);
            }
            File oldBackup = new File(destinationDir, filename + "~");
            if ( oldBackup.isFile() ) {
                oldBackup.delete();
            } else if ( oldBackup.exists() ) {
                System.err.println("error: backup source file \""
                                   + filename + "~\" is not a regular file");
                System.exit(1);
            }
            if ( !file.renameTo(new File(destinationDir, filename + "~")) ) {
                System.err.println("error: can not rename old source code file \""
                                   + filename + "\"");
                System.exit(1);
            }
            if ( verbose ) {
                System.out.println("Saved old source code file as \""
                                   + filename + "~\"");
            }
        }
        //
        // Now create a new source code file...
        //
        try {
            currentFileWriter = new FileWriter(file);
        } catch ( IOException e ) {
            System.err.println("error: can not create \"" + filename
                               + "\": " + e.getLocalizedMessage());
            System.exit(1);
        }
        if ( verbose ) {
            System.out.print("Creating source code file \""
                             + filename + "\"...");
        }
        currentFilename = filename;
        PrintWriter out = new PrintWriter(currentFileWriter, true);
        currentPrintWriter = out;
        //
        // Create automatic header(s)...
        // Note that we always emit the import statements, regardless of
        // whether we're emitting a class file or an interface file consisting
        // of an enumeration.
        //
        out.println("/*");
        out.println(" * Automatically generated by jrpcgen " + VERSION
                    + " on " + startDate);
        out.println(" * jrpcgen is part of the \"Remote Tea\" ONC/RPC package for Java");
        out.println(" * See http://acplt.org/ks/remotetea.html for details");
        out.println(" */");

        //
        // Only generated package statement if a package name has been specified.
        //
        if ( (packageName != null) && (packageName.length() > 0) ) {
            out.println("package " + packageName + ";");
        }

        out.println("import org.acplt.oncrpc.*;");
        out.println("import java.io.IOException;");
        out.println();

        return out;
    }

    /**
     * Closes the source code file previously opened with
     * <code>createJavaSourceFile</code>. This method writes a trailer
     * before closing the file.
     */
    public static void closeJavaSourceFile() {
        //
        // Create automatic footer before closing the file.
        //
        currentPrintWriter.println("// End of " + currentFilename);
        if ( verbose ) {
            System.out.println();
        }
        try {
            currentPrintWriter.close();
            currentFileWriter.close();
        } catch ( IOException e ) {
            System.err.println("Can not close source code file: "
                               + e.getLocalizedMessage());
        }
    }


    /**
     * Dump the value of a constant and optionally first dump all constants
     * it depends on.
     */
    public static void dumpConstantAndDependency(PrintWriter out, JrpcgenConst c) {
        //
        // This simple test avoids endless recursions.
        //
        if ( c.dontTraverseAnyMore ) {
            return;
        }
        c.dontTraverseAnyMore = true;
        String dependencyIdentifier = c.getDependencyIdentifier();
        if ( dependencyIdentifier != null ) {
            //
            // There is a dependency, so try to resolve that first.
            //
            JrpcgenConst dc = (JrpcgenConst) globalIdentifiers.get(dependencyIdentifier);
            if ( dc != null ) {
                dumpConstantAndDependency(out, dc);
                if ( !c.enclosure.equalsIgnoreCase(dc.enclosure) ) {
                    //
                    // In case we depend on a constant which belongs to a
                    // different enclosure then also dump the enclosure.
                    // Note that this code depends on the "value" starts
                    // with the identifier we depend on (which is currently
                    // the case), so we just need to prepend the enclosure.
                    //
                    out.println("    public static final int "
                                + c.identifier
                                + " = " + dc.enclosure + "." + c.value + ";");
                    return;
                }
            }
        }
        //
        // Just dump the plain value.
        //
        out.println("    public static final int "
                    + c.identifier
                    + " = " + c.value + ";");
    }


    /**
     * Generate source code file containing all constants defined in the
     * x-file as well as all implicitely defined constants, like program,
     * version and procedure numbers, etc. This method creates a public
     * interface with the constants as public static final integers.
     */
    public static void dumpConstants() {
        //
        // Create new source code file containing a Java interface representing
        // all XDR constants.
        //
        PrintWriter out = createJavaSourceFile(baseClassname);
        //
        // Spit out some description for javadoc & friends...
        //
        out.println("/**");
        out.println(" * A collection of constants used by the \"" + baseClassname
                    + "\" ONC/RPC program.");
        out.println(" */");
        out.println("public interface " + baseClassname + " {");

        Enumeration globals = globalIdentifiers.elements();
        while ( globals.hasMoreElements() ) {
            Object o = globals.nextElement();
            if ( o instanceof JrpcgenConst ) {
                JrpcgenConst c = (JrpcgenConst) o;
                //
                // Dump only such constants which belong to the global
                // constants enclosure. Ignore all other constants, as those
                // belong to other Java class enclosures.
                //
                if ( baseClassname.equals(c.enclosure) ) {
                    dumpConstantAndDependency(out, c);
                }
            }
        }

        out.println("}");
        closeJavaSourceFile();
    }

    /**
     * Generate a source code file containing all elements of an enumeration
     * defined in a x-file.
     *
     * @param e {@link JrpcgenEnum Description} of XDR enumeration.
     */
    public static void dumpEnum(JrpcgenEnum e) {
        //
        // Create new source code file containing a Java interface representing
        // the XDR enumeration.
        //
        PrintWriter out = createJavaSourceFile(e.identifier);
        //
        // Spit out some description for javadoc & friends...
        //
        out.println("/**");
        out.println(" * Enumeration (collection of constants).");
        out.println(" */");

        out.println("public interface " + e.identifier + " {");
        out.println();

        Enumeration enums = e.enums.elements();
        while ( enums.hasMoreElements() ) {
            JrpcgenConst c = (JrpcgenConst) enums.nextElement();
            //
            // In case an element depends on a global constant, then
            // this constant will automatically be duplicated as part
            // of this enumeration.
            //
            dumpConstantAndDependency(out, c);
        }
        //
        // Close class...
        //
        out.println();
        out.println("}");
        closeJavaSourceFile();
    }

    /**
     * Java base data types for which are XDR encoding and decoding helper
     * methods available.
     */
    private static String [] baseTypes = {
        "void",
        "boolean",
        "byte",
        "short", "int", "long",
        "float", "double",
        "String"
    };

    /**
     * NOTE: "opaque" is considered like "byte" to be a base type...
     * FIXME: char/byte?
     */
    public static String xdrBaseType(String type) {
        int size = baseTypes.length;
        if ( "opaque".compareTo(type) == 0 ) {
            type = "byte";
        }
        for ( int idx = 0; idx < size; ++idx ) {
            if ( baseTypes[idx].compareTo(type) == 0 ) {
                //
                // For base data types simply convert the first letter to
                // an upper case letter.
                //
                return "Xdr" + type.substring(0, 1).toUpperCase()
                           + type.substring(1);
            }
        }
        return null;
    }

    /**
     * Return the en-/decoding syllable XXX appropriate for a base data
     * type including arrays of base data types.
     *
     * @param decl declaration of a member of RPC struct or union.
     *
     * @return <code>null</code>, if the declaration does not specify a base data
     *   type. Otherwise a three-element String array, with [0] containing
     *   the type syllable for base type (including arrays), [1] containing
     *   parameter options when encoding (like maximum sizes, etc), and [2]
     *   containing options for decoding.
     */
    public static JrpcgenEnDecodingInfo baseEnDecodingSyllable(JrpcgenDeclaration decl) {
        String syllable = decl.type;
        boolean isBase = false;
        //
        // Check for Java base data types... if a match is found, then convert
        // the data type name, so that it becomes a valid syllable for use
        // with XDR en-/decoding functions xdrEncodingXXX() etc.
        // Example: "int" --> "Int" (because of xdrEncodingInt())
        // NOTE: we consider "opaque" to be a base type here...
        //
        int size = baseTypes.length;
        String type = decl.type;
        if ( "opaque".compareTo(type) == 0 ) {
            type = "byte";
        }
        for ( int idx = 0; idx < size; ++idx ) {
            if ( baseTypes[idx].compareTo(type) == 0 ) {
                //
                // For base data types simply convert the first letter to
                // an upper case letter.
                //
                isBase = true;
                syllable = syllable.substring(0, 1).toUpperCase()
                           + syllable.substring(1);
                break;
            }
        }
        //
        // Handle special case of enumerations, which have to be represented
        // using ints in the Java language.
        //
        if ( !isBase ) {
            Object o = globalIdentifiers.get(decl.type);
            if ( o instanceof JrpcgenEnum ) {
                isBase = true;
                syllable = "Int";
            }
        }
        //
        // In case we're dealing with an array, then add "Vector" to
        // the syllable to use the appropriate vector en-/decoding method
        // for base data types.
        // NOTE: unfortunately, strings do not adhere to this scheme, as
        // they are considered to be arrays of characters... what a silly
        // idea, as this makes a typedef necessary in case someone needs
        // an array of strings.
        // NOTE: also opaques break the scheme somehow, but the char=byte
        // versus opaque schisma anyhow drives me crazy...
        //
        if ( isBase ) {
            String encodingOpts = null;
            String decodingOpts = null;

            if ( (decl.kind == JrpcgenDeclaration.FIXEDVECTOR)
                 || (decl.kind == JrpcgenDeclaration.DYNAMICVECTOR) ) {
                if ( "opaque".equals(decl.type) ) {
                    if ( decl.kind == JrpcgenDeclaration.FIXEDVECTOR ) {
                        syllable = "Opaque";
                        encodingOpts = checkForEnumValue(decl.size);
                        decodingOpts = checkForEnumValue(decl.size);
                    } else {
                        syllable = "DynamicOpaque";
                        encodingOpts = null;
                        decodingOpts = null;
                    }
                } else if ( !"String".equals(decl.type) ) {
                    if ( decl.kind == JrpcgenDeclaration.FIXEDVECTOR ) {
                        syllable = syllable + "Fixed";
                        encodingOpts = checkForEnumValue(decl.size);
                        decodingOpts = checkForEnumValue(decl.size);
                    }
                    syllable = syllable + "Vector";
                }
            }

            JrpcgenEnDecodingInfo result = new
                JrpcgenEnDecodingInfo(syllable, encodingOpts, decodingOpts);
            return result;
        }
        return null;
    }

    /**
     * Return en- or decoding method appropriate for a struct or union member.
     */
    public static String codingMethod(JrpcgenDeclaration decl, boolean encode) {
        //
        // Skip entries for void arms etc...
        //
        if ( decl.identifier == null ) {
            return "";
        }

        StringBuffer code = new StringBuffer();
        JrpcgenEnDecodingInfo data = baseEnDecodingSyllable(decl);

        if ( data != null ) {
            //
            // It's a base data type (including vectors). So we can use the
            // predefined en-/decoding methods:
            //   - xdr.xdrEncodeXXX(value);
            //   - value = xdr.xdrDecodeXXX(value);
            //
            if ( encode ) {
                code.append("        xdr.xdrEncode");
                code.append(data.syllable);
                code.append("(");
                code.append(decl.identifier);
                if ( data.encodingOptions != null ) {
                    code.append(", ");
                    code.append(data.encodingOptions);
                }
                code.append(");\n");
            } else {
                code.append("        ");
                code.append(decl.identifier);
                code.append(" = xdr.xdrDecode");
                code.append(data.syllable);
                code.append("(");
                if ( data.decodingOptions != null ) {
                    code.append(data.decodingOptions);
                }
                code.append(");\n");
            }
            return code.toString();
        } else {
            //
            // It's not a built-in base data type but instead something that
            // is represented by a class.
            //   - foo.xdrEncode(xdr);
            //   - foo = new FOO();
            //     foo.xdrDecode(xdr);
            // In case of arrays, this is going to be hairy...
            //
            if ( decl.kind == JrpcgenDeclaration.SCALAR ) {
                code.append("        ");
                if ( encode ) {
                    code.append(decl.identifier);
                    code.append(".xdrEncode(xdr);\n");
                } else {
                    code.append(decl.identifier);
                    code.append(" = new ");
                    code.append(decl.type);
                    code.append("(xdr);\n");
                }
                return code.toString();
            //
            // It's not a built-in base data type but instead an indirection
            // (reference) to some instance (optional data).
            //
            } else if ( decl.kind == JrpcgenDeclaration.INDIRECTION ) {
                code.append("        ");
                if ( encode ) {
                    code.append("if ( ");
                    code.append(decl.identifier);
                    code.append(" != null ) { ");
                    code.append("xdr.xdrEncodeBoolean(true); ");
                    code.append(decl.identifier);
                    code.append(".xdrEncode(xdr);");
                    code.append(" } else { ");
                    code.append("xdr.xdrEncodeBoolean(false);");
                    code.append(" };\n");
                } else {
                    code.append("if ( xdr.xdrDecodeBoolean() ) { ");
                    code.append(decl.identifier);
                    code.append(" = new ");
                    code.append(decl.type);
                    code.append("(xdr);");
                    code.append(" } else { ");
                    code.append(decl.identifier);
                    code.append(" = null;");
                    code.append(" };\n");
                }
                return code.toString();
            }
            //
            // Array case... Urgh!
            //
            if ( encode ) {
                code.append("        { ");
                code.append("int $size = ");
                if ( decl.kind == JrpcgenDeclaration.DYNAMICVECTOR ) {
                    //
                    // Dynamic array size. So we need to use the current size
                    // of the Java array.
                    //
                    code.append(decl.identifier);
                    code.append(".length");
                } else {
                    code.append(checkForEnumValue(decl.size));
                }
                code.append("; ");
                if ( decl.kind == JrpcgenDeclaration.DYNAMICVECTOR ) {
                    //
                    // Dynamic array size. So we need to encode size information.
                    //
                    code.append("xdr.xdrEncodeInt($size); ");
                }
                //
                // Now encode all elements.
                //
                code.append("for ( int $idx = 0; $idx < $size; ++$idx ) { ");
                code.append(decl.identifier);
                code.append("[$idx].xdrEncode(xdr); ");
                code.append("} }\n");
            } else {
                code.append("        { ");
                code.append("int $size = ");
                if ( decl.kind == JrpcgenDeclaration.DYNAMICVECTOR ) {
                    //
                    // Dynamic array size. So we need to decode size information.
                    //
                    code.append("xdr.xdrDecodeInt()");
                } else {
                    code.append(checkForEnumValue(decl.size));
                }
                code.append("; ");
                //
                // Now encode all elements.
                //
                code.append(decl.identifier);
                code.append(" = new ");
                code.append(decl.type);
                code.append("[$size]; ");
                code.append("for ( int $idx = 0; $idx < $size; ++$idx ) { ");
                code.append(decl.identifier);
                code.append("[$idx] = new ");
                code.append(decl.type);
                code.append("(xdr); ");
                code.append("} }\n");
            }
            return code.toString();
        }
    }

    /**
     * Checks whether a given data type identifier refers to an enumeration
     * type and then returns Java's int data type instead. In case of the
     * pseudo-type "opaque" return Java's byte data type. For all other
     * data types, the data type identifier is returned unaltered.
     *
     * @param dataType data type identifier to check.
     *
     * @result data type identifier.
     */
    public static String checkForSpecials(String dataType) {
        if ( globalIdentifiers.get(dataType) instanceof JrpcgenEnum ) {
            return "int";
        } else if ( "opaque".equals(dataType) ) {
            return "byte";
        }
        return dataType;
    }

    /**
     * Checks whether a given value references an identifier and then
     * returns the qualified identifier (interface where the value is
     * defined in) or simply the value in case of an integer literal.
     *
     * @param value Either an identifier to resolve or an integer literal.
     *
     * @return Integer literal or qualified identifier.
     */
    public static String checkForEnumValue(String value) {
        if ( value.length() > 0 ) {
            //
            // If the value is an integer literal, then we just have to
            // return it. That's it.
            //
            if ( Character.isDigit(value.charAt(0))
                 || (value.charAt(0) == '-') ) {
                return value;
            }
            //
            // It's an identifier: we now need to find out in which
            // enclosure it lives, so we can return a qualified identifier.
            //
            Object id = jrpcgen.globalIdentifiers.get(value);
            if ( (id != null)
                 && (id instanceof JrpcgenConst) ) {
                JrpcgenConst c = (JrpcgenConst) id;
                if ( c.enclosure == null ) {
                    return c.value;
                }
                return c.enclosure + "." + c.identifier;
            }
        }
        return value;
    }

    /**
     * Generate a source code file containing all elements of a struct
     * defined in a x-file.
     *
     * @param s {@link JrpcgenStruct Description} of XDR struct.
     */
    public static void dumpStruct(JrpcgenStruct s) {
        //
        // Create new source code file containing a Java class representing
        // the XDR struct.
        //
        PrintWriter out = createJavaSourceFile(s.identifier);

        out.println("public class " + s.identifier + " implements XdrAble {");

        Enumeration decls = s.elements.elements();
        while ( decls.hasMoreElements() ) {
            JrpcgenDeclaration d = (JrpcgenDeclaration) decls.nextElement();
            out.print("    public " + checkForSpecials(d.type) + " ");
        if ( ((d.kind == JrpcgenDeclaration.FIXEDVECTOR)
              || (d.kind == JrpcgenDeclaration.DYNAMICVECTOR))
             && !d.type.equals("String") ) {
                out.print("[] ");
            }
            out.println(d.identifier + ";");
        }

        //
        // Now generate code for encoding and decoding this class (structure).
        //
        out.println();
        out.println("    public " + s.identifier + "() {");
        out.println("    }");
        out.println();
        out.println("    public " + s.identifier + "(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        xdrDecode(xdr);");
        out.println("    }");
        out.println();
        out.println("    public void xdrEncode(XdrEncodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        decls = s.elements.elements();
        while ( decls.hasMoreElements() ) {
            out.print(codingMethod((JrpcgenDeclaration) decls.nextElement(), true));
        }
        out.println("    }");

        out.println();
        out.println("    public void xdrDecode(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        decls = s.elements.elements();
        while ( decls.hasMoreElements() ) {
            out.print(codingMethod((JrpcgenDeclaration) decls.nextElement(), false));
        }
        out.println("    }");
        //
        // Close class...
        //
        out.println();
        out.println("}");
        closeJavaSourceFile();
    }

    /**
     * Generate a source code file containing all elements of a union
     * defined in a x-file.
     *
     * @param u {@link JrpcgenUnion Description} of XDR union.
     */
    public static void dumpUnion(JrpcgenUnion u) {
        //
        // Create new source code file containing a Java class representing
        // the XDR union.
        //
        PrintWriter out = createJavaSourceFile(u.identifier);

        out.println("public class " + u.identifier + " implements XdrAble {");
        //
        // Note that the descriminant can not be of an array type, string, etc.
        // so we don't have to handle all the special cases here.
        //
        out.println("    public " + checkForSpecials(u.descriminant.type) + " "
                    + u.descriminant.identifier + ";");

        boolean boolDescriminant = u.descriminant.type.equals("boolean");

        Enumeration arms = u.elements.elements();
        while ( arms.hasMoreElements() ) {
            JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
            //
            // Skip all arms which do not contain a variable but are
            // declared as "void" instead. Also skip all arms which are
            // mapped to another arm.
            //
            if ( (a.element == null) || (a.element.identifier == null) ) {
                continue;
            }
            out.print("    public " + checkForSpecials(a.element.type) + " ");
            if ( ((a.element.kind == JrpcgenDeclaration.FIXEDVECTOR)
                  || (a.element.kind == JrpcgenDeclaration.DYNAMICVECTOR))
                 && !a.element.type.equals("String") ) {
                out.print("[] ");
            }
            out.println(a.element.identifier + ";");
        }

        //
        // Now generate code for encoding and decoding this class (structure).
        //
        out.println();
        out.println("    public " + u.identifier + "() {");
        out.println("    }");
        out.println();
        out.println("    public " + u.identifier + "(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        xdrDecode(xdr);");
        out.println("    }");
        out.println();
        out.println("    public void xdrEncode(XdrEncodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.print(codingMethod(u.descriminant, true));
        if ( !boolDescriminant ) {
            //
            // Produce code using an ordinary switch statement...
            //
            out.println("        switch ( " + u.descriminant.identifier + " ) {");
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
                if ( a.value != null ) {
                    out.println("        case " + checkForEnumValue(a.value) + ":");
                } else {
                    //
                    // It's the default arm.
                    //
                    out.println("        default:");
                }
                //
                // Only emit code if arm does not map to another arm.
                //
                if ( a.element != null ) {
                    if ( a.element.identifier != null ) {
                        //
                        // Arm does not contain void, so we need to spit out
                        // encoding instructions.
                        //
                        out.print("    ");
                        out.print(codingMethod(a.element, true));
                    }
                    out.println("            break;");
                }
            }
            out.println("        }");
        } else {
            //
            // boolean descriminant: here we can have at most two arms, guess
            // why.
            //
            boolean firstArm = true;
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
                if ( a.value == null ) {
                    //
                    // Skip default branch this time...
                    //
                    continue;
                }
                if ( a.element.identifier != null ) {
                    //
                    // Arm contains data, so we need to create encoding
                    // instructions.
                    //
                    out.print("        ");
                    if ( !firstArm ) {
                        out.print("else ");
                    } else {
                        firstArm = false;
                    }
                    out.println("if ( " + u.descriminant.identifier +
                                  " == " + checkForEnumValue(a.value) + " ) {");
                    out.print("    ");
                    out.print(codingMethod(a.element, true));
                    out.println("        }");
                }
            }
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
                if ( (a.value == null) && (a.element.identifier != null) ) {
                    out.print("        ");
                    if ( !firstArm ) {
                        out.print("else ");
                    }
                    out.println("{");
                    out.print("    ");
                    out.print(codingMethod(a.element, true));
                    out.println("        }");
                }
            }
        }
        out.println("    }");

        out.println();
        out.println("    public void xdrDecode(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.print(codingMethod(u.descriminant, false));
        if ( !boolDescriminant ) {
            //
            // Produce code using an ordinary switch statement...
            //
            out.println("        switch ( " + u.descriminant.identifier + " ) {");
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();

                if ( a.value != null ) {
                    out.println("        case " + checkForEnumValue(a.value) + ":");
                } else {
                    //
                    // It's the default arm.
                    //
                    out.println("        default:");
                }
                //
                // Only emit code if arm does not map to another arm.
                //
                if ( a.element != null ) {
                    if ( a.element.identifier != null ) {
                        //
                        // Arm does not contain void, so we need to spit out
                        // encoding instructions.
                        //
                        out.print("    ");
                        out.print(codingMethod(a.element, false));
                    }
                    out.println("            break;");
                }
            }
            out.println("        }");
        } else {
            //
            // boolean descriminant: here we can have at most two arms, guess
            // why.
            //
            boolean firstArm = true;
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
                if ( a.value == null ) {
                    //
                    // Skip default branch this time...
                    //
                    continue;
                }
                if ( a.element.identifier != null ) {
                    //
                    // Arm contains data, so we need to create encoding
                    // instructions.
                    //
                    out.print("        ");
                    if ( !firstArm ) {
                        out.print("else ");
                    } else {
                        firstArm = false;
                    }
                    out.println("if ( " + u.descriminant.identifier +
                                  " == " + checkForEnumValue(a.value) + " ) {");
                    out.print("    ");
                    out.print(codingMethod(a.element, false));
                    out.println("        }");
                }
            }
            arms = u.elements.elements();
            while ( arms.hasMoreElements() ) {
                JrpcgenUnionArm a = (JrpcgenUnionArm) arms.nextElement();
                if ( (a.value == null) && (a.element.identifier != null) ) {
                    out.print("        ");
                    if ( !firstArm ) {
                        out.print("else ");
                    }
                    out.println("{");
                    out.print("    ");
                    out.print(codingMethod(a.element, false));
                    out.println("        }");
                }
            }
        }
        out.println("    }");
        //
        // Close class...
        //
        out.println();
        out.println("}");
        closeJavaSourceFile();
    }

    /**
     * Generate a source code file containing a wrapper class for a typedef
     * defined in a x-file.
     *
     * @param d {@link JrpcgenDeclaration Description} of XDR typedef.
     */
    public static void dumpTypedef(JrpcgenDeclaration d) {
        //
        // Create new source code file containing a Java class representing
        // the XDR struct.
        //
        PrintWriter out = createJavaSourceFile(d.identifier);

        out.println("public class " + d.identifier + " implements XdrAble {");
        out.println();

        String paramType = checkForSpecials(d.type);
        if ( ((d.kind == JrpcgenDeclaration.FIXEDVECTOR)
              || (d.kind == JrpcgenDeclaration.DYNAMICVECTOR))
             && !d.type.equals("String") ) {
            paramType += " []";
        }

        out.print("    public " + paramType + " value;");
        out.println();

        //
        // Now generate code for encoding and decoding this class (typedef).
        //
        JrpcgenDeclaration dstar = null;
        try {
            dstar = (JrpcgenDeclaration) d.clone();
        } catch ( CloneNotSupportedException e ) {
            throw(new RuntimeException("fatal: can not clone JrpcgenDeclaration"));
        }
        dstar.identifier = "value";

        out.println();
        out.println("    public " + d.identifier + "() {");
        out.println("    }");
        out.println();
        out.println("    public " + d.identifier + "(" + paramType + " value) {");
        out.println("        this.value = value;");
        out.println("    }");
        out.println();
        out.println("    public " + d.identifier + "(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        xdrDecode(xdr);");
        out.println("    }");
        out.println();
        out.println("    public void xdrEncode(XdrEncodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.print(codingMethod(dstar, true));
        out.println("    }");

        out.println();
        out.println("    public void xdrDecode(XdrDecodingStream xdr)");
        out.println("           throws OncRpcException, IOException {");
        out.print(codingMethod(dstar, false));
        out.println("    }");
        //
        // Close class...
        //
        out.println();
        out.println("}");
        closeJavaSourceFile();
    }

    /**
     * Generate source code files for all structures, unions and enumerations
     * as well as constants. All constants, which do not belong to enumerations,
     * are emitted to a single interface.
     */
    public static void dumpClasses() {
        Enumeration globals = globalIdentifiers.elements();
        while ( globals.hasMoreElements() ) {
            Object o = globals.nextElement();
            if ( o instanceof JrpcgenEnum ) {
                dumpEnum((JrpcgenEnum) o);
            } else if ( o instanceof JrpcgenStruct ) {
                dumpStruct((JrpcgenStruct) o);
            } else if ( o instanceof JrpcgenUnion ) {
                dumpUnion((JrpcgenUnion) o);
            } else if ( o instanceof JrpcgenDeclaration ) {
                dumpTypedef((JrpcgenDeclaration) o);
            }
        }
    }

    /**
     * Generate source code for client-side stub methods for a particular
     * remote program version.
     *
     * @param out Printer writer to send source code to.
     * @param versionInfo Information about the remote program version for
     *   which source code is to be generated.
     */
    public static void dumpClientStubMethods(PrintWriter out,
                                             JrpcgenVersionInfo versionInfo) {
        int size = versionInfo.procedures.size();
        for ( int idx = 0; idx < size; ++idx ) {
            JrpcgenProcedureInfo proc = (JrpcgenProcedureInfo)
                versionInfo.procedures.elementAt(idx);
            //
            // First spit out the stub method. While we don't need to
            // fiddle around with the data types of the method's
            // parameter and result...
            //
            String paramType = checkForSpecials(proc.parameterType);
            String resultType = checkForSpecials(proc.resultType);

            out.println("    /**");
            out.println("     * Call remote procedure " + proc.procedureId + ".");
            out.println("     * @param params Parameter(s) to the remote procedure call.");
            out.println("     * @return Result from remote procedure call.");
            out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
            out.println("     * @throws IOException if an I/O error occurs.");
            out.println("     */");
            out.println("    public "
                        + resultType + " "
                        + proc.procedureId + "("
                        + (paramType.equals("void") ?
                            "" : paramType + " params")
                        + ")");
            out.println("           throws OncRpcException, IOException {");
            //
            // ...we have to check for them to be base data types, like
            // int, boolean, etc. In this case, we need to wrap them
            // into serializeable XDR data types.
            //
            String xdrParams = xdrBaseType(paramType);
            String params = "params";
            String xdrResult = xdrBaseType(resultType);
            String result = "result";
            //
            // Handle the parameter to the method: in case of void we
            // just reuse a global XDR void data type wrapper singleton.
            // Otherwise we need to allocate a temporary wrapper and
            // initialize it with the method's only parameter.
            //
            if ( paramType.equals("void") ) {
                out.println("        XdrVoid xparams = XdrVoid.XDR_VOID;");
                params = "xparams";
            } else if ( xdrParams != null ) {
                out.println("        " + xdrParams + " xparams = new "
                            + xdrParams + "(params);");
                params = "xparams";
            }
            //
            // Handle the result of the method: similiar to what we did
            // above. However, in all other cases we always need to
            // create a result object, regardless of whether we have to
            // deal with a basic data type (except void) or with some
            // "complex" data type.
            //
            if ( resultType.equals("void") ) {
                out.println("        XdrVoid xresult = XdrVoid.XDR_VOID;");
                result = "xresult";
            } else if ( xdrResult != null ) {
                out.println("        " + xdrResult + " xresult = new "
                            + xdrResult + "();");
                result = "xresult";
            } else {
                out.println("        " + resultType + " result = new "
                            + resultType + "();");
            }
            //
            // Now emit the real ONC/RPC call using the (optionally
            // wrapped) parameter and (optionally wrapped) result.
            //
            out.println("        client.call("
                        + baseClassname + "." + proc.procedureId
                        + ", " + params + ", " + result + ");");
            //
            // In case of a wrapped result we need to return the value
            // of the wrapper, otherwise we can return the result
            // itself (which then is not a base data type). As a special
            // case, we can not return void values...anyone for a
            // language design with first class void objects?!
            //
            if ( xdrResult != null ) {
                if ( !resultType.equals("void") ) {
                    out.println("        return xresult."
                    + resultType.toLowerCase() + "Value();");
                }
            } else {
                out.println("        return result;");
            }
            //
            // Close the stub method (just as a hint, as it is
            // getting rather difficult to see what code is produced
            // at this stage...)
            //
            out.println("    }");
            out.println();
        }
    }

    /**
     * Generate source code for the client stub proxy object. This client
     * stub proxy object is then used by client applications to make remote
     * procedure (aka method) calls to an ONC/RPC server.
     */
    public static void dumpClient(JrpcgenProgramInfo programInfo) {
        //
        // When several versions of a program are defined, we search for the
        // latest and greatest one. This highest version number ist then
        // used to create the necessary <code>OncRpcClient</code> for
        // communication when the client proxy stub is constructed.
        //
        int version = Integer.parseInt(
            ((JrpcgenVersionInfo)programInfo.versions.elementAt(0)).versionNumber);
        int versionSize = programInfo.versions.size();
        for ( int idx = 1; idx < versionSize; ++idx ) {
            int anotherVersion = Integer.parseInt(
                ((JrpcgenVersionInfo)programInfo.versions.elementAt(idx)).versionNumber);
            if ( anotherVersion > version ) {
                version = anotherVersion;
            }
        }

        //
        // Create new source code file containing a Java class representing
        // the XDR struct.
        // In case we have several programs defines, build the source code
        // file name from the program's name (this case is identified by a
        // null clientClass name).
        //
        String clientClass = jrpcgen.clientClass;
        if ( clientClass == null ) {
            clientClass = baseClassname + "_" + programInfo.programId + "_Client";
            System.out.println("CLIENT: " + clientClass);
        }
        PrintWriter out = createJavaSourceFile(clientClass);

        out.println("import java.net.InetAddress;");
        out.println();

        out.println("/**");
        out.println(" * The class <code>" + clientClass + "</code> implements the client stub proxy");
        out.println(" * for the " + programInfo.programId + " remote program. It provides method stubs");
        out.println(" * which, when called, in turn call the appropriate remote method (procedure).");
        out.println(" */");
        out.println("public class " + clientClass +
                    " extends OncRpcClientStub {");
        out.println();
        //
        // Generate constructors...
        //
        out.println("    /**");
        out.println("     * Constructs a <code>" + clientClass + "</code> client stub proxy object");
        out.println("     * from which the " + programInfo.programId + " remote program can be accessed.");
        out.println("     * @param host Internet address of host where to contact the remote program.");
        out.println("     * @param protocol {@link org.acplt.oncrpc.OncRpcProtocols Protocol} to be");
        out.println("     *   used for ONC/RPC calls.");
        out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
        out.println("     * @throws IOException if an I/O error occurs.");
        out.println("     */");
        out.println("    public " + clientClass + "(InetAddress host, int protocol)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        super(host, "
                    + baseClassname + "." + programInfo.programId + ", "
                    + version + ", 0, protocol);");
        out.println("    }");
        out.println();

        out.println("    /**");
        out.println("     * Constructs a <code>" + clientClass + "</code> client stub proxy object");
        out.println("     * from which the " + programInfo.programId + " remote program can be accessed.");
        out.println("     * @param host Internet address of host where to contact the remote program.");
        out.println("     * @param port Port number at host where the remote program can be reached.");
        out.println("     * @param protocol {@link org.acplt.oncrpc.OncRpcProtocols Protocol} to be");
        out.println("     *   used for ONC/RPC calls.");
        out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
        out.println("     * @throws IOException if an I/O error occurs.");
        out.println("     */");
        out.println("    public " + clientClass + "(InetAddress host, int port, int protocol)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        super(host, "
                    + baseClassname + "." + programInfo.programId + ", "
                    + version + ", port, protocol);");
        out.println("    }");
        out.println();

        out.println("    /**");
        out.println("     * Constructs a <code>" + clientClass + "</code> client stub proxy object");
        out.println("     * from which the " + programInfo.programId + " remote program can be accessed.");
        out.println("     * @param client ONC/RPC client connection object implementing a particular");
        out.println("     *   protocol.");
        out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
        out.println("     * @throws IOException if an I/O error occurs.");
        out.println("     */");
        out.println("    public " + clientClass + "(OncRpcClient client)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        super(client);");
        out.println("    }");
        out.println();

        out.println("    /**");
        out.println("     * Constructs a <code>" + clientClass + "</code> client stub proxy object");
        out.println("     * from which the " + programInfo.programId + " remote program can be accessed.");
        out.println("     * @param host Internet address of host where to contact the remote program.");
        out.println("     * @param program Remote program number.");
        out.println("     * @param version Remote program version number.");
        out.println("     * @param protocol {@link org.acplt.oncrpc.OncRpcProtocols Protocol} to be");
        out.println("     *   used for ONC/RPC calls.");
        out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
        out.println("     * @throws IOException if an I/O error occurs.");
        out.println("     */");
        out.println("    public " + clientClass + "(InetAddress host, int program, int version, int protocol)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        super(host, program, version, 0, protocol);");
        out.println("    }");
        out.println();

        out.println("    /**");
        out.println("     * Constructs a <code>" + clientClass + "</code> client stub proxy object");
        out.println("     * from which the " + programInfo.programId + " remote program can be accessed.");
        out.println("     * @param host Internet address of host where to contact the remote program.");
        out.println("     * @param program Remote program number.");
        out.println("     * @param version Remote program version number.");
        out.println("     * @param port Port number at host where the remote program can be reached.");
        out.println("     * @param protocol {@link org.acplt.oncrpc.OncRpcProtocols Protocol} to be");
        out.println("     *   used for ONC/RPC calls.");
        out.println("     * @throws OncRpcException if an ONC/RPC error occurs.");
        out.println("     * @throws IOException if an I/O error occurs.");
        out.println("     */");
        out.println("    public " + clientClass + "(InetAddress host, int program, int version, int port, int protocol)");
        out.println("           throws OncRpcException, IOException {");
        out.println("        super(host, program, version, port, protocol);");
        out.println("    }");
        out.println();

        //
        // Generate method stubs... This is getting hairy in case someone
        // uses basic data types as parameters or the procedure's result.
        // In these cases we need to encapsulate these basic data types in
        // XDR-able data types.
        //
        for ( int versionIdx = 0; versionIdx < versionSize; ++versionIdx ) {
            JrpcgenVersionInfo versionInfo = (JrpcgenVersionInfo)
                programInfo.versions.elementAt(versionIdx);
            dumpClientStubMethods(out, versionInfo);
        }
        //
        // Close class...done!
        //
        out.println("}");
        closeJavaSourceFile();
    }

    /**
     *
     */
    public static void dumpServerStubMethodCall(PrintWriter out,
                                                JrpcgenProcedureInfo proc) {
        //
        // We need to check for the parameter and result to be base data
        // types, like int, boolean, etc. In this case, we need to unwrap them
        // before we hand them over to the real remote procedure.
        //
        String paramType = checkForSpecials(proc.parameterType);
        String xdrParams = xdrBaseType(paramType);
        String xdrResult = xdrBaseType(checkForSpecials(proc.resultType));
        //
        // Handle the parameter to the method: in case of base data types,
        // we have to do the unwrapping first. Otherwise we can just pull
        // the appropriate object from the stream. In case of void we can
        // skip the serializing step almost completely.
        //
        String params = "";
        if ( proc.parameterType.equals("void") ) {
            out.println("                call.retrieveCall(XdrVoid.XDR_VOID);");
        } else if ( xdrParams != null ) {
            out.println("                " + xdrParams + " params = new "
                        + xdrParams + "();");
            out.println("                call.retrieveCall(params);");
            params = "params."
                     + paramType.toLowerCase() + "Value()";
        } else {
            out.println("                " + paramType + " params = new "
                        + paramType + "();");
            out.println("                call.retrieveCall(params);");
            params = "params";
        }

        if ( proc.resultType.equals("void") ) {
            out.println("                " + proc.procedureId + "(" + params + ");");
            out.println("                call.reply(XdrVoid.XDR_VOID);");
        } else if ( xdrResult != null ) {
            out.println("                " + xdrResult + " result = new "
                        + xdrResult + "("
                        + proc.procedureId + "(" + params + "));");
            out.println("                call.reply(result);");
        } else {
            out.println("                " + proc.resultType + " result = "
                        + proc.procedureId + "(" + params + ");");
            out.println("                call.reply(result);");
        }
    }

    /**
     *
     */
    public static void dumpServerStubMethods(PrintWriter out,
                                             JrpcgenVersionInfo versionInfo) {
        int procSize = versionInfo.procedures.size();

        for ( int idx = 0; idx < procSize; ++idx ) {
            JrpcgenProcedureInfo proc = (JrpcgenProcedureInfo)
                versionInfo.procedures.elementAt(idx);
            String params = "";
            if ( !proc.parameterType.equals("void") ) {
                params = checkForSpecials(proc.parameterType) + " params";
            }
            String result = checkForSpecials(proc.resultType);

            out.println("    public abstract " + result + " "
                        + proc.procedureId + "(" + params + ");");
            out.println();
        }
    }

    /**
     *
     */
    public static void dumpServer(JrpcgenProgramInfo programInfo) {
        //
        // Create new source code file containing a Java class representing
        // the XDR struct.
        // In case we have several programs defines, build the source code
        // file name from the program's name (this case is identified by a
        // null clientClass name).
        //
        String serverClass = jrpcgen.serverClass;
        if ( serverClass == null ) {
            serverClass = baseClassname + "_" + programInfo.programId + "_ServerStub";
        }
        PrintWriter out = createJavaSourceFile(serverClass);

        out.println("import org.acplt.oncrpc.server.*;");
        out.println();

        out.println("/**");
        out.println(" */");
        out.println("public abstract class " + serverClass +
                    " extends OncRpcServerStub implements OncRpcDispatchable {");
        out.println();

        //
        // Generate constructor...
        //
        out.println("    public " + serverClass + "()");
        out.println("           throws OncRpcException, IOException {");
        out.println("        this(0);");
        out.println("    }");
        out.println();

        out.println("    public " + serverClass + "(int port)");
        out.println("           throws OncRpcException, IOException {");
        //
        // For every version specified, create both UDP and TCP-based
        // transports.
        //
        out.println("        info = new OncRpcServerTransportRegistrationInfo [] {");
        int versionSize = programInfo.versions.size();
        for ( int versionIdx = 0; versionIdx < versionSize; ++versionIdx ) {
            JrpcgenVersionInfo versionInfo = (JrpcgenVersionInfo)
                programInfo.versions.elementAt(versionIdx);
            out.println("            new OncRpcServerTransportRegistrationInfo("
                        + baseClassname + "." + programInfo.programId + ", "
                        + versionInfo.versionNumber + "),");
        }
        out.println("        };");

        out.println("        transports = new OncRpcServerTransport [] {");
        out.println("            new OncRpcUdpServerTransport(this, port, info, 32768),");
        out.println("            new OncRpcTcpServerTransport(this, port, info, 32768)");
        out.println("        };");
        //
        // Finish constructor method...
        //
        out.println("    }");
        out.println();

        //
        // Generate dispatcher code...
        //
        out.println("    public void dispatchOncRpcCall(OncRpcCallInformation call, int program, int version, int procedure)");
        out.println("           throws OncRpcException, IOException {");

        for ( int versionIdx = 0; versionIdx < versionSize; ++versionIdx ) {
            JrpcgenVersionInfo versionInfo = (JrpcgenVersionInfo)
                programInfo.versions.elementAt(versionIdx);
            out.print(versionIdx == 0 ? "        " : "        } else ");
            out.println("if ( version == " + versionInfo.versionNumber + " ) {");
            int procSize = versionInfo.procedures.size();
            out.println("            switch ( procedure ) {");
            for ( int procIdx = 0; procIdx < procSize; ++procIdx ) {
                //
                //
                //
                JrpcgenProcedureInfo procInfo = (JrpcgenProcedureInfo)
                    versionInfo.procedures.elementAt(procIdx);
                out.println("            case " + procInfo.procedureNumber + ": {");
                dumpServerStubMethodCall(out, procInfo);
                out.println("                break;");
                out.println("            }");
            }
            out.println("            default:");
            out.println("                call.failProcedureUnavailable();");
            out.println("            }");
        }

        out.println("        } else {");
        out.println("            call.failProcedureUnavailable();");
        out.println("        }");
        out.println("    }");
        out.println();

        //
        // Generate the stub methods for all specified remote procedures.
        //
        for ( int versionIdx = 0; versionIdx < versionSize; ++versionIdx ) {
            JrpcgenVersionInfo versionInfo = (JrpcgenVersionInfo)
                programInfo.versions.elementAt(versionIdx);
            dumpServerStubMethods(out, versionInfo);
        }

        //
        // Close class...done!
        //
        out.println("}");
        closeJavaSourceFile();
    }


    /**
     * Create the source code files based on the parsed information from the
     * x-file.
     */
    public static void dumpFiles() {
        dumpConstants();
        dumpClasses();
        for ( int i = 0; i < programInfos.size(); ++i ) {
            JrpcgenProgramInfo progInfo =
                (JrpcgenProgramInfo) programInfos.elementAt(i);
            if ( !noClient ) {
                dumpClient(progInfo);
            }
            if ( !noServer ) {
                dumpServer(progInfo);
            }
        }
    }

    /**
     * The main part of jrpcgen where all things start.
     */
    public static void main(String[] args) {
        //
        // First parse the command line (options)...
        //
        int argc = args.length;
        int argIdx = 0;
        for ( ; argIdx < argc; ++argIdx ) {
            //
            // Check to see whether this is an option...
            //
            String arg = args[argIdx];
            if ( (arg.length() > 0)
                 && (arg.charAt(0) != '-') ) {
                break;
            }
            //
            // ...and which option is it?
            //
            if ( arg.equals("-d") ) {
                // -d <dir>
                if ( ++argIdx >= argc ) {
                    System.out.println("jrpcgen: missing directory");
                    System.exit(1);
                }
                destinationDir = new File(args[argIdx]);
            } else if ( arg.equals("-package")
                 || arg.equals("-p") ) {
                // -p <package name>
                if ( ++argIdx >= argc ) {
                    System.out.println("jrpcgen: missing package name");
                    System.exit(1);
                }
                packageName = args[argIdx];
            } else if ( arg.equals("-c") ) {
                // -c <class name>
                if ( ++argIdx >= argc ) {
                    System.out.println("jrpcgen: missing client class name");
                    System.exit(1);
                }
                clientClass = args[argIdx];
            } else if ( arg.equals("-s") ) {
                // -s <class name>
                if ( ++argIdx >= argc ) {
                    System.out.println("jrpcgen: missing server class name");
                    System.exit(1);
                }
                serverClass = args[argIdx];
            } else if ( arg.equals("-debug") ) {
                debug = true;
            } else if ( arg.equals("-nobackup") ) {
                noBackups = true;
            } else if ( arg.equals("-noclient") ) {
                noClient = true;
            } else if ( arg.equals("-noserver") ) {
                noServer = true;
            } else if ( arg.equals("-parseonly") ) {
                parseOnly = true;
            } else if ( arg.equals("-verbose") ) {
                verbose = true;
            } else if ( arg.equals("-version") ) {
                System.out.println("jrpcgen version \"" + VERSION + "\"");
                System.exit(1);
            } else if ( arg.equals("-help") || arg.equals("-?") ) {
                printHelp();
                System.exit(1);
            } else if ( arg.equals("--") ) {
                //
                // End of options...
                //
                ++argIdx;
                break;
            } else {
                //
                // It's an unknown option!
                //
                System.out.println("Unrecognized option: " + arg);
                System.exit(1);
            }
        }
        //
        // Otherwise we regard the current command line argument to be the
        // name of the x-file to compile. Check, that there is exactly one
        // x-file specified.
        //
        if ( (argIdx >= argc) || (argIdx < argc - 1) ) {
            printHelp();
            System.exit(1);
        }
        String xfilename = args[argIdx];
        File xfile = new File(".", xfilename);
        //
        // Get the base name for the client and server classes, it is derived
        // from the filename.
        //
        if ( baseClassname == null ) {
            String name = xfile.getName();
            int dotIdx = name.lastIndexOf('.');
            if ( dotIdx < 0 ) {
                baseClassname = name;
            } else {
                baseClassname = name.substring(0, dotIdx);
            }
        }
        //
        //
        //
        FileInputStream in = null;
        try {
            in = new FileInputStream(xfilename);
        } catch ( FileNotFoundException e ) {
            System.out.println("jrpcgen: can not open " + xfilename);
            System.exit(1);
        }
        JrpcgenScanner scanner = new JrpcgenScanner(in);
        JrpcgenParser parser = new JrpcgenParser(scanner);

        jrpcgen.globalIdentifiers.put("TRUE", new JrpcgenConst("TRUE", "true"));
        jrpcgen.globalIdentifiers.put("FALSE", new JrpcgenConst("FALSE", "false"));

        try {
            Symbol sym = parser.parse();
            if ( !parseOnly ) {
                if ( programInfos.size() <= 1 ) {
                    if ( clientClass == null ) {
                        clientClass = baseClassname + "Client";
                    }
                    if ( serverClass == null ) {
                        serverClass = baseClassname + "ServerStub";
                    }
                }
                dumpFiles();
            }
        } catch ( JrpcgenParserException pe ) {
            System.out.println("jrpcgen: compilation aborted.");
            System.exit(1);
        } catch ( Exception e ) {
            e.printStackTrace();
            System.exit(1);
        }
/*
        try {
            Symbol sym;
            do {
                sym = scanner.next_token();
                System.out.println(sym.toString());
            } while ( sym.sym != JrpcgenSymbols.EOF );
        } catch (java.io.IOException e) {
            System.out.print("I/O error while scanning: ");
            System.out.println(e);
            System.exit(1);
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
*/
    }

}

    /**
     * The class <code>JrpcgenEnDecodingInfo</code> contains information which
     * is necessary to generate source code calling appropriate XDR encoding
     * and decoding methods.
     *
 * @version $Revision: 1.1.1.1 $ $Date: 2001/11/02 01:55:30 $ $State: Exp $ $Locker:  $
     * @author Harald Albrecht
     */
    class JrpcgenEnDecodingInfo {

        /**
         * Construct a <code>JrpcgenEnDecodingInfo</code> object containing
         * information for generating source code for encoding and decoding
         * of XDR/Java base data types.
         *
         * @param syllable Syllable of encoding/decoding method.
         * @param encodingOptions Optional parameters necessary to encode
         *   base data type.
         * @param decodingOptions Optional parameters necessary to decode
         *   base data type.
         */
        public JrpcgenEnDecodingInfo(String syllable, String encodingOptions,
                                     String decodingOptions) {
            this.syllable = syllable;
            this.encodingOptions = encodingOptions;
            this.decodingOptions = decodingOptions;
        }

        /**
         * (Type) syllable of the encoding or decoding method. The full name
         * of the encoding or decoding method is always in the form of
         * "xdrEncodeXXX(...)" or "xdrDecodeXXX(...)", where "XXX" is the
         * syllable contained in this attribute.
         */
        public String syllable;

        /**
         * Optional parameters to use when encoding a base data type. This
         * typically includes the size parameter for encoding fixed-size
         * vectors/arrays. When this attribute is not <code>null</code>, then
         * these parameters need to be appended. The attribute never contains
         * a leading parameter separator (aka "comma").
         */
        public String encodingOptions;

        /**
         * Optional parameters to use when decoding a base data type. This
         * typically includes the size parameter for decoding fixed-size
         * vectors/arrays. When this attribute is not <code>null</code>, then
         * these parameters need to be appended. The attribute never contains
         * a leading parameter separator (aka "comma").
         */
        public String decodingOptions;

    }


// End of file jrpcgen.java
