/* 
 * @(#)CryptoTest.java
 * 
 * This software is released under the GNU General Public License.
 * http://www.gnu.org/copyleft/gpl.html
 * 
 * Under no circumstances does the author of this software assume
 * any sort of liability pertaining to the use, modification, or
 * distribution of this software.
 * 
 * In other words, use this code AT YOUR OWN RISK!
 */

package security.crypto;

import java.math.*;
import java.security.*;

/**
 * An application class to test and benchmark the functionality
 * of the RSA cryptography classes.
 * 
 * @author	Jared Klett
 * @version	1.0 (7/14/00)
 */

public class CryptoTest {

	public static void main(String[] args) {

		// Generate to random prime numbers for key generation
		//
		SecureRandom random = new SecureRandom();
		BigInteger x = new BigInteger(128, 100, random);
		BigInteger y = new BigInteger(128, 100, random);

		// Begin benchmark
		//
		long start = System.currentTimeMillis();

		// Generate key pair
		//
		RSAKeyPairGenerator kpg = new RSAKeyPairGenerator(128, x, y);
		KeyPair pair = kpg.generateKeyPair();

		// End benchmark
		//
		long delta = System.currentTimeMillis() - start;

		System.out.println("Keys generated in " + delta + " ms");
		System.out.println("Public key: " + pair.getPublic());
		System.out.println("Private key: " + pair.getPrivate());
		System.out.println();

		// Test reading/writing from file
		kpg.writeKeys("test", pair);
		pair = kpg.readKeys("test");

		System.out.println("Public key: " + pair.getPublic());
		System.out.println("Private key: " + pair.getPrivate());
		System.out.println();

		// Begin benchmark
		//
		start = System.currentTimeMillis();

		// Encrypt
		//
		BigInteger[] ciphertext = RSACrypto.encrypt(args[0], (RSAPublicKey)pair.getPublic());
		

		// End benchmark
		//
		delta = System.currentTimeMillis() - start;

		System.out.println("Ciphertext: ");

		for(int i = 0; i < ciphertext.length; i++)
			System.out.println(ciphertext[i]);

		System.out.println("Encrypted in " + delta + " ms");

		// Begin benchmark
		//
		start = System.currentTimeMillis();

		// Decrypt
		//
		String s = RSACrypto.decrypt(ciphertext, (RSAPrivateKey)pair.getPrivate());

		// End benchmark
		//
		delta = System.currentTimeMillis() - start;

		System.out.println("Decrypted in " + delta + " ms");
		System.out.println("Plaintext: " + s);


		// Testing Big Integer Array Stuff

		System.out.println("Testing Big Integer Conversion");
		String test = new String(args[0]);
		System.out.println("original: " + args[0]);

		BigInteger[] biArray = RSACrypto.stringToBigInt(args[0]);
		String conv = RSACrypto.bigIntToString(biArray);
		System.out.println("converted: " + conv);
		


		BigInteger[] biArray2 = new BigInteger[10];
		for (int i=0; i<10; i++)
		    biArray2[i] = new BigInteger(128, random);
		String conv2 = RSACrypto.bigIntToString(biArray2);
		System.out.println("converted: " + conv2);


		BigInteger[] biArray3 =
		    RSACrypto.byteArrToBigInt
		    (RSACrypto.bigIntToByteArr(biArray2));

		if (biArray2.equals(biArray3))
		    System.err.println("SUCCESS");
		else {
		    System.err.println("FAIL");
		    for (int i=0; i<biArray2.length; i++)
			System.err.println(biArray2[i] + "\t" +
					   biArray3[i]);
		}



//  		BigInteger[] biArray2 = new BigInteger[10];
//  		for (int i=0; i<10; i++)
//  		    biArray2[i] = new BigInteger(8, random);
//  		String conv2 = RSACrypto.bigIntToString(biArray2);
//  		System.out.println("converted: " + conv2);

//  		BigInteger[] biArray3 =
//  		    RSACrypto.stringToBigInt(conv2);

//  		if (biArray2.equals(biArray3))
//  		    System.err.println("SUCCESS");
//  		else {
//  		    System.err.println("FAIL");
//  		    for (int i=0; i<biArray2.length; i++)
//  			System.err.println(biArray2[i] + "\t" +
//  					   biArray3[i]);
//  		}

		
		
	}

} // class CryptoTest
