package ins.namespace;

import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;

/**
 * Represents an Attribute-Value tree node in a NameSpecifier.
 */
public class AVelement
{
    // VARIABLES
    
    private Attribute a;
    private Value v;
    private Vector children; // Vector of AVelement 
                             // (children of this AVelement)
    
    // CONSTRUCTORS
    
    /**
     * Creates an AVelement. Used by NameSpecifier.
     */
    protected AVelement()
    {
	children = new Vector(4);
	a=null;
	v=null;
    }

    /**
     * Creates an AVelement from an Attribute and a Value.
     * @param a the Attribute in the AVelement.
     * @param v the Value in the AVelement.
     */
    public AVelement(Attribute a, Value v)
    {
	this.a = a;
	this.v = v;

	children = new Vector(4);
    }

    /**
     * Creates an AVelement from its wire representation..
     * @param s the String to create the AVelement from.
     * @exception CantParseString the String given was bad.
     */
    public AVelement(String s)
	throws CantParseString
    {
    	StringTokenizer st;
	int level;    // bracket nesting level
	String next;  // the next token
	String child; // the currnet AVelement string

	// System.out.println("AVelement: "+s);

	children = new Vector(4);

	st = new StringTokenizer(s, "[=]", true);

	// We discard stuff until we get to the first "["
	while (st.hasMoreTokens()) {
	    next = st.nextToken();
	    
	    if (next.equals("["))
		break;
	}

	// The next token is the Attribute
	
	if (! st.hasMoreTokens()) {
	    throw new CantParseString("Expected an Attribute, " +
				      "but ran out of tokens in " + s);
	}

	next = st.nextToken();

	if (next.equals("[") || next.equals("=") || next.equals("]")) {
	    throw new CantParseString("Expected an Attribute, not a " +
				      next + " in " + s);
	}

	a = new Attribute(next);

	// The next token is the =

	if (! st.hasMoreTokens()) {
	    throw new CantParseString("Expected an =, " +
				      "but ran out of tokens in " + s);
	}

	next = st.nextToken();

	if (! next.equals("=")) {
	    throw new CantParseString("Expected an =, not a " +
				      next + " in " + s);
	}	    

	// The next token is the Value

	if (! st.hasMoreTokens()) {
	    throw new CantParseString("Expected a Value, " +
				      "but ran out of tokens in " + s);
	}

	next = st.nextToken();

	if (next.equals("[") || next.equals("=") || next.equals("]")) {
	    throw new CantParseString("Expected a Value, not a " +
				      next + " in " + s);
	}

	v = new Value(next);

	// Now we deal with the children of this AVelement.
	// We start off at bracket level 1, add one every time we see a "[",
	// and subtract one every time we see a "]".
	// If we get back to 1, we've finished a child.
	// If we get back to 0, we've finished this AVelement.

	level = 1;
	child = "";
	while (st.hasMoreTokens()) {
	    // Append the token to the child.
	    next = st.nextToken();
	    child = child + next;
	    
	    if (next.equals("[")) {
		level++;
	    } else if (next.equals("]")) {
		level--;
		if (level == 1) {
		    // Done the child. Create it, add it, and clear the string.

		    AVelement c = new AVelement(child);
		    children.addElement((Object)c);
		    child = "";
		} else if (level == 0) {
		    // Done the whole AVelement.

		    break;
		}
	    }
	}
    }

    /**
     * Creates an AVelement that is a copy of ave.
     * @param ave the AVelement to copy.
     */
    public AVelement(AVelement ave)
    {
	a = new Attribute(ave.a);
	v = new Value(ave.v);
	
	children = new Vector(ave.children.size());

	for (Enumeration e = ave.children.elements(); e.hasMoreElements(); ) {
	    children.addElement(new AVelement((AVelement)e.nextElement()));
	}

    }

    // METHODS
    
    /**
     * Add AVelement c as a child of this AVelement.
     * @param c the AVelement to be added as a child
     */
    public void addAVelement(AVelement c)
    {
	children.addElement(c);
    }

    /**
     * Remove AVelement c as a child of this AVelement.
     * @param c the AVelement to be removed as a child
     */
    public void removeAVelement(AVelement c)
    {
	children.removeElement(c);
    }

    /**
     * Remove all AVelements (children) of this AVelement.
     */
    public void removeAllAVelements()
    {
	children.removeAllElements();
    }

    /**
     * Checks if an AVelement is a leaf (i.e. has no children).
     * @return true if this AVelement is a leaf node, false otherwise.
     */
    public boolean isLeaf()
    {
	return(children.isEmpty());
    }

    /**
     * Gets the Attribute of this AVelement.
     * @return the Attribute of this AVelement.
     */
    public Attribute getAttribute()
    {
	return(a);
    }

    /**
     * Gets the Value of this AVelement.
     * @return the Value of this AVelement.
     */
    public Value getValue()
    {
	return(v);
    }

    /**
     * Sets the Value of this AVelement.
     */
    public void setValue(Value v) {
	this.v = v;
    }

    /**
     * Returns an Enumeration of the AVelement children of this AVelement.
     */
    public Enumeration getAVelements()
    {
	return(children.elements());
    }

    /**
     * Gets the AVelement child of this AVelement with a particular Attribute.
     * @param a the Attribute of the child AVelement to be returned.
     * @return the child AVelement with Attribute a, or null if not found.
     */
    public AVelement getAVelement(Attribute a)
    {
	
	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    AVelement c = (AVelement)e.nextElement();
	    
	    if (c.a.equals(a)) {
		return(c);
	    }
	}
	
	return(null);
    }

    /**
     * Tests two AVelements for equality.
     * @param obj the object to test equality with.
     * @return true if they are equal, false otherwise.
     */
    public boolean equals(Object obj)
    {
	if (!(obj instanceof AVelement)) return false;
	AVelement ave = (AVelement)obj;

	// First part of test is because the could both be null,
	// as they are in a NameSpecifier

	if (! (a == null && ave.a == null)) {
	    if (a == null || ave.a == null) {
		return(false);
	    }
	    
	    if (! a.equals(ave.a)) {
		return(false);
	    }
	}

	if (! (v == null && ave.v == null)) {
	    if (v == null || ave.v == null) {
		return(false);
	    }
	    
	    if (! v.equals(ave.v)) {
		return(false);
	    }
	}

	// Check if ave has children but this does not -> return false
	if (children.size()==0 && ave.children.size()>0)
	    return false;

	// The following was modified by Magda to correct comparisons
	// for the cases where the same attribute appeared more than once
	/*for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	  AVelement c1 = (AVelement)e.nextElement();
	    
	  // Not particularly efficient -- may want to optimize for the
	  // common case that both AVelements have their children in
	  // the same order.
	  AVelement c2 = ave.getAVelement(c1.a);
	  
	  if (c2 == null || ! c1.equals(c2)) {
	  return(false);
	  }
	  }
	*/
	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    AVelement c1 = (AVelement)e.nextElement();
	    
	    boolean found = false;
	    for (Enumeration e2 = ave.children.elements(); e2.hasMoreElements() && !found; ) {
		AVelement c2 = (AVelement)e2.nextElement();
		
		if ( c1.equals(c2) ) {
		    found = true;
		}
	    }
	    if (!found)
		return(false);
	}
	// End modification by Magda
	
	return(true);
    }

    /**
     * Returns a String representation of the AVelement.
     * @return a String representation of the AVelement.
     */
    public String toString()
    {
	String output = "";

	if (a != null && v != null)
	  output = output + "[" + a + "=" + v;

	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    output = output + (AVelement)e.nextElement();
	}

	if (a != null && v != null)
	  output = output + "]";

	return(output);
    }

}
