package ins.namespace;

import java.util.Date;
import java.util.Vector;
import java.util.Enumeration;
import ins.inr.IHandler;
import ins.inr.HostRecord;
import ins.inr.RouteEntry;
import ins.inr.Node;
import ins.inr.NameUpdate;

/**
 * This represents a NameRecord in the NameTree.
 */
public class NameRecord
{
    // VARIABLES
  
    // Constants for announceStatus variable.
    public static final boolean announce   = true;
    public static final boolean noAnnounce = false;

    // the id to be assigned to the next NameRecord
    private static int next_id = 0; 
    private static Object next_id_lock = new Object();
  
    final int id;                   // the id of this NameRecord

    private int appMetric;          // the anycast metric
    private Date expireTime;        // when the name info expires
    private boolean announceStatus; // true if the route should be announced

    private long announcer;

    // The actual route information
    private boolean isNeighbor;
    // private int neighbor;       // neighbor the route is to
    private long INRuid;        // the UID of INR announcing this name
    private RouteEntry routeEntry;  // "pointer" to the entry of RouteTable
		// for INRuid  (INRuid itself is enough, this is redundant
		// but to speed up lookup such that no more lookup of INRuid
		// in the RouteTable necessary).
    private IHandler handler; // handler the route is to

    // Early binding information
    private HostRecord hostRecord;

    // Cache - not yet implemented 
    private Cache cache;            // cache of the object
    private long lifetime;          // initial lifetime of cached object

    // -- internal use for extracting name
    private Vector parents;         // Vector of ValueElement
    // (contains parents in the routing tree)
    // used for determining the NameSpecifier 
    // of a NameRecord


    
    // CONSTRUCTORS
  
    /**
     * Creates a new NameRecord.
     * @param h the handler the route is to
     * @param m the application-controlled metric for the name
     * @param rl the lifetime of the name in milliseconds
     * @param cl the lifetime of cached data in milliseconds
     * @param a true if the route should be announced
     * @param ann the announcer of this NameRecord
     */
    public NameRecord(IHandler h, 
		      int m, long rl, long cl, boolean a,
		      long ann)
    {
	id = getNextId();

	isNeighbor = false;
	handler = h;

	hostRecord = new HostRecord();

	appMetric = m;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}

	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = cl;

	announceStatus = a;
	announcer = ann;
	
	if (a && ann == 0)
	    throw new RuntimeException("Oops!");

	parents = new Vector();
    }

    /**
     * Creates a new NameRecord.
     * @param h the handler the route is to
     * @param hr the early-binding information of the destination
     * @param m the application-controlled metric for the name
     * @param rl the lifetime of the route in milliseconds
     * @param cl the lifetime of cached data in milliseconds
     * @param a true if the route should be announced
     * @param ann the announcer of this NameRecord
     */
    public NameRecord(IHandler h, HostRecord hr,
		      int m, long rl, long cl, boolean a,
		      long ann)
    {
	id = getNextId();

	isNeighbor = false;
	handler = h;

	hostRecord = hr;

	appMetric = m;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}
	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = cl;

	announceStatus = a;
	announcer = ann;
	
	if (a && ann == 0)
	    throw new RuntimeException("Oops!");

	parents = new Vector();
    }

  
    /**
     * Creates a new NameRecord.
     * @param inr the INRuid of neighbor the route is to
     * @param m the application-controlled metric for the name
     * @param rl the lifetime of the route in milliseconds
     * @param cl the lifetime of cached data in milliseconds
     * @param a true if the route should be announced
     * @param ann the announcer of this NameRecord
     */
    public NameRecord(long inr, RouteEntry re,
		      int m, long rl, long cl, boolean a, long ann)
    {
	id = getNextId();

	isNeighbor = true;
//	neighbor = n;
	INRuid = inr;
	routeEntry = re;

	hostRecord = new HostRecord();

	appMetric = m;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}
	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = cl;

	announceStatus = a;
	announcer = ann;

	if (a && ann == 0)
	    throw new RuntimeException("Oops!");

	parents = new Vector();
    }

    /**
     * Creates a new NameRecord.
     * @param n the neighbor the route is to
     * @param hr the early-binding information of the destination
     * @param m the application-controlled metric for the name
     * @param rl the lifetime of the route in milliseconds
     * @param cl the lifetime of cached data in milliseconds
     * @param a true if the route should be announced
     * @param ann the announcer of this NameRecord
     */
    public NameRecord(long inr, RouteEntry re, HostRecord hr,
		      int m, long rl, long cl, boolean a, long ann)
    {
	id = getNextId();

	isNeighbor = true;
//	neighbor = n;
	INRuid = inr;
	routeEntry = re;

	hostRecord = hr;

	appMetric = m;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}

	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = cl;

	announceStatus = a;
	announcer = ann;

	if (a && ann == 0)
	    throw new RuntimeException("Oops!");

	parents = new Vector();
    }

    public NameRecord(NameUpdate nu, RouteEntry re, long rl)
    {
	id = getNextId();

	isNeighbor = true;
	INRuid = nu.INRuid;
	routeEntry = re;

	hostRecord = nu.hostRecord;

	appMetric = nu.appMetric;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}

	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = nu.cacheLifetime;

	announceStatus = true;
	announcer = nu.announcer;

	if (announcer == 0)
	    throw new RuntimeException("Oops!");

	parents = new Vector();

    }  

    // this is for testing purposes only!
    /*    public NameRecord() {
	id = getNextId();

	isNeighbor = true;
	INRuid = System.currentTimeMillis();
	
	Node node = null;
	routeEntry = new RouteEntry(INRuid, node, 3, 123, 1000000);

	hostRecord = new HostRecord();

	Date now = new Date();
	expireTime = new Date(now.getTime() + 1000000);

	appMetric = 0;
	lifetime = 0;

	announceStatus = true;
	announcer = System.currentTimeMillis();

	parents = new Vector();

	}*/

    // METHODS

    /**
     * Updates the NameRecord.
     */
    public void update(long inr, RouteEntry re, int newMetric, long newRl, long newCl, 
	HostRecord newHR, boolean newA)
    {
	int oldMetric = appMetric;

	    INRuid = inr;
	    routeEntry = re;
	    appMetric = newMetric;

	    // Set the expire time
	    if (newRl==-1)
		expireTime = null;
	    else
	    {
		Date now = new Date();
		expireTime = new Date(now.getTime() + newRl);
	    }

	    lifetime = newCl;
	    hostRecord = newHR;
	    announceStatus = newA;

    }

    /**
     * Updates the NameRecord
     */
    public void update(IHandler h, int newMetric, 
	long newRl, long newCl,	HostRecord newHR, boolean newA)
    {
	int oldMetric = appMetric;

	    handler = h;
	    appMetric = newMetric;

	    // Set the expire time
	    if (newRl==-1)
		expireTime = null;
	    else
	    {
		Date now = new Date();
		expireTime = new Date(now.getTime() + newRl);
	    }

	    lifetime = newCl;
	    hostRecord = newHR;
	    announceStatus = newA;
    }


    public void update (NameUpdate nu, RouteEntry re, long rl)
    {
	INRuid = nu.INRuid;

	routeEntry = re;

	hostRecord = nu.hostRecord;
	appMetric = nu.appMetric;

	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}

	// Don't bother creating cache until we need it, but set the lifetime.
	lifetime = nu.cacheLifetime;

	announceStatus = true;
	if (announcer != nu.announcer)
	{
	    System.out.println("ASSERT: update() is only for the same announcer");
	    return;
	}
    }

    /**
     * Returns true if this NameRecord is to a neighbor.
     */
    public boolean isNeighbor()
    {
     	 return(isNeighbor);
    }

    /**
     * Returns the neighbor this NameRecord is to.
     * @exception WrongRouteType This NameRecord is not to a neighbor.
     */
    public Node getNeighbor()
	throws WrongRouteType
    {
	if (isNeighbor) {
	    if (routeEntry==null) return null;
	    return(routeEntry.nextHop);
	} else {
	    throw(new WrongRouteType());
	}
    }
  

    /**
     * Returns the handler this NameRecord is to.
     * @exception WrongRouteType This NameRecord is not to a neighbor.
     */
    public IHandler getHandler()
	throws WrongRouteType
    {
	if (isNeighbor)
	    throw(new WrongRouteType());
	else
	    return(handler);
    }

    public byte[] getAddress()
    {
	if (hostRecord.address == null) {
	    System.err.println("null address in NameRecord");
	    return(new byte[4]);
	} else {
	    return(hostRecord.address);
	}
    }

    /**
     * Returns the metric of this NameRecord.
     */  
    public int getAppMetric()
    {
	Date now = new Date();
	//	if (now.after(expireTime))
	//	    return(Integer.MAX_VALUE);

	return(appMetric);
    }

    /**
     * Returns the announcement status of this NameRecord.
     */
    public boolean getAnnounceStatus()
    {
	return(announceStatus);
    }

    /**
     * Returns the announcer of this NameRecord.
     */
    public long getAnnouncer()
    {
	return(announcer);
    }

    /**
     * Returns the announcer of this NameRecord.
     */
    public long getINRuid()
    {
	return(INRuid);
    }

    /**
     * Returns the RouteEntry of this NameRecord.
     */
    public RouteEntry getRouteEntry()
    {
	return(routeEntry);
    }

    public HostRecord getHostRecord()
    {
	return hostRecord;
    }

/*    public int getUDPPort()
    {
	return(hostRecord.UDPport);
    }
*/

    /**
     * Returns the cached data of this NameRecord, if any, or null otherwise.
     */  
    public byte[] getCachedData()
    {
	if (cache == null)
	    return(null);

	return(cache.getData());
    }

    /**
     * Sets the cached data of this NameRecord to be data.
     */
    public void setCachedData(byte[] data)
    {
	if (cache == null) {
	    cache = new Cache(data, lifetime);
	} else {
	    cache.update(data, lifetime);
	}
    }

    /**
     * Returns the initial lifetime of cached data for this NameRecord.
     */
    public long getCacheLifetime()
    {
	return(lifetime);
    }
    

    /**
     * Returns the remainling lifetime of this NameRecord in milliseconds.
     */
/*    public long getTimeLeft()
    {
	Date now = new Date();
	
	if (now.after(expireTime)) {
	    return(0);
	} else {
	    return (expireTime.getTime() - now.getTime());
	}
    }*/

    /** 
     * Returns true if the NameRecord is expired.
     */
    public boolean testExpiration()
    {
	return testExpiration(new Date());
    }

    /** 
     * Returns true if the NameRecord is expired.
     */
    public boolean testExpiration(Date now)
    {
	if (expireTime==null)	// never expire
	    return false;

	if (now.after(expireTime)) {
	    // Tell the parents I'm expired

	    for (Enumeration e = getParents(); e.hasMoreElements(); ) {
		ValueElement p = (ValueElement)e.nextElement();

		p.removeNameRecordHere(this);
	    }

	    return(true);
	} else {
	    return(false);
	}
	
    }

    public void removeParentPointers()
    {
	    // Tell the parents I'm expired

	    for (Enumeration e = getParents(); e.hasMoreElements(); ) {
		ValueElement p = (ValueElement)e.nextElement();

		p.removeNameRecordHere(this);
	    }
    }

    public void setExpireTime(long rl)
    {
	// Set the expire time
	if (rl==-1)
	    expireTime = null;
	else
	{
	    Date now = new Date();
	    expireTime = new Date(now.getTime() + rl);
	}

    }

    public void setINRuid(long uid)
    {
	INRuid = uid;
    }

    public void setRouteEntry(RouteEntry re)
    {
	routeEntry = re;
    }

    /**
     * Adds ValueElement p as a parent of this NameRecord.
     */
    void addParent(ValueElement p)
    {
	parents.addElement(p);
    }

    /**
     * Returns an Enumeration over the parents of this NameRecord.
     */
    Enumeration getParents()
    {
	return(parents.elements());
    }

    /**
     * Returns a String representation of the NameRecord.
     */
    public String toString()
    {
	StringBuffer output = new StringBuffer();

	Date now = new Date();
	output.append( "[" + id + ": ");
	
	if (isNeighbor) {
	    output.append("Nbr. #");
	    output.append((getNeighbor()==null)?"null":getNeighbor().toString());
	} else {
	    output.append("App. " + handler.toString());
	}

	output.append(" (" + appMetric + ") - " + "Expires in " +
	 ((expireTime==null)?"never":durationToString(expireTime.getTime() - now.getTime())) +
	    ", Cache: " + cache + ", announced by " + announcer + "]");
    
	return(output.toString());
    }


    private static int getNextId() 
    { 
	int result;
	synchronized (next_id_lock) { result = next_id++; }
	return result;
    }

    /** 
     * Added by Magda for use in the peer-to-peer networking simulator
     */ 
    public int getID() {
	return id;
    }

    /**
     * Returns whether this NameRecord's destination is different
     * from another NameRecord's destination 
     */
    public boolean destinationsEqual(NameRecord re) 
    {
	if (re.isNeighbor() != isNeighbor) return false;
	if (isNeighbor) {
	    return ((re.routeEntry.nextHop.equals(routeEntry.nextHop)) &&
		    (re.getHostRecord().equals(hostRecord)));
	} else {
	    return (handler.equals(re.getHandler()));
	}
    }


    private String durationToString(long d)
    {
	long years, days, hours, minutes, seconds;

	years = d / 31536000000l;
	d = d % 31536000000l;

	days = d / 86400000l;
	d = d % 86400000l;

	hours = d / 3600000l;
	d = d % 3600000l;

	minutes = d / 60000l;
	d = d % 60000l;

	seconds = d / 1000l;

	String output = "";

	if (years == 1l) {
	    output = output + "1 year ";
	} else if (years > 1l) {
	    output = output + years + " years ";
	}

	if (days == 1l) {
	    output = output + "1 day ";
	} else if (days > 1l) {
	    output = output + days + " days ";
	}

	if (hours == 1l) {
	    output = output + "1 hour ";
	} else if (hours > 1l) {
	    output = output + hours + " hours ";
	}

	if (minutes == 1l) {
	    output = output + "1 minute ";
	} else if (minutes > 1l) {
	    output = output + minutes + " minutes ";
	}

	if (seconds == 1l) {
	    output = output + "1 second ";
	} else if (seconds > 1l) {
	    output = output + seconds + " seconds ";
	}

	return(output);
    }
}
