package ins.namespace;

import java.util.Vector;
import java.util.Enumeration;
import java.lang.ArrayIndexOutOfBoundsException;

/**
 * This class represents a set of NameRecord objects.
 */
public class NameRecordSet
{
    // VARIABLES

    // if true, NameRecordSet contains all NameRecord objs
    private boolean allRouteEntries; 
    
    private Vector routeEntries;     // Vector contains type NameRecord

    // Representation Invariant:
    // if (allRouteEntries == true) routeEntries.isEmpty() = true;
    // routeEntries is sorted in ascending order of NameRecord.id
    
    // CONSTRUCTORS
    
    /**
     * Creates an empty NameRecordSet.
     */
    NameRecordSet()
    {
	allRouteEntries = false;
	routeEntries = new Vector();
    }
    
    // METHODS

    /**
     * Adds all the NameRecord objects to the NameRecordSet.
     */
    void addAllRouteEntries()
    {
	allRouteEntries = true;
	routeEntries.removeAllElements();
    }

    /**
     * Adds the NameRecord r to the NameRecordSet.
     */
    void addNameRecord(NameRecord r)
    {
	// If it's expired, don't add it.
	if (r.testExpiration()) {
	    return;
	}

	// If we're storing allRouteEntries, it's already in here.
	if (allRouteEntries) {
	    return;
	}

	// Get rid of expired entries.
	expireRouteEntries();

	// If routeEntries is empty, we can just add it.
	if (routeEntries.isEmpty()) {
	    routeEntries.addElement(r);
	    return;
	}

	// We've got at least one entry.
	for (int i=0; i < routeEntries.size(); i++) {
	    NameRecord r2 = (NameRecord)routeEntries.elementAt(i);

	    // It's already in there.
	    if (r2.id == r.id) {
		return;
	    }

	    // This is the spot.
	    if (r2.id > r.id) {
		routeEntries.insertElementAt(r, i);
		return;
	    }
	}

	// We didn't find a spot for it, so we insert it at the end.
	routeEntries.addElement(r);
    }

    /**
     * Removes the NameRecord r from the NameRecordSet.
     */
    void removeNameRecord(NameRecord r)
    {
	// Can't really remove a single route from allRouteEntries.
	if (allRouteEntries) {
	    return;
	}

	routeEntries.removeElement(r);
    }

    /**
     * Intersects the NameRecordSet with NameRecordSet s.
     * (r = r intersect s)
     */
    void intersectWith(NameRecordSet s)
    { 
	int i, j;        // index in this, s
	NameRecord m, n; // NameRecord at i, j
 
	// If s contains allRouteEntries, intersection with it is same.
	if (s.allRouteEntries) {
	    return;
	}

	// If this contains allRouteEntries, copy routes from s.
	if (allRouteEntries) {
	    allRouteEntries = false;
	    for (j=0; j < s.routeEntries.size(); j++) {
		routeEntries.addElement(s.routeEntries.elementAt(j));
	    }

	    return;
	}

	// Neither contains allRouteEntries; see which ones in this are in s.

	// Get rid of expired entries.
	expireRouteEntries();

	// Start with the first element of each Vector.
	i = 0;
	j = 0;
	try {
	    m = (NameRecord)routeEntries.elementAt(i);
	    n = (NameRecord)s.routeEntries.elementAt(j);

	    // We'll eventually break out of this infinite loop with an
	    // exception, when we get to the end of one of the Vectors,
	    // since each option of the if statement makes progress towards
	    // the end.
	    while (true) {
		if (n.id == m.id) {
		    // The id is the same; we keep the element and move on
		    // in both Vectors.

		    i++;
		    m = (NameRecord)routeEntries.elementAt(i);
		    j++;
		    n = (NameRecord)s.routeEntries.elementAt(j);
		} else if (n.id < m.id) {
		    // j is behind; help it catch up by increasing it.

		    j++;
		    n = (NameRecord)s.routeEntries.elementAt(j);
		} else { // (n.id > m.id)
		    // i is behind; help it catch up by removing the
		    // current element, which slides everything else back.

		    routeEntries.removeElementAt(i);
		    m = (NameRecord)routeEntries.elementAt(i);
		}
	    }
	} catch (ArrayIndexOutOfBoundsException e) {
	    // If we've come to the end of i, then we can stop.
	    // If we've come to the end of j, we delete everything else in i,
	    // and then we can stop.

	    if (j == s.routeEntries.size())
		routeEntries.setSize(i);

	    // falls through to return
	}
    }

    /**
     * Unions the NameRecordSet with NameRecordSet s.
     * (r = r union s)
     */
    void unionWith(NameRecordSet s)
    { 
	int i;               // index in this
	NameRecord m;        // NameRecord in this at i
	NameRecord n = null; // Route entry in s; null to placate compiler
 
	// If this contains allRouteEntries, it still does.
	if (allRouteEntries)
	    return;

	// If s contains allRouteEntries, now this does too.
	if (s.allRouteEntries) {
	    allRouteEntries = true;
	    routeEntries.removeAllElements();
	    return;
	}

	// Neither contains allRouteEntries; add ones from s to this.

	// Get rid of expired entries.
	expireRouteEntries();

	Enumeration e = s.getRouteEntries();
	try {
	    // If this ends before s, we'll get an exception and then we can 
	    // add everything from s onto the end of this.

	    // Start with the first element of this
	    i = 0;

	    // Add the elements of s to the right spot in this
	    while (e.hasMoreElements()) {
		n = (NameRecord)e.nextElement();

		// Try to fetch the NameRecord for this
		m = (NameRecord)routeEntries.elementAt(i);

		// Keep on going through routeEntries until at the right spot
		/*
		while (m.id <= n.id) {
		    i++;
		    m = (NameRecord)routeEntries.elementAt(i);
		}
		*/
		// edited by sarfraz
		while (m.id < n.id) {
		    i++;
		    m = (NameRecord)routeEntries.elementAt(i);
		}
		if (m.id != n.id)

		// Stick it here, and advance i to the element it was at before
		// the insert.
		    routeEntries.insertElementAt(n,i);
		i++;
	    }
	} catch (ArrayIndexOutOfBoundsException ex) {
	    // We've come to the end of this, add everything from s.

	    routeEntries.addElement(n);
	    while (e.hasMoreElements()) {
		n = (NameRecord)e.nextElement();
		routeEntries.addElement(n);
	    }

	}
    }

    /**
     * Return true if the NameRecordSet is empty.
     */
    public boolean isEmpty()
    {
	// Get rid of expired entries.
	expireRouteEntries();

	return((! allRouteEntries) && routeEntries.isEmpty());
    }

    /**
     * Return an Enumeration over the NameRecord objects in the set.
     */
    public Enumeration getRouteEntries()
    {
	// Get rid of expired entries.
	expireRouteEntries();

	return routeEntries.elements();
    }

    public NameRecord[] toArray()
    {
	// Get rid of expired entries.
	expireRouteEntries();

	NameRecord[] a = new NameRecord[routeEntries.size()];

	routeEntries.copyInto(a);

	return(a);
    }

    private void expireRouteEntries()
    {
	java.util.Date now = new java.util.Date();
	for (Enumeration e = routeEntries.elements(); e.hasMoreElements(); ) {
	    NameRecord r = (NameRecord)e.nextElement();

	    if (r.testExpiration(now)) {
		routeEntries.removeElement(r);
	    }
	}
    }

    /**
     * Returns a String representation of this NameRecordSet.
     */
    public String toString() 
    {
	// Get rid of expired entries.
	expireRouteEntries();

	String output;

	if (allRouteEntries) {
	    return("{ * }");
	}

	output = "{";

	for (Enumeration e = routeEntries.elements(); e.hasMoreElements(); ) {
	    output = output + ((NameRecord)e.nextElement()).id;
	    if (e.hasMoreElements()) {
		output = output + ", ";
	    }
	}

	output = output + "}";

	return(output);
    }

}
