package ins.namespace;

import java.lang.Math;
import java.lang.Runtime;
import ins.inr.*;

/**
 * This program tests the performance of routing.
 * @see RouteTree
 * @see NameSpecifier
 * @author Elliot Schwartz <elliot@mit.edu>
 */
public class TestRoutePerf
{

    /**
     * Test routing performance.
     */
    public static void main(String[] args)
    {
	boolean b=false;
	if (args.length >0) 
	    b= args[0].equals("index");

	Parameter[] p = new Parameter[6];


	//                        low  high   inc   def
	p[0] = new Parameter( "d", 0,     4,    0,    3);
	p[1] = new Parameter("ra", 2,    10,    0,    3);
	p[2] = new Parameter("rv", 1,    10,    0,    3);
	p[3] = new Parameter("na", 1,     3,    0,    2);
	p[4] = new Parameter( "n", 100,  15000,  100,   50);
	p[5] = new Parameter( "l", 1000, 1000,  0, 1000); // No inc of l.
	
	runPerfTests(p, b);
    }

    /**
     * Runs many performance tests.
     */
    private static void runPerfTests(Parameter[] p, boolean useIndex)
    {
	boolean repeat_param;
	
	// Repeat the first run of this parameter to avoid caching problems.
	repeat_param = false;
	
	// For each parameter
	for (int i=0; i<6; i++) {
	    
	    // Skip over this parameter if there's no increment.
	    if (p[i].i == 0)
		continue;
	    
	    // Set all parameters to the default
	    System.out.print("# Constants:");
	    int[] params = new int[6];
	    for (int j=0; j<6; j++) {
		params[j] = p[j].d;
		if (j != i) {
		    System.out.print(" " + p[j].n + "=" + p[j].d);
		}
	    }
	    System.out.println("\n" + "# Variables: "+p[i].n+" lps");
	    
	    // Range through the chosen parameter
	    for (int k=p[i].l; k<=p[i].h; k+=p[i].i) {
		params[i] = k;
		int range_repeat = 1;
		for (int rr=0; rr < range_repeat; rr++) {
		    Result result
			= perfTest(params[0],  // d
				   params[1],  // ra
				   params[2],  // rv
				   params[3],  // na
				   params[4],  // n
				   params[5],
				   useIndex); // l
		    System.gc();

		    System.out.println(k + "\t" +
				       (float)params[5]*1000.0/
				       (float)result.time + "\t" +
				       result.memory);
		}
	    }

	    System.out.print("\nNEXT SET\n\n");

	    // Repeat if first run of this parameter.
	    if (repeat_param) {
		i--;
		repeat_param = false;
	    }
	}
    }

    /**
     * Runs the performance test.
     * @param d the depth of the RouteTree
     * @param ra the number of possible different attributes in the RouteTree
     * @param rv the number of possible different values in the RouteTree
     * @param na the number of attributes for each Route/NameSpecifier
     * @param n the number of routes in the RouteTree
     * @param l the number of lookups to do
     * @return the Result of the test
     */
    private static Result perfTest(int d, int ra, int rv, int na, int n, int l, boolean useIndex)
    {
	NameStoreInterface rt;       // RouteTree to run tests on.
	NameSpecifier[] ns; // The NameSpecifiers that are in rt.
	NameRecord[] re;    // The RouteEntries that are in rt.
	int[] rand;         // Random numbers.
	long time;          // For computing the elapsed time.

	// Preliminary sanity checks.

	if (na > ra) {
	    System.out.println("na must be <= ra!");
	    return new Result();
	}

	// Create the new RouteTree.
	if (useIndex) {
	    //System.out.println("*** using index ***");
	    rt = new ins.namespace.index.NameIndex(10000);
	}
	else
	    rt = new NameTree();

	// Add RouteEntries with random NameSpecifiers to it. Keep a list of
	// the NameSpecifiers and Route Entries.
	ns = new NameSpecifier[n];
	re = new NameRecord[n];

	// RouteTree creation.
	// time = System.currentTimeMillis();

	for (int i=0; i<n; i++) {
	    
	    NameSpecifier s = createNameSpecifier(d, ra, rv, na);
	    ns[i] = s;

	    NameRecord r = new NameRecord(null, 0, -1, 0, false, 0);

	    re[i] = r;

	    rt.addNameRecord(s, r);
	}
	// time = System.currentTimeMillis() - time;
	// System.out.println("Route tree created in "+time+" ms.");

	// RouteTree extraction.
	// time = System.currentTimeMillis();
	//	for (int i=0; i<e; i++) {
	//	    rt.getNameSpecifier(re[random(n)]);
	//	}
	// time = System.currentTimeMillis() - time;
	// System.out.println("Extracted " + e + 
	//                    " NameSpecifiers in "+time+" ms.");

	// RouteTree lookups.

	// Pregenerate random numbers
	rand = new int[l];
	for (int i=0; i<l; i++) {
	    rand[i] = random(n);
	}
	
	NameRecord[] lres;

	System.gc();
	time = System.currentTimeMillis();
	for (int i=0; i<l; i++) {
	    lres = rt.lookup(ns[rand[i]]);
	}
	time = System.currentTimeMillis() - time;
	//System.out.println("Looked up "+l+" Routes in "+time+" ms.");

	Runtime runtime = Runtime.getRuntime();
	long total, free;
	ns = null;
	re = null;
	rand = null;
	System.gc();
	total = runtime.totalMemory();
	free = runtime.freeMemory();

	Result result = new Result();
	result.time = time;
	result.memory = total-free;
	return(result);
    }

    /**
     * Creates a random NameSpecifier.
     * @param d the depth of the NameSpecifier
     * @param ra the number of possible different attributes
     * @param rv the number of possible different values
     * @param na the actual number of attributes used at each level
     * @return the random NameSpecifier.
     */
    private static NameSpecifier createNameSpecifier(int d, int ra, int rv,
						     int na)
    {
	// Create the new NameSpecifier
	NameSpecifier ns = new NameSpecifier();

	// Check if the requested depth is zero.
	if (d == 0) {
	    return(ns);
	}

	// If not, create the first level of AVelements.

	// Create an array to make sure we don't reuse attribute numbers.
	boolean[] used = new boolean[ra];
	
	// Create na AVelements.
	for (int i=0; i < na; i++) {

	    // Pick a random number between 0 and ra-1,
	    // and keep picking one if it's already used.
	    int a;
	    do {
		a = random(ra);
	    } while (used[a]);
	    
	    // Mark it as being used.
	    used[a] = true;

	    // Pick a value to use; it's alright to reuse values.
	    int v;
	    v = random(rv);

	    // Create the AVelement by calling createAVelement to
	    // recursively fill in the rest of the NameSpecifier.
	    AVelement ave = createAVelement(new Attribute(Integer.toString(a)),
					    new Value(Integer.toString(v)),
					    d - 1, ra, rv, na);

	    // Add the AVelement to the NameSpecifier.
	    ns.addAVelement(ave);
	}
	
	return(ns);
    }

    /**
     * Creates a random AVelement.
     * @param d the depth of the NameSpecifier from this AVelement
     * @param ra the number of possible different attributes
     * @param rv the number of possible different values
     * @param na the actual number of attributes used at each level
     * @return the random AVelement.
     */
    private static AVelement createAVelement(Attribute attribute, Value value,
					     int d, int ra, int rv, int na)
    {
	// Create the new AVelement
	AVelement avElement = new AVelement(attribute, value);

	// Check if we're at the end of the recursion.
	if (d == 0) {
	    return(avElement);
	}

	// If not, create the next level of AVelements.

	// Create an array to make sure we don't reuse attribute numbers.
	boolean[] used = new boolean[ra];
	
	// Create na AVelements.
	for (int i=0; i < na; i++) {

	    // Pick a random number between 0 and ra-1,
	    // and keep picking one if it's already used.
	    int a;
	    do {
		a = random(ra);
	    } while (used[a]);
	    
	    // Mark it as being used.
	    used[a] = true;

	    // Pick a value to use; it's alright to reuse values.
	    int v;
	    v = random(rv);
	    
	    // Create the child AVelement by calling createAVelement to
	    // recursively fill in the rest of the AVelement.
	    AVelement ave = createAVelement(new Attribute(Integer.toString(a)),
					    new Value(Integer.toString(v)),
					    d - 1, ra, rv, na);

	    // Add the child AVelement to our AVelement.
	    avElement.addAVelement(ave);
	}

	return(avElement);
    }

    /**
     * Creates a RouteTree with depth of rd (i.e. tree has 2rd levels),
     * ra attributes/value, rv values/attribute.

    private static RouteTree createRouteTree(int rd, int ra, int rv)
    {
	RouteTree r;
	
	r = new RouteTree();

	// Check if the requested depth is zero.
	if (rd == 0) {
	    return(r);
	}

	// If not, create the first level of attribute and value elements.

	for (int i=0; i<ra; i++) {
	    
	    AttributeElement a;

	    a = new AttributeElement(new Attribute(Integer.toString(i)));

	    for (int j=0; j<rv; j++) {

		ValueElement v2;

		v2 = recursiveCreate(j, rd - 1, ra, rv);

		a.addValueElement(v2);
	    }

	    r.addAttributeElement(a);
	}

	return(r);
    }
     */

    /*
    public static void foo() {

	// Check if the requested depth is zero.
	if (rd == 0) {
	    return(r);
	}

	// If not, create the first level of attribute and value elements.

	for (int i=0; i<ra; i++) {
	    
	    AttributeElement a;

	    a = new AttributeElement(new Attribute(Integer.toString(i)));

	    for (int j=0; j<rv; j++) {

		ValueElement v2;

		v2 = recursiveCreate(j, rd - 1, ra, rv);

		a.addValueElement(v2);
	    }

	    r.addAttributeElement(a);
	}

	return(r);
    }
    */

    /**
     * Recursively creates part of a route tree with a remaining depth of
     * rd, ra attributes/value, rv values/attribute

    private static ValueElement recursiveCreate(int id, int rd, int ra, int rv)
    {

	ValueElement v;

	v = new ValueElement(new Value(Integer.toString(id)));

	// Check if we're at the end of the recursion.
	if (rd == 0) {
	    return(v);
	}

	// If not, create the next level of attribute and value elements.

	for (int i=0; i<ra; i++) {
	    
	    AttributeElement a;

	    a = new AttributeElement(new Attribute(Integer.toString(i)));

	    for (int j=0; j<rv; j++) {

		ValueElement v2;

		v2 = recursiveCreate(j, rd - 1, ra, rv);

		a.addValueElement(v2);
	    }

	    v.addAttributeElement(a);
	}

	return(v);
    }
     */

    /**
     * Returns a random number between 0 and x - 1.
     */
    final static int random(int x)
    {
	double r;
	
	do {
	    r = Math.random();
	} while (r == 1.0);

	return((int)(r * x));
    }

    private static void usage()
    {
	System.err.println("Usage: java namespace.TestRoutePerf n");
	System.err.println("          n = the number of routes");
	System.exit(-1);
    }

}	

class Result
{
    public long time;
    public long memory;
}

class Parameter
{
    public String n;
    public int l, h, i, d;

public Parameter(String n, int l, int h, int i, int d)
{
    this.n = n;
    this.l = l;
    this.h = h;
    this.i = i;
    this.d = d;
}
}



