package ins.namespace;

import java.util.Vector;
import java.util.Enumeration;

/**
 * Represents a Value element in a NameTree.
 */
class ValueElement
{
    // VARIABLES
    
    Value v;
    private Vector children;         // Vector of AttributeElement
                                     // (contains orthogonal values)
    private AttributeElement parent;
    private NameRecordSet NameRecordSet;

    AVelement PTR;                   // Temporary data storage
                                     // used by an invocation of
                                     // NameRecord.extract()
                                     // Access to this field in the
                                     // entire route tree should be
                                     // serialized.
                                     //
                                     // Only not "private" because NameTree
                                     // needs to access it.

    static Attribute vspaceAttr = new Attribute("vspace");

    // CONSTRUCTORS
    
    /**
     * Creates a new ValueElement.
     * @param v the Value of the ValueElement
     */
    ValueElement(Value v)
    {
	this.v = v;
	children = new Vector(4);
	parent = null;
	NameRecordSet = new NameRecordSet();
	PTR = null;
    }

    // METHODS
    
    /**
     * Sets the parent of this ValueElement.
     * @param a the AttributeElement to set the parent to
     */
    void setParent(AttributeElement a)
    {
	parent = a;
    }

    /**
     * Add a child AttributeElement to this ValueElement.
     * @param a the AttributeElement to be added
     */
    void addAttributeElement(AttributeElement a)
    {
	a.setParent(this);
	children.addElement((Object)a);
    }

    /**
     * Adds a NameRecord to the NameRecordSet of this ValueElement.
     * @param r the NameRecord to be added
     */
    private void addNameRecordHere(NameRecord r)
    {
	r.addParent(this);
	NameRecordSet.addNameRecord(r);
    }

    /**
     * Remove a NameRecord from the NameRecordSet of this ValueElement.
     * @param r the NameRecord to be removed
     */
    void removeNameRecordHere(NameRecord r)
    {
	NameRecordSet.removeNameRecord(r);
    }

    /**
     * Returns true if this ValueElement is a leaf node.
     */
    boolean isLeaf()
    {
	return(children.isEmpty());
    }

    /**
     * Returns an Enumeration of the AttributeElements.
     */
    Enumeration getAttributeElements()
    // added by sarfraz
    {
	return(children.elements());
    }

    /**
     * Returns a NameRecordSet with all the NameRecords stored in this
     * and ValueElements in the tree rooted at this
     */
    NameRecordSet getAllNameRecords()
    // added by sarfraz
    {
        NameRecordSet S = new NameRecordSet();
        S.unionWith(NameRecordSet);
        if (!isLeaf()){
            for (Enumeration e1 = getAttributeElements(); e1.hasMoreElements();){
                AttributeElement Ta = (AttributeElement) e1.nextElement();
                for (Enumeration e2 = Ta.getValueElements(); e2.hasMoreElements();){
                    ValueElement Tv = (ValueElement) e2.nextElement();
                    S.unionWith(Tv.getAllNameRecords());
                }
            }
        }
        return S;
    }

    /**
     * Looks up (part of) a NameSpecifier.
     * @param n the part of the NameSpecifier (AVelement) to be looked up
     */
    NameRecordSet lookup(AVelement n)
    {
	// This code is the same as the code in NameTree.java

	NameRecordSet S;  // S is the NameRecordSet we're eventually going to return.

	// Start with S = the set of all possible route entries
	S = new NameRecordSet();
	S.addAllRouteEntries();

	// for each attribute-value pair p = (na,nv) in n
	for (Enumeration e1 = n.getAVelements(); e1.hasMoreElements(); ) {
	    AVelement p  = (AVelement)e1.nextElement();
	    Attribute na = p.getAttribute();
	    Value     nv = p.getValue();

	    // bad if not on the top level
	    //if (vspaceAttr.equals(na)) continue; // don't lookup vspace

	    // Ta = the child of T (this) such that name(Ta) = name(na)
	    AttributeElement Ta;
	    try {
		Ta = findAttributeElement(na);
	    } catch (ElementNotFound ex) {
		if (na.toString().equals("vspace"))      // sarfraz_edit
		    continue;                            // sarfraz_edit
		else                                     // sarfraz_edit
		    return(new NameRecordSet());         // sarfraz_edit
	    }

	    ValueElement Tv;
	    // if nv == *
	    if (nv.isWildcard()) {
		// Wildcard matching.

		NameRecordSet Sprime = new NameRecordSet();

		for (Enumeration e2 = Ta.getValueElements();
		     e2.hasMoreElements(); 
		     ) {
		    Tv = (ValueElement)e2.nextElement();
		    //		    Sprime.unionWith(Tv.NameRecordSet);
		    Sprime.unionWith(Tv.getAllNameRecords()); // sarfraz_edit
		}

		S.intersectWith(Sprime);
	    } else {
		// Normal matching.

		try {
		    Tv = Ta.findValueElement(nv);
		} catch (ElementNotFound ex) {
		    return(new NameRecordSet());
		}
	
		//		if (Tv.isLeaf() || p.isLeaf()) {
		if (p.isLeaf()) {                           // sarfraz_edit
		    //		    S.intersectWith(Tv.NameRecordSet);
		    S.intersectWith(Tv.getAllNameRecords());
		} else {
		    S.intersectWith(Tv.lookup(p));
		}
	    }
	}

	//	S.unionWith(NameRecordSet);                // sarfraz_edit

	return(S);
    }

    /**
     * Recursively add a NameRecord to this substree of the NameTree.
     * @param r the NameRecord to add
     * @param n the part of the NameSpecifier to work on from here (this)
     */
    void addNameRecord(AVelement n, NameRecord r)
    {

	// Figure out where to put it in the tree.
	for (Enumeration e = n.getAVelements(); e.hasMoreElements(); ) {
	    AVelement p  = (AVelement)e.nextElement();
	    Attribute na = p.getAttribute();
	    Value     nv = p.getValue();

	    if (vspaceAttr.equals(na)) continue; // don't add vspace

	    AttributeElement Ta;
	    try {
		Ta = findAttributeElement(na);
	    } catch (ElementNotFound ex) {
		// Didn't find the AttributeElement; create it.

		Ta = new AttributeElement(na);
		addAttributeElement(Ta);
	    }

	    ValueElement Tv;
	    try {
		Tv = Ta.findValueElement(nv);
	    } catch (ElementNotFound ex) {
		// Didn't find the ValueElement; create it.

		Tv = new ValueElement(nv);
		Ta.addValueElement(Tv);
	    }

	    // If this is the bottom of this part of the name specifier,
	    // add the route. Otherwise, to a recursive call (sort of).
	    if (p.isLeaf()) {
		Tv.addNameRecordHere(r);
	    } else {
		Tv.addNameRecord(p,r);
	    }

	}
    }

    /**
     * Finds an AttributeElement with a particular Attribute.
     * @param a1 the Attribute that is being looked for
     * @return the AttributeElement that matches
     * @exception ElementNotFound
     *  No AttributeElement with that Attribute was found.
     */
    private AttributeElement findAttributeElement(Attribute a1)
	throws ElementNotFound
    {
	for (Enumeration e = children.elements(); e.hasMoreElements();) {
	    AttributeElement ae = (AttributeElement)e.nextElement();
	    Attribute a2 = ae.a;

	    if (a1.equals(a2)) {
		return(ae);
	    }
	}

	throw(new ElementNotFound());
    }

    /**
     * Fills in the current PTR entry, if necessary, and then recursively
     * calls itself on its parent.
     */
    void extractTrace(AVelement s, Vector touched)
    {
	if (PTR != null) {
	    // This has already been done, just add the subtree s on.

	    if (s != null)
		PTR.addAVelement(s); // graft

	} else {
	    // This hasn't been done yet.

	    // Create new AVelement, and mark the PTR as touched.
	    PTR = new AVelement(parent.a, v);
	    touched.addElement(this);

	    // Add the subtree on.
	    if (s != null)
		PTR.addAVelement(s); // graft

	    // Now try to create the parent, with the AVelement we 
	    // just created as the subtree.
	    parent.getParent().extractTrace(PTR, touched);
	}
    }

    /**
     * Return a String representation of this sub-NameTree that is pretty,
     * and is indented to indentLevel.
     */
    public String toPrettyString(int indentLevel)
    {
	String output = "";
	AttributeElement a;

	for (int i = 0; i < indentLevel; i++)
	    output = output + "  ";
	output = output + v;

	String rs = NameRecordSet.toString();
	int spaces = 79 - output.length() - rs.length();

	for (int i = 0; i < spaces; i++)
	    output = output + " ";
	output = output + rs + "\n";

	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    a = (AttributeElement)e.nextElement();
	    output = output + a.toPrettyString(indentLevel+1);
	}
	
	return(output);
    }

    /** 
     * Added by Magda to compute the size in terms of children nodes
     * and the node itself
     */
    public synchronized int size()
    {
	int nbNodes = 0;
	AttributeElement a;
	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    a = (AttributeElement)e.nextElement();
	    nbNodes = nbNodes + a.size();
	}
	// This node counts if it has any childred or if it has name records
	if (( nbNodes != 0 ) || ( !NameRecordSet.isEmpty() ))
	    nbNodes++;

	return nbNodes;
    }

    /**
     * Returns a String representation of the ValueElement.
     */
    public String toString()
    {
	String output;
	AttributeElement a;

	output = "(" + v;

	for (Enumeration e = children.elements(); e.hasMoreElements(); ) {
	    a = (AttributeElement)e.nextElement();
	    output = output + " " + a.toString();
	}

	output = output + ")";

	return(output);
    }

}
