package ins.namespace.index;


import ins.namespace.*;
import java.util.*;
import java.security.*;

public class NameIndex implements NameStoreInterface {


    // constants
   
    // number of deleted records before we go back and free the slots
    static final int RECORD_HOLES_THRESHOLD = 20;
    // number MessageDigest to pool.. should be approx 1+max number of threads
    static final int NUMBER_HASH_OBJECTS = 5;
    static final int DEFAULT_DOCLIST_LENGTH = 10;
    
    // static allocation vars...
    // specify limits to the system..
    // max number of DocLists possible in an index merge.. keep high
    // max depth possible on any NameSpecifier
    static final int MAX_NAMESPECIFIER_DEPTH = 25;
    // since simple namespecifier can generate 10 doclists
    static final int MAX_DOCLISTS_MERGE = 2000;
    // max number of results returnable by a lookup
    static final int MAX_NUMBER_RESULTS = 500;


    static final String EQUALS = "=";
    static final String EMPTYSTRING = "";
    
    static final int EMPTYHASH = EMPTYSTRING.hashCode();

    // actual index array variables; MD5 -> DocList table
    int indexNextIncrement; // next incement to grow index by
    int indexSize; // number of elts in index
    int[] indexHashes; // array of MD5 hashes
    DocList[] indexLists; // array of corresponding lists

    // record list variables; int -> NameRecord ie recordList[1] = (NameRecord)
    NameRecord[] recordList; // array of NameRecords - DocLists made of
                             // indexes of recordList
    String[] recordSpecifiers; // corresponding array of strings of 'keys'
    IntStack recordOpenHoles; // empty holes in the record array that can
                              // be reused
    IntStack deadSlots = null; // records that have been deleted but not 
                               // cleaned up out of DocLists..
    int recordNextPos;  // the next free slot in recordList     
    int recListNextInc; // next increment to grow record List by

    // vars taken from NameTree implementation
    private String vspace;
    private boolean aggregate=false;

    // reverse record list mapping; NameRecord -> int
    Hashtable reverseTable;

    // static arrays used to avoid allocations..
    
    // used by lookup and addNameRecord
    String[] ancestorArray = new String[MAX_NAMESPECIFIER_DEPTH*2];           
    // used by lookup
    DocList[] docListList = new DocList[MAX_DOCLISTS_MERGE];
    int[] docListPos = new int[MAX_DOCLISTS_MERGE];
    int[] tmpResultsList = new int[MAX_NUMBER_RESULTS];

    // possible places left to optimize:
    // - dump the Hashtable... implement an intHashTable... no casting.. less
    // space..
    // - we currently keep strings of all Specifiers so that we can
    // do the extract.. might be a better less memory intensive way
    // - deletion.. expireRecords and deleteRecord both do deletes in a
    // a similar way..  delete from master list.. and then 
    // when threshold is it garbage collect by sweeping through ALL
    // DocLists.. is very expensive in large size.. so instead should
    // take advantage of the fact that we have the Specifiers..

    // NameIndex()
    // initialSize - initial size of hashtable space
    //               NEED TO INSERT SOME KIND OF CRITERIA
    //               for example: if you expect 1000 items size should be?
    public NameIndex(int initialSize) {

	indexNextIncrement = initialSize * 2;
	indexHashes = new int[initialSize];
	indexLists = new DocList[initialSize];
	indexSize = 0;

	recordList = new NameRecord[initialSize];
	recordSpecifiers = new String[initialSize];
	recordNextPos = 0;
	recListNextInc = initialSize*2;
	recordOpenHoles = new IntStack(RECORD_HOLES_THRESHOLD);
	deadSlots = new IntStack(RECORD_HOLES_THRESHOLD);

	reverseTable = new Hashtable(initialSize);

    }
    

    // rawInsert: the core insert method.. 
    //            - very simply takes the MD5 indexVal 
    //            finds the corresponding DocList and adds the int name to that
    //            - does expansion if the table is full
    // params:
    //        byte[] indexVal:  MD5 key to insert
    //        int name: index of NameRecord to insert
    // returns:  false on success, true on failure
    private final boolean rawInsert(int indexVal, int name) {
	
	int left = 0, right = indexSize, pivot;
	
	// bsearch to find where to insert..
	while (left < right)   {
	    pivot = ((right + left) >> 1);
	    
	    //	    int comp = Hash.MD5compare(indexVal, indexHashes[pivot]);
	    
	    if (indexVal < indexHashes[pivot]) {

		right = pivot;
	    }
	    else if (indexVal > indexHashes[pivot]) {
		left = pivot+1;
	    }
	    else {
		// insert if doclist is already there..
		indexLists[pivot].insert(name);
		return false;
	    }

	}
	
	// current arrays are too small to support another insert
	if (indexSize >= indexHashes.length) {
	    System.out.println("*** NameIndex: index resize "+indexSize+" -> "+(indexSize+indexNextIncrement)+" ***");
	    // need to grow everyone
	    int[] tmpHashes = null;
	    DocList[] tmpLists = null;
	    tmpHashes = new int[indexSize + indexNextIncrement];
	    tmpLists = new DocList[indexSize + indexNextIncrement];
	    // exponentially increase growth factor
	    indexNextIncrement = indexNextIncrement * 2;

	    System.arraycopy(indexHashes, 0, tmpHashes, 0, indexSize);
	    System.arraycopy(indexLists, 0, tmpLists, 0, indexSize);

	    indexHashes = tmpHashes;
	    indexLists = tmpLists;

	}

	// shuffle elements out of the way to insert..
	if (indexSize-left > 0) {
	    System.arraycopy(indexHashes, left, indexHashes, left+1, indexSize-left);
	    System.arraycopy(indexLists, left, indexLists, left+1, indexSize-left);
	}
	
	// do the insert
	indexHashes[left] = indexVal;
	indexLists[left] = new DocList(DEFAULT_DOCLIST_LENGTH, name);
	indexSize++;
	
	return false;
	
    }

    // insertAVelement: insert an NameRecord index into appropriate DocLists
    //                  described by the AVelement
    // 
    // params: 
    //         AVelement ave:  the avelement that describes the NameRecord
    //                         - however not the root AVElement
    //                           since we expect ave to have attribs which
    //                           root from NameSpecifier does not
    //         String[] ancestors:  a caller allocated array for storing
    //                              each level of recursion so that
    //                              the MD5 can hash over all ancestors
    //                              will over run array if 2*depth is greater
    //                              than actual recursion depth
    //         int depth:           depth describing current execution
    //                              used to figure out how many levels
    //                              to hash over and where to insert this
    //                              level's strings.
    //         int record:          record to insert
    // effects: inserts record into the appropriate DocLists specified by
    //          ave
    // 
    // returns: false on success, true on failure
    // problems: recursive implementation is slow.. should be done 
    //           iteratively for more perf...
    private final boolean insertAVelement(AVelement ave, String[] ancestors, int depth, int record) {
	int hashval;
	Enumeration e;
	StringBuffer sb = new StringBuffer();

	// hashing attribute name, is parent of value
	ancestors[depth] = ave.getAttribute().toString();
	
	for (int i=0; i < depth+1; i++) {
	    sb.append(ancestors[i]);
	    sb.append(EQUALS);
	}

	hashval = sb.toString().hashCode();
	
	// insert by attribute
	rawInsert(hashval, record);
		
	if (ave.getValue().isWildcard()) {
	    // if its a wildcard.. do nothing.. cause
	    // we treat wildcards as not there in the index.
	    // better hope that this was at the end and not in the
	    // middle of an fully qualified branch.. cause index
	    // does not do wildcards in the middle
	}
	else {
	    sb = new StringBuffer();
	    // hash value name.. 
	    ancestors[depth+1] = ave.getValue().toString();

	    for (int i=0; i < depth+2; i++) {
		sb.append(ancestors[i]);
		sb.append(EQUALS);
	    }

	    hashval = sb.toString().hashCode();
	
	    // insert by value
	    rawInsert(hashval, record);
	    
	    e = ave.getAVelements();
	    // recurse on children..
	    while (e.hasMoreElements()) {
		insertAVelement((AVelement)e.nextElement(), ancestors, depth+2, record);
	    }
	}
	return false;
    }

    // returns number of doclists it got from dl..
    // 
    // lookupAVelement: find all DocLists pointed to by AVelement - recursive
    // 
    // params: 
    //         AVelement ave:  the avelement that is being looked up
    //                         - however not the root AVElement
    //                           since we expect ave to have attribs which
    //                           root from NameSpecifier does not
    //         String[] ancestors:  a caller allocated array for storing
    //                              each level of recursion so that
    //                              the MD5 can hash over all ancestors
    //                              will over run array if 2*depth is greater
    //                              than actual recursion depth
    //         int depth:  depth describing current execution
    //                     used to figure out how many levels
    //                     to hash over and where to insert this
    //                     level's strings.
    //         DocList[] dl:    caller allocated array to place results
    //                          ArrayBoundsException if too many results found
    //         int record:          record to insert
    // effects:  puts results in dl[] from lookup
    // returns:  number of DocLists found so far.recursive calls increment this
    // problems: recursive implementation is slow.. should be done 
    //           iteratively for more perf...

    private final int lookupAVelement(AVelement ave, String[] ancestors, int depth, DocList[] dl, int pos) {
	
	Enumeration e;
	int hashval;
	StringBuffer sb = new StringBuffer();

	// hash attrib and parents
	ancestors[depth] = ave.getAttribute().toString();

	for (int i=0; i< depth+1; i++) {
	    sb.append(ancestors[i]);
	    sb.append(EQUALS);
	}

	hashval = sb.toString().hashCode();

	// do the lookup.. only add if not null
	if ((dl[pos] = rawLookup(hashval)) != null) {
	    pos++;
	}

	if (ave.getValue().isWildcard()) {
	    // if its a wildcard.. do nothing.. cause
	    // we treat wildcards as not there in the index.
	    // better hope that this was at the end and not in the
	    // middle of an fully qualified branch.. cause index
	    // does not do wildcards in the middle
	    // free hash
	}
	else {
	    sb = new StringBuffer();
	    
	    ancestors[depth+1] = ave.getValue().toString();

	    for (int i=0; i< depth+2; i++) {
		sb.append(ancestors[i]);
		sb.append(EQUALS);
	    }

	    hashval = sb.toString().hashCode();

	    // lookup on value.. only add if not null
	    if ((dl[pos] = rawLookup(hashval)) != null) {
		pos++;
	    }
	    
	    // recurse on children..
	    e = ave.getAVelements();
	    while (e.hasMoreElements()) {
		pos = lookupAVelement((AVelement)e.nextElement(), ancestors, depth+2, dl, pos);
	    }
	}
	return pos;
	
    }
    
    // rawLookup:  looks up by MD5 and returns corresponding DocList
    // params:  
    //       byte[] indexVal:   MD5 key
    // returns:
    //       associated DocList of null if key not found
    private final DocList rawLookup(int indexVal) {
	int left = 0, right = indexSize, pivot;
	
	// bsearch to find where to insert..
	while (left < right)   {
	    pivot = ((right + left) >> 1);
	    
	    //int comp = Hash.MD5compare(indexVal, indexHashes[pivot]);
	    
	    if (indexVal < indexHashes[pivot]) {

		right = pivot;
	    }
	    else if (indexVal > indexHashes[pivot]) {
		left = pivot+1;
	    }
	    else {
		// found
		return indexLists[pivot];
	
	    }

	}

	// return null when nothing found
	return null;
    }

    // expireRecords: sweep through all records and delete the ones that are
    //                expired
    // effects: deletes the expired records
    //
    private final void expireRecords() {
	NameRecord nr;
	DocList masterList;
	int toFree;
	java.util.Date now = new java.util.Date() ;

	// get list of all records
	masterList = rawLookup(EMPTYHASH);

	for (int i=0; i< masterList.length; i++) {
	    nr = recordList[i];
	    if (nr.testExpiration(now)) {
		// delete... 
		masterList.delete(i);
		
		// add to slots to gc
		deadSlots.push(i);
		
		// gc if reach threshold
		if (deadSlots.curpointer+1 >= RECORD_HOLES_THRESHOLD) {
		    // garbage collect all the slots..
		    System.out.println("*** NameIndex: gc expire ***");
		    while (!deadSlots.empty()) {
			toFree = deadSlots.pop();
			// sweep through all DocLists and delete
			for (int j=0; j < indexSize; j++) {
			    indexLists[j].delete(toFree);
			}
			// put back on reusable stack..
			recordOpenHoles.push(toFree);
		    }
		
		}
	    }
	}

    }
    
    // intersect
    // none of the input arguments are mutated.
    // parameters:
    // Object[] lists - array of arrays each of which are the arrays from doclists
    // int[]    lengths - corresponding lengths of each of the arrays in lists
    // int[]    positions - corresponding to each array all initted to 0
    // int      numLists - number of elements in lists and lengths
    // int      minSize - the minimum size of all the lists
    // int[]    storeArray - caller allocated temporary array
    // returns:
    // Array of NameRecords..
    private final NameRecord[] intersect(DocList[] lists, int[] positions, int numLists, int[] storeArray) {
 
        int outputCount                   = 0; // used to generate final array
        NameRecord[] outputArray          = null;
        int currentList =                   1; // current list being examined..
                                               // start at the second one
        int compareNum     =               -1; // current number being analyzed
        int listCount   =                   1;  // number of lists this value is in..
        if (numLists < 1) {
            return null;
        }
        if (lists[0] == null) {
            return null;
        }
	
	// reset all the positions
	for (int i=0; i < numLists; i++) {
	    positions[i] = 0;
	}

	// NO LONGER NECCESSARY SINCE WE USE CALLER ALLOCATED TEMP ARRAY
	//          minSize = lists[0].length;
	
	//          for (int i=1; i< numLists; i++) {
	//              if (lists[i] == null) {
	//                  return null;
	//              }
	
	//              if (lists[i].length < minSize) {
	//                  minSize = lists[i].length;
	//              }
	//          }
 	//          storeArray          = new int[minSize];
 
        compareNum = lists[0].docs[0]; // initialize for the first one..
 
        LIST: while (true) {
            if (compareNum > lists[currentList].docs[positions[currentList]]) {
                // we have to walk the list case
                positions[currentList]++;                                                       POSITION: while (positions[currentList] < lists[currentList].length) {
                    if (compareNum >  lists[currentList].docs[positions[currentList]]) {
                        // still too low.. march on..
                        positions[currentList]++;
                    }
                    else {
                        continue LIST; // back to normal
                    }
                }
                // if we get out here it means that we have nothing...
                // and we shoud bail..
 
                break;
            }
            else {
                if (compareNum == lists[currentList].docs[positions[currentList]]) {
                    listCount++; // add 1 to the totall if it gets to all then full
                    if (listCount == numLists) {
                        // we got ourselves a canidate..
                        storeArray[outputCount] = compareNum;
                        outputCount++;
                        listCount = 1;
                        positions[currentList]++;
                        if (positions[currentList] < lists[currentList].length)
{
                            // update and roll..
                            compareNum = lists[currentList].docs[positions[currentList]];
                            continue LIST; // don't want to incrment line number..
                        }
                        else {
                            // end..
                            break;
                        }
                    }
                    // advance list
                }
                else if (compareNum < lists[currentList].docs[positions[currentList]]) {
                    // essentially a reset.. the curNumber is dead..
                    listCount = 1; // reset
                    compareNum = lists[currentList].docs[positions[currentList]];
                }
 
                currentList ++; // move to the next list..
                if (currentList == numLists) {
                    currentList = 0;  // if end of list is encountered wrap around                                  
		}
            }
        }
 
	// ok to return new Array[0]  cause the caller expects size info..
	// ok to new the array since it is returned via the interface
	outputArray = new NameRecord[outputCount];
	for (int i=0; i< outputCount; i++) {
	    outputArray[i] = recordList[storeArray[i]];
	    
	}
	return outputArray;
    
    }                                                                           

    //////////////////////////////////////////////////////////////////////
    // interface specified methods
    //////////////////////////////////////////////////////////////////////
    // who knows what this does.. stolen from NameTree
    public final void setVspace(String vspace) {
	this.vspace = vspace;
    }
    
    /**
     * Lookup NameSpecifier s in the NameTree, and return its NameRecordSet.
     */
    public final synchronized NameRecord[] lookup(NameSpecifier s) {

	Enumeration attribs;
	int lists=1; // count for the no attrib masterlist..
	
	// expire records before we do lookup
	expireRecords();

	// get the masterList.. the null Hash
	docListList[0] = rawLookup(EMPTYHASH);

	attribs = s.getAVelements();
	while (attribs.hasMoreElements()) {
	    lists = lookupAVelement((AVelement)(attribs.nextElement()), ancestorArray, 0, docListList, lists);
	}
	
	return intersect(docListList, docListPos, lists, tmpResultsList);
    }

    /**
     * Adds NameRecord r for NameSpecifier s to this NameTree.
     * Overrides addNameRecord in ValueElement.
     */
    public final synchronized void addNameRecord(NameSpecifier ns, NameRecord r) {
	int recordPos;
	Enumeration attribs;
	
	// if its expired, don't add it..
	if (r.testExpiration()) {
	    return;
	}
	
	// if there are open holes use those first..
	if (!recordOpenHoles.empty()) {
	    recordPos = recordOpenHoles.pop();
	    recordList[recordPos] = r;
	    recordSpecifiers[recordPos] = ns.toString();
	}
	else {
	    if (recordNextPos >= recordList.length) {
		System.out.println("*** NameIndex: recordList resize "+recordList.length+" -> "+(recordList.length+recListNextInc)+" ***");
		NameRecord[] tmpRecord = new NameRecord[recordList.length+recListNextInc];
		String[] tmpSpecifiers = new String[recordList.length+recListNextInc];
		recListNextInc = recListNextInc *2;
		System.arraycopy(recordList, 0, tmpRecord, 0, recordList.length);
		System.arraycopy(recordSpecifiers, 0, tmpSpecifiers, 0, recordList.length);
		recordSpecifiers = tmpSpecifiers;
		recordList = tmpRecord;
	    }
	    recordPos = recordNextPos;
	    recordNextPos++;
	    recordList[recordPos] = r;
	    recordSpecifiers[recordPos] = ns.toString();
	}
	
	// enter it into reverse table.. used for deletes..
	reverseTable.put(r, new Integer(recordPos));

	// indexing on the null root case...

	rawInsert(EMPTYHASH, recordPos);

	// insert on the attribs of the NameSpecifier
	attribs = ns.getAVelements();
	while (attribs.hasMoreElements()) {
	    insertAVelement((AVelement)(attribs.nextElement()), ancestorArray, 0, recordPos);
	}
    }

    /**
     * Extracts and returns the NameSpecifier associated with the NameRecord r.
     */
    public final synchronized NameSpecifier extract(NameRecord r) {
	// terribly inefficient that we have to keep the NameSpecifier strings
	// around
	int base;
	
	base = ((Integer)(reverseTable.get(r))).intValue();

	return new NameSpecifier(recordSpecifiers[base]);
    }

    /**
     * Removes NameRecord r from this NameTree.	
     * For now, it only removes the entry from Vector
     * later will add code to remove the unused branches of the NameTree
     */
    public final synchronized void removeNameRecord(NameRecord r) {
	// sneaky snaeky.. since we have the null record.. that
	// gets andded everytime with the other queries.. we only
	// have to delete the record from that list .. muhaha..
	DocList masterList;
	int deleteIndex, toFree=-1;

	masterList = rawLookup(EMPTYHASH);
	// find the index.. this sucks that we have to do this..
	deleteIndex = ((Integer)(reverseTable.get(r))).intValue();
	
	// delete from masterList
	masterList.delete(deleteIndex);
	
	// add to slots to gc
	deadSlots.push(deleteIndex);

	// gc if reach threshold
	if (deadSlots.curpointer+1 >= RECORD_HOLES_THRESHOLD) {
	    System.out.println("*** NameIndex: gc remove ***");
	    // garbage collect all the slots..
	    while (!deadSlots.empty()) {
		toFree = deadSlots.pop();
		// sweep through all DocLists and delete
		for (int i=0; i < indexSize; i++) {
		    indexLists[i].delete(toFree);
		}
		// put back on reusuable
		recordOpenHoles.push(toFree);
	    }
	}
	
	return;
    }

    /**
     * Returns an Enumeration of the NameRecords in this NameTree.
     */
    // maybe should be synchronized but we don't use this method for 
    // expiration anyways.. hopefully no one else uses it for anything
    // important
    public final Enumeration getNameRecords() {
	// make the empty lookup..
	//System.out.println("NameIndex: getNameRecords");
	// get masterList
	DocList dl = rawLookup(EMPTYHASH);

	return new AllRecordsEnumeration(dl);
    }

    // inner class to provide efficient enumeration
    private class AllRecordsEnumeration implements java.util.Enumeration {
	int position =0;
	DocList dl;
	public AllRecordsEnumeration(DocList d) {
	    dl = d;
	}

	public boolean hasMoreElements() {
	    return (position < dl.length);
	}

	public Object nextElement() {
	    Object ro = recordList[dl.docs[position]];
	    position++;
	    return ro;

	}

    }

    /** Is this an aggregate vspace */
    public final boolean isAggregateVspace() {
	// who knows what this is.. stolen from NameTree
	return aggregate;
    }

    public final void makeAggregateVspace() {
	// who knows what this is.. stolen from NameTree
	aggregate = true;
    }

    /**
     * Returns an String that is a printed version of the NameRecords.
     */
    public final synchronized String nameRecordsToString() {
	// expire records since this is an read
	expireRecords();
	return "needs to be implemented";
    }

    /**
     * Return a String representation of this NameTree that is pretty.
     * Overrides toPrettyString in ValueElement.
     */
    public final synchronized String toPrettyString() {
	// expire records since this is an read
	expireRecords();
	return "needs to be implemented";
    }

    /**
     * Returns a String representation of this NameTree.
     * Overrides toString in ValueElement.
     */
    public final synchronized String toString() {
	// expire records since this is an read
	expireRecords();
	return "needs to be implemented";
    }

}








