#!/bin/bash

###############################################################
# Suite for running INS/Twine Network                         # 
#                                                             #
# Author   :  Kalpak Kothari, Magdalena Balazinska            #
# Created  :  Dec 01 2001                                     #
###############################################################

$Id: run.sh,v 1.6 2002/03/21 00:01:57 mbalazin Exp $

# This is a script that helps start and stop a Twine network on multiple machines.
#
# We need the environment variable INR_HOME to set the classpath for all scripts
printf "Checking for INS_HOME"...
if [ -z $INS_HOME ] # INS_HOME not set
then
 printf "not found.\n"
 echo "Please set environment variable INS_HOME. See setclasspath.sh for more details."
 exit 0
else
 printf "found.\n\n"
fi 

#helper usage. use to print at various places.
USAGE="Usage: $0 <portsFile> <hostsFile> [help ckrstart ckrstop inrstart inrstop getlogs extractlogs cleanlogs allstart allstop]"

if [ $# -lt 2 ] 
then 
    echo "$USAGE"
    echo "   <portsFile> - the name of the file specifying the ports to use"
    echo "   <hostsFile> - the name of the file with the list of hosts and nb resolvers per host"
    echo "For verbose help, use the \"help\" option"
    exit 0
fi

# 1. Configuration
# ----------------

#########
portsFile=$1 
hostsFile=$2
vnodes=20 # Using 20 virtual nodes by default

FIRST_CKR_PORT=`awk '{ if ($1=="FIRST_CKR_PORT") print $2 }' $portsFile`
RPC_OFFSET=`awk '{ if ($1=="RPC_PORT_OFFSET") print $2 }' $portsFile`
UDP_OFFSET=`awk '{ if ($1=="UDP_PORT_OFFSET") print $2 }' $portsFile`

CURHOST="";
CURNUM=0;
j=0;

LISTCKR=()	# array which maintains list of hostnames and corresponding ckr ports
LISTRPC=()	# array which maintains list of rpc ports
LISTINR=()	# array which maintains list of inr ports

NEWHOSTCKR=	# hostname ckrport1 ckrport2 ...
NEWHOSTRPC=	# rpcport1 rpcport2 ...
NEWHOSTINR=	# inrport1 inrport2 ...

for i in `cat $hostsFile`;  # get hostnames and number of resolvers to run on each
 do 
    if [ "$j" -eq "0" ] 
	then
	    CURHOST=$i
	    NEWHOSTCKR=$CURHOST
	    NEWHOSTRPC=
	    NEWHOSTINR=
	else 
	    CURNUM=$i
	    #echo "parsed: $CURHOST $CURNUM"
	    # create list of CKR, INR
	    for ((k=0; k < CURNUM ; k++))
		do 
		    let "curckrport = $FIRST_CKR_PORT + k"
		    let "currpcport = $FIRST_CKR_PORT + $RPC_OFFSET + k"
		    let "curinrport = $FIRST_CKR_PORT + $UDP_OFFSET + k"
		    #echo "  Ports: $curckrport $currpcport $curinrport"
		    NEWHOSTCKR="${NEWHOSTCKR} ${curckrport}"
		    NEWHOSTRPC="${NEWHOSTRPC} ${currpcport}"
		    NEWHOSTINR="${NEWHOSTINR} ${curinrport}"
	    done
	    LISTCKR=("${LISTCKR[@]}" "${NEWHOSTCKR}")
	    LISTRPC=("${LISTRPC[@]}" "${NEWHOSTRPC}")
	    LISTINR=("${LISTINR[@]}" "${NEWHOSTINR}")
    fi
    let "j += 1"
    let "j %= 2"
done

# extract the first ckr node for bootstrapping
TEMPLISTCKR=${LISTCKR[0]}
FIRSTCKRNODETEMP=(`echo ${TEMPLISTCKR}`)
FIRSTCKRNODE="${FIRSTCKRNODETEMP[0]}:${FIRSTCKRNODETEMP[1]}"
unset TEMPLISTCKR
unset FIRSTCKRNODETEMP

# ---------------- end configuration ------------------------------------

# 2. Test Functions
# -----------------

# how many apps are currently sending packets
CURAPPS=0
export APP_OPT

len=${#LISTCKR[@]}

for ((i=0; i < len ; i++))
do SUBLISTCKR=${LISTCKR[$i]}
 TEMPLISTCKR=(`echo ${SUBLISTCKR}`)
 num=${#TEMPLISTCKR[@]}
 SERVER=${TEMPLISTCKR[0]};
 SUBLISTRPC=${LISTRPC[$i]}
 RPCPORTLIST=(`echo ${SUBLISTRPC}`)
 SUBLISTINR=${LISTINR[$i]}
 INRPORTLIST=(`echo ${SUBLISTINR}`)
 portnum=${#INRPORTLIST[@]}
 case "$3" in
    ckrstart)
	    for ((j=1; j < num ; j++))
	     do NODE=${TEMPLISTCKR[$j]}
	     echo -n "STARTING ckr on $SERVER:$NODE..."
	     ssh $SERVER "${INS_HOME}/ins/scripts/network/runckr.sh ${INS_HOME} $FIRSTCKRNODE $NODE $vnodes &"
	     echo -n "done"
	     echo ""
	     echo -n "sleeping 5 seconds (let chord stabilize)..."
	     sleep 5
	     echo -n "done"
	     echo ""
            done
	    ;;
    ckrstop)
	    echo -n "STOPPING all ckr nodes on $SERVER..."
	    ssh $SERVER killall ckr
	    echo -n "done"
	    echo ""
	    ;;
    inrstart)
	    for ((j=0; j < portnum ; j++))
	     do INRPORT=${INRPORTLIST[$j]}
		RPCPORT=${RPCPORTLIST[$j]}
		let "k = j+1"
		CKRPORT=${TEMPLISTCKR[$k]}
	     echo -n "STARTING INR on $SERVER:$INRPORT with rpc port $RPCPORT and ckr port ${CKRPORT}..."
	     ssh $SERVER "${INS_HOME}/ins/scripts/network/runinr.sh ${INS_HOME} ${INRPORT} ${CKRPORT} ${RPCPORT} &"
	     echo -n "done"
	     echo ""
            done
	    ;;
    inrstop)
	     echo -n "STOPPING all INRs on $SERVER..."
	     ssh $SERVER "$INS_HOME/ins/scripts/network/killinr.sh"
	     echo -n "done"
	     echo ""
	    ;;
    getlogs)
	    echo -n "Retrieving logs from $SERVER..."
	    ssh $SERVER "$INS_HOME/ins/scripts/network/tarlog.sh $SERVER"
	    scp $SERVER:/tmp/$SERVER.log.tgz .
	    echo -n "done"
	    echo ""
	    ;;
    extractlogs)
	    echo "Extracting logs for $SERVER..."
	    mkdir -p $SERVER
	    cd $SERVER
	    tar xzvf ../$SERVER.log.tgz
	    cd ..
	    echo -n "done"
	    echo ""
	    ;;
    cleanlogs)
	    echo -n "Deleting existing logs on $SERVER..."
	    ssh $SERVER "rm -rf /tmp/$USER/inr /tmp/$SERVER.log.tgz"
	    echo -n "done"
	    echo ""
	    ;;
    allstart)
	    echo "Starting INS/Twine Network"
	    eval echo "$0 $1 $2 ckrstart"
	    $0 $1 $2 ckrstart
	    $0 $1 $2 inrstart
	    echo "All done."
	    exit 0
	    ;;
    allstop)
	    echo "Stopping INS/Twine Network"
	    $0 $1 $2 ckrstop
	    $0 $1 $2 inrstop
	    echo "All done."
	    exit 0
	    ;;
    help)
	    echo "$USAGE"
	    echo ""
	    echo "Start/Stop Twine Network"
	    echo "------------------------"
	    echo "  ckrstart: start ckrs"
	    echo "  ckrstop : stop ckrs"
	    echo "  inrstart: start INRs"
	    echo "  inrstop : stop INRs"
	    echo ""
	    echo "  allstart : start all ckrs and INRs"
	    echo "  allstop  : stops all ckrs and INRs"
	    echo ""
	    echo "Log Tools:"
	    echo "----------"
	    echo "  getlogs : retrive tgz compressed twine logs from on all hosts"
	    echo "  extractlogs : uncompress retrieved logs"
	    echo "  cleanlogs : erase all twine logs on on all hosts"
	    echo ""
	    exit 0
	    ;;
    *)
	    echo "$USAGE"
	    exit 0
 esac
done

