/*
 *  This source file is copyright (C) 1999 by joeshmoe.com
 *  It may be freely used for non-commercial / development purposes.
 *  Commericial use requires the payment of a nominal license fee.
 *  Contact license@joeshmoe.com to purchase a license for commericial use.
 *
 *  It is requested that all users who modify this source code send their
 *  changes to joeshmoe.com
 * 
 *  Bugs reports / modifications can sent to bugs@joeshmoe.com.
 */

package joeshmoe.mpeg;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 *  Reads and sets ID3 tags in MPEG audio files.  Currently only V1 ID3 tags
 *  are supported.
 *  @see ID3Tagger
 **/

public class ID3Tagger {

    /**
     * reads an ID3 tag from a file.  Returns null if no tag is present.
     **/
    public static ID3Tag readTag (File pFile)
				throws IOException, FileNotFoundException
    {
				if (!pFile.exists())
						throw new FileNotFoundException (pFile.getName());

				RandomAccessFile f = new RandomAccessFile (pFile, "r");

				if (f.length() < 128) {
						f.close();
						return null;
				}

				f.seek (f.length() - 128);

				ID3Tag t = new ID3Tag();

				byte [] tagID = new byte [3];
				byte [] song = new byte [30];
				byte [] artist = new byte [30];
				byte [] album = new byte [30];
				byte [] year = new byte [4];
				byte [] comment = new byte [30];
				byte [] genre = new byte [1];
    
				f.readFully (tagID);

				if (!(new String(tagID)).equals ("TAG")) {   // no ID3 tag in file
						f.close(); 
						return null;
				}

				f.readFully(song);
				t.setSong(new String (song).trim());

				f.readFully(artist);
				t.setArtist (new String (artist).trim());

				f.readFully(album);
				t.setAlbum (new String (album).trim());

				f.readFully(year);
				t.setYear (new String (year).trim());

				f.readFully(comment);
				t.setComment (new String (comment).trim());

				f.readFully(genre);
				t.setGenreID ((int) genre[0]);

				f.close();

				return t;
    }

    /**
     * write an ID3 tag to a file.
     * Note: will write a tag to the specified file whether or not it is a
     * valid MPEG file.
     **/
    public static boolean writeTag (File pFile, ID3Tag pTag) 
				throws IOException, FileNotFoundException 
    {
				if (!pFile.exists())
						throw new FileNotFoundException (pFile.getName());

				RandomAccessFile f = new RandomAccessFile (pFile, "rw");

				// see if the tag already exists
				if (readTag (pFile) == null)
						f.seek (f.length());
				else
						f.seek (f.length() - 128);

				f.writeBytes ("TAG");

				StringBuffer song;
				if (pTag.getSong() != null)
						song = new StringBuffer(pTag.getSong());
				else
						song = new StringBuffer();
				song.setLength(30);
				f.writeBytes (song.toString());
    
				StringBuffer artist;
				if (pTag.getArtist() != null)
						artist = new StringBuffer (pTag.getArtist());
				else
						artist = new StringBuffer();
				artist.setLength(30);
				f.writeBytes (artist.toString());

				StringBuffer album;
				if (pTag.getAlbum() != null) {
						album = new StringBuffer (pTag.getAlbum());
				}
				else
						album = new StringBuffer();
				album.setLength(30);
				f.writeBytes (album.toString());
    
				StringBuffer year;
				if (pTag.getYear() != null)
						year = new StringBuffer (pTag.getYear());
				else
						year = new StringBuffer();
				year.setLength(4);
				f.writeBytes (year.toString());

				StringBuffer comment;
				if (pTag.getComment() != null) 
						comment = new StringBuffer (pTag.getComment());
				else
						comment = new StringBuffer();
				comment.setLength (30);
				f.writeBytes (comment.toString());

				f.writeByte (pTag.getGenreID());

				f.close();
	
				return true;
    }
}
