/*
 *  This source file is copyright (C) 1999 by joeshmoe.com
 *  It may be freely used for non-commercial / development purposes.
 *  Commericial use requires the payment of a nominal license fee.
 *  Contact license@joeshmoe.com to purchase a license for commericial use.
 *
 *  It is requested that all users who modify this source code send their
 *  changes to joeshmoe.com
 * 
 *  Bugs reports / modifications can sent to bugs@joeshmoe.com.
 */

package joeshmoe.mpeg;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

/**
 * Probes an MPEG audio file (usually .MP3, .MP2, or .MPE) to determine a
 * variety of information.  Currently, this class can properly handle MPEG1
 * and MPEG 2.  MPEG2.5 is not supported.  All three MPEG audio layers are
 * supported.  This class uses ID3Tagger to read an ID3 tag (if present) 
 * from MPEG audio file.  ID3Tagger currently supports only V1 ID3 tags.
 **/
public class MPEGProber {
    private static final int [] bitRateV1L1 =
    {  -1,  32,  64,  96, 128, 160, 192, 224,
       256, 266, 320, 352, 384, 416, 448,  -1 };
    private static final int [] bitRateV1L2 =
    {  -1,  32,  48,  56,  64,  80,  96, 112,
       128, 160, 192, 224, 256, 320, 384,  -1 };
    private static final int [] bitRateV1L3 =
    {  -1,  32,  40,  48,  56,  64,  80,  96,
       112, 128, 160, 192, 224, 256, 320,  -1 };
    private static final int [] bitRateV2L1 = 
    {  -1,  32,  48,  56,  64,  80,  96, 112,
       128, 144, 160, 176, 192, 224, 256,  -1 };
    private static final int [] bitRateV2L2 =
    {  -1,   8,  16,  24,  32,  40,  48,  56,
       64,  80,  96, 112, 128, 144, 160,  -1 };
    private static final int [] bitRateV2L3 = // same as V2L2
    {  -1,   8,  16,  24,  32,  40,  48,  56,
       64,  80,  96, 112, 128, 144, 160,  -1 }; 

    private static final int [] [] [] bitRates =
    { { {}, bitRateV2L1, bitRateV2L2, bitRateV2L3 },
      { {}, bitRateV1L1, bitRateV1L2, bitRateV1L3 } };

    private static final int [] [] sampleRates =
    { { 22050, 24000, 16000 },     // MPEG2
      { 44100, 48000, 32000 } };   // MPEG1

    private static final int [] bs =
    { -1, 384, 1152, 1152 };

    /**
     * Probes the MPEG audio file referenced by pFile and returns an
     * MPEGInfo object containing information about the file.
     * @exception MPEGProbeException Indicates that there were problems
     reading the information from the file.
    **/
    public static MPEGInfo probeFile (File pFile) 
				throws FileNotFoundException, IOException, MPEGProbeException
    {
				FileInputStream theInput = new FileInputStream (pFile);
				try {
				int theStart = 0;
				
				byte [] b = new byte [4];
				if (theInput.read (b) < 4)
						throw new MPEGProbeException
								("Not a valid MPEG audio file: too short");
				
				ProbedMPEGInfo theInfo = new ProbedMPEGInfo ();

				theStart += 4;
				int theHead = extractInt (b, 0, 4);
				while (!isMPEGHeader(theHead)) {
						int theByte = theInput.read (); theStart++;
						if (theByte == -1) {
								throw new MPEGProbeException
										("Not a valid MPEG audio file: cannot find first frame");
						}
						theHead <<= 8;
						theHead |= theByte;
				}

				int theVersion = (theHead >> 19) & 1;

				theInfo.setLayer (4 - (theHead >> 17) & 3);
				if (theInfo.getLayer () == 4) 
						throw new MPEGProbeException
								("Invalid layer specified in MPEG file");

				theInfo.setCRCProtected (((theHead >> 16) & 1) == 1 ? false : true);
				int theBitrate = (theHead >> 12) & 15;
				theInfo.setSamplingRate (sampleRates[theVersion][(theHead >> 10) & 3]);

				theInfo.setChannelMode (((theHead >> 6) & 3) + 1);
				int theModeExt = (theHead >> 4) & 3;
				if (theInfo.getChannelMode() == 2)
						if (theInfo.getLayer() == 3) {
								theInfo.setIntensityBands (-1);
								if ((theModeExt & 1) == 1)
										theInfo.setIntensityStereo (true);
								if ((theModeExt & 2) == 2)
										theInfo.setMSStereo (true);
						} else
								theInfo.setIntensityBands (theModeExt + 1);
				else
						theInfo.setIntensityBands (-1);

				theInfo.setCopyright (((theHead >> 3) & 1) == 1 ? true : false);
				theInfo.setOriginal (((theHead >> 2) & 1) == 1 ? true : false);

				int theOffset = 0;
				
				if (theVersion == 1) {
						theInfo.setVersion (1);
						if (theInfo.getChannelMode() != 4)
								theOffset = 32;  // stereo
						else
								theOffset = 17;  // mono
				} else if (theVersion == 2) {
						theInfo.setVersion (2);
						if (theInfo.getChannelMode() != 4) 
								theOffset = 17; // stereo
						else
								theOffset = 9; // mono
				}

				// read the ID3 tag if there is one
				theInfo.setID3(ID3Tagger.readTag (pFile));
				
				// check for VBR
				if (theInput.skip (theOffset) != theOffset)
						throw new MPEGProbeException ("Could not seek in MPEG file");
				theInput.read (b);

				String theMagic  = new String (b);
				if (theMagic.equals ("Xing") || theMagic.equals ("Lame")) {
						theInfo.setVBR (true);

						theInput.read (b);
						int theFlags = extractInt (b, 0, 4);

						int theFrames = 0;
						if ((theFlags & 0x0001) == 0x0001) {// frame count
								theInput.read (b);
								theFrames = extractInt (b, 0, 4);
						}

						if ((theFlags & 0x0002) == 0x0002) { // byte count
								theInput.read (b);
								theInfo.setFileSize(extractInt (b, 0, 4));
						}

						theInfo.setLength (((long) theFrames * 1000 * 
																bs[theInfo.getLayer()]) / 
															 theInfo.getSamplingRate());
						theInfo.setBitrate ((int) ((theInfo.getFileSize() * 8) / 
																			 theInfo.getLength()));
				} else {
						theInfo.setFileSize (pFile.length() - theStart - 4 -
																 (theInfo.getID3() != null ? 128 : 0));
						theInfo.setBitrate 
								(bitRates[theInfo.getVersion()][theInfo.getLayer()]
								 [theBitrate]);
						if (theInfo.getBitrate() == -1)
								throw new MPEGProbeException ("Illegal bitrate specified");

						theInfo.setLength 
								(theInfo.getFileSize() / (theInfo.getBitrate() / 8));
				}

				return theInfo;
				} finally {
						theInput.close();
				}
    }

    /**
     * Extracts an int from pLen bytes starting at pOffset in array pBytes
     **/
    private static int extractInt (byte [] pBytes, int pOffset, int pLen) {
				int theInt = 0;
				int i = 0;
				while (true) {
						theInt += pBytes[pOffset + i] & 0xff;
						i++;
						if (i == pLen)
								break;
						theInt <<= 8;
				}

				return theInt;
    }

    private static boolean isMPEGHeader (int pHeader) {
				/**
				 * This test shamelessly ganked from X11AMP
				 **/
				if( (pHeader & 0xffe00000) != 0xffe00000)
						return false;
				if(((pHeader>>17)&3) == 0)
						return false;
				if( ((pHeader>>12)&0xf) == 0xf)
						return false;
				if( ((pHeader>>10)&0x3) == 0x3 )
						return false;
				if( ((pHeader>>19)&1)==1 && ((pHeader>>17)&3)==3 && 
						((pHeader>>19)&1)==1 )
						return false;

				return true;
    }
}


class ProbedMPEGInfo implements MPEGInfo {
    int mVersion;
    public int getVersion () {
				return mVersion;
    }
    public void setVersion (int pVersion) {
				mVersion = pVersion;
    }

    int mLayer;
    public int getLayer () {
				return mLayer;
    }
    public void setLayer (int pLayer) {
				mLayer = pLayer;
    }

    boolean mVBR;
    public boolean isVBR () {
				return mVBR;
    }
    public void setVBR (boolean pVBR) {
				mVBR = pVBR;
    }

    int mBitrate;
    public int getBitrate () {
				return mBitrate;
    }
    public void setBitrate (int pBitrate) {
				mBitrate = pBitrate;
    }

    int mSamplingRate;
    public int getSamplingRate () {
				return mSamplingRate;
    }
    public void setSamplingRate (int pSamplingRate) {
				mSamplingRate = pSamplingRate;
    }

    int mChannelMode;
    public int getChannelMode () {
				return mChannelMode;
    }
    public void setChannelMode (int pChannelMode) {
				mChannelMode = pChannelMode;
    }

    boolean mIntensityStereo;
    public boolean isIntensityStereo () {
				return mIntensityStereo;
    }
    public void setIntensityStereo (boolean pIntensityStereo) {
				mIntensityStereo = pIntensityStereo;
    }

    boolean mMSStereo;
    public boolean isMSStereo () {
				return mMSStereo;
    }
    public void setMSStereo (boolean pMSStereo) {
				mMSStereo = pMSStereo;
    }

    int mIntensityBands;
    public int getIntensityBands () {
				return mIntensityBands;
    }
    public void setIntensityBands (int pIntensityBands) {
				mIntensityBands = pIntensityBands;
    }
		
    long mLength;
    public long getLength () {
				return mLength;
    }
    public void setLength (long pLength) {
				mLength = pLength;
    }

    long mFileSize;
    public long getFileSize () {
				return mFileSize;
    }
    public void setFileSize (long pFileSize) {
				mFileSize = pFileSize;
    }

    boolean mCRCProtected;
    public boolean isCRCProtected () {
				return mCRCProtected;
    }
    public void setCRCProtected (boolean pCRCProtected) {
				mCRCProtected = pCRCProtected;
    }
		
    boolean mCopyright;
    public boolean isCopyright () {
				return mCopyright;
    }
    public void setCopyright (boolean pCopyright) {
				mCopyright = pCopyright;
    }
		
    boolean mOriginal;
    public boolean isOriginal () {
				return mOriginal;
    }
    public void setOriginal (boolean pOriginal) {
				mOriginal = pOriginal;
    }

    ID3Tag mID3;
    public ID3Tag getID3 () {
				return mID3;
    }
    public void setID3 (ID3Tag pID3) {
				mID3 = pID3;
    }
}
