/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: daemon.cc,v 1.7 2002/10/02 22:10:32 snoeren Exp $
 *
 * Migrate API implementation.
 *
 */

#include "migrate_handler.hh"

#include "fdpass.h"

#include "daemon.hh"

void
Daemon::ravail()
{
  migrate_handler::MigrateSession *msession;
  static mmsghdr hdr;

  if(read(fd, &hdr, sizeof(hdr)) != sizeof(hdr))
    goto garbage;

  // It's for us, right? 
  ts_debug_1("Recieving message for %d", hdr.pid);
  assert(hdr.pid == getpid());

  switch(hdr.type) {

  case CONT_MSG: {
    char cont[sizeof(migrate_continuation)];
    migrate_session session;

    // Read session update
    if(read(fd, cont, sizeof(migrate_continuation)) !=
       sizeof(migrate_continuation))
      goto garbage;
    // Read session update
    if(read(fd, &session, sizeof(session)) != sizeof(session))
      goto garbage;
    
    // Notify appropriate session
    if(!(msession = migrate_handler::sessions[session.id])) {
      ts_debug_2("I don't know anything about session %d", session.id);
    } else {
      msession->update(&session, false);
      msession->cwc((migrate_continuation *)cont);
    }
    break;
  }
  case SESSION_MSG: {

    migrate_session session;

    // Read session update
    if(read(fd, &session, sizeof(session)) != sizeof(session))
      goto garbage;
    
    ts_debug_2("Received update for session %d!", session.id);
    
    // Notify appropriate session
    if(!(msession = migrate_handler::sessions[session.id])) {
      ts_debug_2("I don't know anything about session %d", session.id);
    } else {
      msession->update(&session, true);
    }
    break;
  }
  case CONNECTION_MSG: {

    migrate_connection conn;
    unsigned int len = sizeof(conn);
    int newfd = -1;

    // Recieve connection update and new fd
    if ((newfd = rcvfd(fd, &conn, &len)) < 0) {
      ts_error("Unable to receive file descriptor: %s", strerror(errno));
      goto garbage;
    }
    
    // Notify appropriate connection flow flow_handler
    if(!(msession = migrate_handler::sessions[(int)conn.session])) {
      ts_error("I don't know anything about session %d", conn.session);
    } else {
      msession->updateConn(newfd, &conn);
    }
    break;
  }

  case CONNREBIND_MSG: {

    migrate_connection conn;

    // Read session update
    if(read(fd, &conn, sizeof(conn)) != sizeof(conn))
      goto garbage;

    // Notify appropriate connection flow flow_handler
    if(!(msession = migrate_handler::sessions[(int)conn.session])) {
      ts_error("I don't know anything about session %d", conn.session);
    } else {
      msession->moveConn(&conn);
    }
    break;
  }

  default:
    ts_fatal("Unidentified message type!");
  }
      
  return;

 garbage:

  // Close up shop with the daemon
  ts_error("Migrate daemon returned garbage");
  set_ractive(false);
  set_wactive(false);
  close(fd);
  fd = -1;
  ::clients.erase(this);
  return;  
}

int
Daemon::sendto(mmsgtype msgtype, int pid, const char *buf, unsigned int buflen)
{  
    static mmsghdr hdr;

    hdr.type = msgtype;
    hdr.pid = pid;
    hdr.len = buflen;

    ts_debug_1("Sending a type %d msg to daemon", msgtype);

    if(write(fd, &hdr, sizeof(hdr)) != sizeof(hdr)) {
	ts_debug_2("sendto_daemon: %s\n", strerror(errno));
	return errno;
    }
    if(write(fd, buf, buflen) != (int)buflen) {
	ts_debug_2("sendto_daemon: %s\n", strerror(errno));
	return errno;
    }

    return 0;
}


