//
// Migrate Session Layer
//
// Alex C. Snoeren <snoeren@lcs.mit.edu>
//
// Copyright (c) 2002 Massachusetts Institute of Technology.
//
// This software is being provided by the copyright holders under the GNU
// General Public License, either version 2 or, at your discretion, any later
// version. For more information, see the `COPYING' file in the source
// distribution.
//
// $Id: migrate_handler.cc,v 1.27 2002/10/10 20:48:32 snoeren Exp $
//
// Migrate API implementation.
//

#include <hash_map>

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <ring.hh>

#include "migrate_handler.hh"


hash_map<int,migrate_handler::MigrateSession*,hash<int> > migrate_handler::sessions;
Daemon * migrate_handler::migrate_daemon = NULL;
int migrate_handler::migrate_daemon_refcnt = 0;

migrate_handler::~migrate_handler() {
  
  // Remove ourselves from the session
  if(session) {
    session->removeConn(this);
    session = NULL;
  }

  // Decrement refcount and destroy daemon connection, if necessary
  if(--migrate_daemon_refcnt == 0) {
    ts_debug_1("Last migrate flow_handler, closing daemon");
    delete migrate_daemon;
    migrate_daemon = 0;
  }

  ts_debug_1("Killing off migrate_handler, %d left", migrate_daemon_refcnt);
  
}


void
migrate_handler::may_write(flow_handler *from, bool may) {
  // Check to make sure this is an active downstream bothering us
  if(from == downstream[0]) {
    assert(upstream);
    upstream->may_write(this, may);
  } else {
    ts_debug_2("Ignoring may_write from old child");
  }
}

int
migrate_handler::ioctl(string target, int optname, string value, string& out)
{
  if (target == "migrate") {
    switch(optname) {

    case MIG_AVAIL: {
      int version = 1;
      out = string((const char *)&version, sizeof(int));
      return 0;
    }

    case MIG_SESSION_CREATE:
      if (session) {
	ts_debug_4("ioctl: session already exists");
	return -EISCONN;
      }
      session = new MigrateSession(*((int *)value.c_str()));
      // Add ourselves as the first session
      session->addConn(this);
      out = string((const char *)session->getOldSession(),
		   sizeof(migrate_session));
      ts_debug_4("ioctl: Created a new session %d", session->id());
      return 0;

    case MIG_ADD_CONN: {
      MigrateSession *newsession;
      if (value.length() != sizeof(migrate_session)) {
	ts_debug_4("Invalid migrate_session");
	return -EINVAL;
      }
      newsession = sessions[((migrate_session *)value.c_str())->id];
      if(!newsession) {
	ts_debug_4("Invalid session specified");
	return -EINVAL;
      }
      if(session) {
	ts_debug_4("Connection already has a session");
      }
      ts_debug_4("Preparing to update structures from %d to %d",
		 session ? session->id() : -1, newsession->id());
      session = newsession;
      newsession->addConn(this);
      ts_debug_4("Notifying daemon");
      updateConns();
      return 0;
    }
    case MIG_REM_CONN:
      if (!session) {
	ts_debug_4("Connection doesn't have a session");
	return -ENOTCONN;
      }
      session->removeConn(this);
      session = NULL;
      return 0;

    case MIG_SESSION_CLOSE:
      {
	// XXX: Don't actually do anything yet
	// session->close();
	return 0;
      }

    case MIG_SET_FUNC:
      ts_debug_1("Setting function name");
      if (value.length() != sizeof(migrate_lookupfunc)) {
	ts_debug_4("Invalid migrate_lookupfunc");
	return -EINVAL;
      }
      session->setlf(this, (migrate_lookupfunc *)value.c_str());
      return 0;

    case MIG_SET_NAME:
      ts_debug_1("Setting lookup name");
      session->setDname(value);
      return 0;

    case MIG_SET_STORE:
      ts_debug_1("Setting attriute/value store");
      session->setDb(*((int *)value.c_str()));
      return 0;

    case MIG_GET_SESSION:
      out = string((const char *)session->getOldSession(),
		   sizeof(migrate_session));
      return 0;

    case MIG_NEW_HP: {
      struct sockaddr_in addr;

      if(value.length() != sizeof(struct in_addr)) {
	ts_debug_4("Invalid in_addr structure");
	return -EINVAL;
      }

      memset(&addr, 0, sizeof(addr));
      memcpy(&addr.sin_addr, value.c_str(), sizeof(struct in_addr));

      ts_debug_1("Accepting new end point location: %s",
		 inet_ntoa(addr.sin_addr));
      session->setPaddr(address((const void *)&addr.sin_addr,
				sizeof(struct in_addr)));
      migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
					      (const char *)session->getOldSession(),
					      sizeof(migrate_session));
      return 0;
    }

    case MIG_SET_HANDLER: {
      ts_debug_1("Setting handler");
      // Sanity check the function
      if(value.length() != (sizeof(mig_handler) + sizeof(int))) {
	return -EINVAL;
      }
      int fd = *((int *)(value.c_str() + sizeof(mig_handler)));
      session->sethf(this, fd, *((mig_handler *)value.c_str()));
      return 0;
    }
    case MIG_MIGRATE:
      {
	migrate_session *sess = session->getOldSession();

	ts_debug_1("Requesting migrate");
	
	if(value.length() != sizeof(struct sockaddr_in)) {
	  ts_debug_4("Invalid sockaddr_in structure");
	  return -EINVAL;
	}
	
	ts_debug_1("Accepting new end point location: %s",
		   inet_ntoa(((struct sockaddr_in *)value.c_str())->sin_addr));
	
	sess->state = MIGRATE_LOST;
	memcpy(&sess->laddr, value.c_str(), value.length());
	migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
						(const char *)sess,
						sizeof(migrate_session));
	return 0;
      }
    case MIG_FREEZE:
      {
	migrate_session *sess = session->getOldSession();

	ts_debug_1("Requesting suspend");

	sess->state = MIGRATE_FROZEN;
	migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
						(const char *)sess,
						sizeof(migrate_session));
	return 0;
      }
    case MIG_CONT:
      {
	ts_debug_1("Receiving continuation");
	if (value.length() != sizeof(migrate_continuation)) {
	  ts_debug_4("Invalid migrate_continuation");
	  return -EINVAL;
	}
	assert(session);
	session->setCont(true);
	migrate_handler::migrate_daemon->sendto(CONT_MSG, session->id(),
						(const char *)value.c_str(),
						sizeof(migrate_continuation));
	return 0;
      }
    case MIG_COMPCONT:
	ts_debug_1("Receiving COMPLETE continuation info");
	migrate_daemon->sendto(COMPCONT_MSG, session->id(),
			       (const char *)value.c_str(), value.length());
	// Pass the daemon's FD down and let mighandler take it from here
	return encode_pass_fd(dup(migrate_daemon->getfd()), out);
 
    default:
      return -EINVAL;
    }
  }
  return -EINVAL;
}



void
migrate_handler::MigrateSession::init(address & a, migrate_state state)
{
  _paddr = a;  
  if(migrate_handler::migrate_daemon->is_present()) {
    // Set up a session with the daemon
    ts_debug_1("Attemping to initialize session %s to %s\n", _dname.c_str(),
	       _paddr.c_str());
    _state = state;
    migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
					    (const char *)getOldSession(),
					    sizeof(migrate_session));
  } else {
    _state = MIGRATE_NOTSUPPORTED;
  }
}


void
migrate_handler::MigrateSession::close()
{
  set_state(MIGRATE_CLOSED, true);

  ts_debug_1("Closing down session %d", _id);

  if(migrate_daemon->is_present())
    migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
			   (const char *)getOldSession(),
			   sizeof(migrate_session));

  for(hash_map<int, migrate_handler*, hash<int> >::iterator i = conns.begin() ;
      i != conns.end(); i++) {
    (*i).second->close();
    (*i).second->changeSession(NULL, NULL);
  }

  migrate_handler::sessions.erase(_id);

  delete this;
}


void
migrate_handler::MigrateSession::updateConn(int fd,
					    const migrate_connection *conn)
{
  ts_debug_1("Received %d for connection %d on session %d", fd, 
	     conn->fd, _id);
  conns[conn->fd]->migrateConn(fd, this);
}

void
migrate_handler::MigrateSession::setlf(migrate_handler *handler,
				       const migrate_lookupfunc *lf)
{
  memcpy(&_lf, lf, sizeof(migrate_lookupfunc));
  _lfhandler = handler;
}

void
migrate_handler::MigrateSession::sethf(migrate_handler *handler,
				       int ufd, mig_handler uf)
{
  _ufhandler = handler;
  _uf = uf;
  _ufd = ufd;
}

void
migrate_handler::MigrateSession::removeConn(const migrate_handler *conn)
{
  conns.erase((int)conn);

  // Support AUTOCLOSE
  if (flags() & M_AUTOCLOSE)
    if(conns.size()==0)
      close(); 
}

void
migrate_handler::MigrateSession::moveConn(const migrate_connection *conn)
{
  migrate_handler *mh = (migrate_handler *)conn->fd;
  migrate_handler::MigrateSession *newSession;

  ts_debug_1("Old session %d", id());
  if(!(newSession = migrate_handler::sessions[conn->rfd])) {
    ts_debug_2("I don't know anything about session %d", conn->rfd);
  } else {
    if(newSession != this) {
      newSession->addConn(mh);
      mh->changeSession(newSession, conn);
      removeConn(mh);
    }
  }
}


// Old-school functions


migrate_session *
migrate_handler::MigrateSession::getOldSession()
{
  static migrate_session session;

  session._fd = _ufd;
  session.id = _id;
  session.pid = _pid;
  session.state = _state;
  memcpy(&session.laddr, _laddr.addr(), _laddr.addrlen());
  memcpy(&session.paddr, _paddr.addr(), _paddr.addrlen());
  session.db = (void *)_db;
  session.flags = _flags;
  strncpy((char *)&session.dname, _dname.c_str(), M_MAXNAMESIZE);
  strncpy((char *)&session.pname, _pname.c_str(), M_MAXNAMESIZE);
  memcpy(&session.newf, &_lf, sizeof(migrate_lookupfunc));
  session.pbufsize = _pbufsize;
  session.freeze = _uf;

  return &session;
}

void
migrate_handler::MigrateSession::set_state(migrate_state state, bool notify)
{
  migrate_state oldstate = _state;

  _state = state;

  // Is this a state change?
  if(state != oldstate) {
    // Notify internal connections of state change
    for(hash_map<int,migrate_handler*,hash<int> >::iterator i = conns.begin();
	i != conns.end(); i++)
      (*i).second->stateChange(oldstate);
  }

  switch(state) {
  case MIGRATE_LOST:
    // XXX: Handle resolve
    if(_lf.func) {
      assert(_lfhandler);
      assert(_lfhandler->messages_enabled());
      ts_debug_1("Sending resolve request to app");
      _lfhandler->message(MIG_NEW_HP,
			  string((char *)&_lf, sizeof(migrate_lookupfunc)));
      break;
    } else {
      ts_debug_1("App didn't specify how to resolve");
    }
    break;

  case MIGRATE_LMIGRATING:
  case MIGRATE_PMIGRATING: 
  case MIGRATE_FROZEN:
    if(notify && _uf && _ufhandler && (state != oldstate)) {
      migrate_session * _session = getOldSession();
      _ufhandler->message((((state != MIGRATE_FROZEN) &&
			    (oldstate != MIGRATE_FROZEN)) ? MIG_MIGRATE :
			   MIG_FREEZE) , string((char *)_session,
						sizeof(migrate_session)));
    }
    break;

  default:
    // We don't do anything interesting
    break;
  }

}

void
migrate_handler::MigrateSession::cwc(migrate_continuation * cont)
{
  string buf((const char *)cont, sizeof(migrate_continuation));

  buf.append((const char *)getOldSession(), sizeof(migrate_session));

  // XXX: Can we call a continuation without a handler?
  assert(_cont && _uf && _ufhandler);
  _cont = false;
  _ufhandler->message(MIG_CONT, buf);
}

void
migrate_handler::MigrateSession::update(migrate_session * session, bool notify)
{
  // We should be updating ourself
  assert(session->id == _id);

  // Copy in stuff the daemon may have changed
  _pid = session->pid;
  _laddr = address((const void *)&session->laddr, sizeof(session->laddr));
  _pbufsize = session->pbufsize;
  
  ts_debug_1("Received update for session %d: peer %s buf %u state %d",
	     session->id, _laddr.c_str(), session->pbufsize,
	     session->state);

  // Update state
  set_state(session->state, notify);
}


migrate_connection * const
migrate_handler::getOldConn(int rfd, migrate_handler::MigrateConnection * mc) const
{
  static migrate_connection conn;

  assert(mc);

  conn.fd = _magic;
  conn.rfd = rfd;
  if((migrate_session *)mc->session())
    conn.session = (migrate_session *)mc->session()->id();
  else
    conn.session = 0;
  ts_debug_1("Creating a connection message with session id %d",
	     (int)conn.session);
  memcpy(&conn.saddr, mc->saddr().addr(), mc->saddr().addrlen());
  memcpy(&conn.daddr, mc->daddr().addr(), mc->daddr().addrlen());
  memcpy(&conn.csaddr, mc->csaddr().addr(), mc->csaddr().addrlen());
  memcpy(&conn.cdaddr, mc->cdaddr().addr(), mc->cdaddr().addrlen());
  conn.type = flow_handler::get_type();
  // conn.sync = sync;
 
  return &conn;
}

void
migrate_handler::freezeConn()
{
  assert(upstream);
  upstream->may_write(this, false);
}

void
migrate_handler::stateChange(migrate_state oldstate)
{
  switch (session->state()) {
  case MIGRATE_NOTCONNECTED:
    if(oldstate != MIGRATE_NOTCONNECTED) {
      // We should only get this if we were waiting and other side closed,
      // any other time implies the connections are likely to fail soon after.
      assert(upstream);
      ts_debug_1("Session closed, connection goes with it");
      upstream->may_write(this, true);
      if(may_avail_now())
	upstream->avail(this, data());
    }
    break;
  case MIGRATE_FROZEN:
    freezeConn();
    break;
  default:
    /* Nothing interesting to do */
    break;
  }
}


bool
migrate_handler::may_exit(void)
{
  ts_debug_1("Considering closing %d in state %d with %s", session->id(),
	     session->state(), session->cont() ? "cont" : "");
  // We can't exit until our session has shutdown
  if(session && ((session->state() != MIGRATE_CLOSED) &&
		 (session->state() != MIGRATE_NOTSUPPORTED) &&
		 (session->state() != MIGRATE_NOTCONNECTED) &&
		 !((session->state() == MIGRATE_FROZEN) && session->cont()))) {
    ts_debug_1("Migrate won't exit, session in state %d", session->state());
    return false;
  } else
    return true;
}
