//
// Migrate Session Layer
//
// Alex C. Snoeren <snoeren@lcs.mit.edu>
//
// Copyright (c) 2001-2 Massachusetts Institute of Technology.
//
// This software is being provided by the copyright holders under the GNU
// General Public License, either version 2 or, at your discretion, any later
// version. For more information, see the `COPYING' file in the source
// distribution.
//
// $Id: migrate_handler.hh,v 1.22 2002/10/08 19:02:59 snoeren Exp $
//
// Migrate TESLA Flow Handler
//

#ifndef HH_MIGRATE_HANDLER
#define HH_MIGRATE_HANDLER

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <hash_map>

#include <tesla/flow_handler.hh>
#include <tesla/client.hh>

#include "daemon.hh"

class migrate_handler : public flow_handler {

public:

  class MigrateSession;

  static Daemon *  migrate_daemon;
  static int       migrate_daemon_refcnt;
  static hash_map<int, MigrateSession*, hash<int> > sessions;

  class MigrateSession {

  private:

    int _id;                        // Local session ID
    int _pid;                       // ID at remote endpoint
    
    migrate_state   _state;         // Session state
    
    address         _laddr;         // Local endpoint
    address         _paddr;         // Remote endpoint
    string          _pname;         // Remote hostname
    string          _dname;         // Local hostname
    
    int             _db;            // Attribute/value store id
    int             _flags;         // Session flags 

    int             _pbufsize;      // Default peer RCVBUF size

    bool            _cont;          // Have we registered a continuation?

    migrate_lookupfunc _lf;         // Function to find peer
    migrate_handler   *_lfhandler;  // Handler to call function in
    int                _ufd;        // TESLA-internal fd
    mig_handler        _uf;         // Function to handle updates
    migrate_handler   *_ufhandler;  // Handler to notify with session updates

    hash_map<int, migrate_handler*, hash<int> > conns; // Member connections
    
  public:
    
    MigrateSession(int flags) : _id(-1), _state(MIGRATE_NOTCONNECTED),
				_db(0), _flags(flags), _pbufsize(0),
				_cont(false), _lfhandler(NULL),
				_ufd(-1), _ufhandler(NULL)
    {
      migrate_session session;
      mmsghdr hdr;

      // Initialize parameters
      memset(&_lf, 0, sizeof(migrate_lookupfunc));
      
      if(migrate_handler::migrate_daemon->is_present()) {
      
	// Notify daemon of our existance
	migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
						(const char *)getOldSession(),
						sizeof(migrate_session));

	// XXX: This is a god-awful hack and needs to be fixed ASAP
      waiting:
	if(::read(migrate_handler::migrate_daemon->getfd(),
		&hdr, sizeof(hdr)) != sizeof(hdr))
	  ts_fatal("Read failed miserabily %s", strerror(errno));
	assert((hdr.pid == getpid()) && (hdr.type == SESSION_MSG));
	if(::read(migrate_handler::migrate_daemon->getfd(),
		&session, sizeof(session)) != sizeof(session))
	  ts_fatal("Read failed miserably %s", strerror(errno));
	else {
	  migrate_handler::MigrateSession *msession;
	  if((msession = migrate_handler::sessions[session.id])) {
	    ts_debug_1("Async update, trying again");
	    msession->update(&session, true);
	    goto waiting;
	  }
	  ts_debug_1("Received session id %d", session.id);
	  setId(session.id);
	  migrate_handler::sessions[session.id] = this;
	}
      }
    }

    // Serialization constructor
    MigrateSession(int flags, int id, int pid, address laddr, address paddr,
		   string pname, string dname, int pbufsize, int ufd)
      : _id(id), _pid(pid),  _state(MIGRATE_LOST), _laddr(laddr),
	_paddr(paddr), _pname(pname), _dname(dname), _flags(flags),
	_db(0), _pbufsize(pbufsize), _lfhandler(NULL), _ufd(ufd),
	_ufhandler(NULL)
    {
      ts_debug_1("Restored session %d:%d for fd %d", id, pid, ufd);
      migrate_handler::sessions[id] = this;
    }

    ~MigrateSession() {}

    // Accessors
    int id() const { return _id; }
    int pid() const { return _pid; }
    migrate_state state() const { return _state;}
    int flags() const { return _flags; }
    address laddr() const { return _laddr; }
    address paddr() const { return _paddr; }
    string pname() const { return _pname; }
    string dname() const { return _dname; }
    int db() const { return _db; }
    migrate_lookupfunc *lf() { return &_lf; }
    int pbufsize() const { return _pbufsize; }
    bool cont() const { return _cont; }
    int ufd() const { return _ufd; }

    // Initialization functions
    void init(address & a, migrate_state state);
    void setId(int id) { assert(_id == -1); _id = id; }
    void close();

    void addConn(migrate_handler *conn) { ts_debug_1("Adding connection %d", conn->magic()); conns[conn->magic()] = conn; }
    void removeConn(const migrate_handler *conn);
    void updateConn(int fd, const migrate_connection *conn);
    void moveConn(const migrate_connection *conn);
    void setlf(migrate_handler *handler, const migrate_lookupfunc *lf);
    void sethf(migrate_handler *handler, int ufd, const mig_handler uf);
    void setPaddr(address paddr) { _paddr = paddr; }
    void setPname(string pname) { _pname = pname; }
    void setDname(string dname) { _dname = dname; }
    void setDb(int db) { _db = db; }
    void setCont(bool cont) { _cont = cont; }
    void set_state(migrate_state state, bool notify);

    void cwc(migrate_continuation * cwc);
    void update(migrate_session *session, bool notify);

    // Old-school interface
    migrate_session *getOldSession();    
  };

  class MigrateConnection {

  private:

    address                _saddr;       // Requested source address
    address                _daddr;       // Requested destination address
    address                _csaddr;      // Current source address
    address                _cdaddr;      // Current destination address

    MigrateSession *       _session;
    flow_handler *         _downstream;

  public:

    MigrateConnection(address saddr, address daddr, address csaddr,
		      address cdaddr,  MigrateSession * session,
		      flow_handler * h) :
      _saddr(saddr), _daddr(daddr), _csaddr(csaddr), _cdaddr(cdaddr),
      _session(session), _downstream(h) {}

    // Accessors
    address saddr() const { return _saddr; }
    address daddr() const { return _daddr; }
    address csaddr() const { return _csaddr; }
    address cdaddr() const { return _cdaddr; }
    MigrateSession * session() const { return _session; }
    flow_handler * downstream() const { return _downstream; }

    void update_current(address newsaddr, address newdaddr)
    {
      _csaddr = newsaddr;
      _cdaddr = newdaddr;
    }
    void set_downstream(flow_handler *h) { _downstream = h; }
    void change_session(MigrateSession *session) { _session = session; }
  };

protected:

  int                    _magic;       // Internal identifier
  MigrateSession *       session;      // Our parent session

  virtual void updateConns() { assert(0); }

public:

  migrate_handler(init_context& ctxt) : flow_handler(ctxt), _magic((int)this),
					session(NULL) {

    int daemonfd = -1;
    
    // Connect to local daemon exactly once
    if(!migrate_daemon) {

      ts_debug_1("Migrate %s connecting to local daemon", VERSION);
      
      if ((daemonfd = ::socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	ts_fatal("socket %s", strerror(errno));
	return;
      }
      migrate_daemon = new Daemon(daemonfd);
      if(migrate_daemon->is_present()) {
	ts_debug_1("Contacted Migrate daemon");      
      } 
    }

    // Update the reference count regardless
    ++migrate_daemon_refcnt;

    ts_debug_1("Created migrate_handler, now %d", migrate_daemon_refcnt);
  }

  ~migrate_handler(); 

  virtual void changeSession(MigrateSession *newSession,
			     const migrate_connection *conn) { assert(0); }
  virtual void may_write(flow_handler *from, bool may);
  virtual bool may_exit();
  virtual bool save_state(oserial& out) const { assert(0); }
  virtual void migrateConn(int fd, MigrateSession *session) { assert(0); }
  virtual int ioctl(string target, int optname, string value, string& out);
  virtual void stateChange(migrate_state oldstate);

  virtual void freezeConn();

  int magic() const { return _magic; }

  // Old-school interface
  migrate_connection * const getOldConn(int rfd,
					MigrateConnection *mc) const;
};

// Serialization code
inline void freeze(oserial& o,
		   const migrate_handler::MigrateSession * const & session)
{
  ts_debug_1("Saving session %d:%d", session->id(), session->pid());
  freeze(o, session->id());          // Local session ID
  freeze(o, session->pid());         // ID at remote endpoint
  freeze(o, session->laddr());       // Local endpoint
  freeze(o, session->paddr());       // Remote endpoint

  freeze(o, session->pname());       // Remote hostname
  freeze(o, session->dname());       // Local hostname

  freeze(o, session->flags());       // Session flags    
  freeze(o, session->pbufsize());    // Default peer RCVBUF size
  
  freeze(o, session->ufd());         // TESLA-internal fd

  assert(session->cont());           // Should only be serializing w/conts
#if 0
    migrate_lookupfunc _lf;         // Function to find peer
    migrate_handler   *_lfhandler;  // Handler to call function in
    mig_handler        _uf;         // Function to handle updates
    migrate_handler   *_ufhandler;  // Handler to notify with session updates
#endif
}

inline void freeze(oserial& o,
		   const migrate_handler::MigrateConnection * const & conn)
{
  ts_debug_1("Saving connection from %s", conn->saddr().c_str());
  freeze(o, conn->saddr()); // Requested source address
  freeze(o, conn->daddr()); // Requested destination address
}


inline void unfreeze(iserial& i, migrate_handler::MigrateSession *& session)
{
  int flags, id, pid, db, pbufsize, ufd;
  address laddr, daddr;
  string pname, dname;

  unfreeze(i, id);                  // Local session ID
  unfreeze(i, pid);                 // ID at remote endpoint
  unfreeze(i, laddr);               // Local endpoint
  unfreeze(i, daddr);               // Remote endpoint

  unfreeze(i, pname);               // Remote hostname
  unfreeze(i, dname);               // Local hostname
    
  unfreeze(i, flags);               // Session flags
  unfreeze(i, pbufsize);            // Default peer RCVBUF size

  unfreeze(i, ufd);                 // TESLA-internal fd
  session = new migrate_handler::MigrateSession(flags, id, pid, laddr, daddr,
						pname, dname, pbufsize, ufd);
  session->setCont(true);
}

inline void unfreeze(iserial& i, migrate_handler::MigrateConnection *& conn)
{
  address saddr, daddr;
  unfreeze(i, saddr); // Requested source address
  unfreeze(i, daddr); // Requested destination address
  conn = new migrate_handler::MigrateConnection(saddr, daddr,
						address(), address(),
						NULL, NULL);
}

#endif


