/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: migrate_handler_tcp.hh,v 1.11 2002/10/10 22:49:46 snoeren Exp $
 *
 * Migrate API implementation.
 *
 */

#ifndef HH_MIGRATE_HANDLER_UDP
#define HH_MIGRATE_HANDLER_UDP

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif

#include "migrate_handler.hh"

#define DEFAULT_SLUSH 1024


// Serialization stuff
inline void freeze(oserial& o, const ring_buffer<char> * const & ring)
{
  freeze(o, ring->size());
  freeze(o, (unsigned int)(ring->head() - ring->buff()));
  freeze(o, (unsigned int)(ring->tail() - ring->buff()));
  freeze(o, ring->headtime());
  freeze(o, ring->tailtime());
  freeze(o, ring->ringtime());
  freeze(o, string(ring->buff(), ring->size()));
}

inline void unfreeze(iserial& i, ring_buffer<char> *& ring)
{
  unsigned int size, headtime, tailtime, ringtime, head, tail;
  string buff;
  
  unfreeze(i, size);
  unfreeze(i, head);
  unfreeze(i, tail);
  unfreeze(i, headtime);
  unfreeze(i, tailtime);
  unfreeze(i, ringtime);
  unfreeze(i, buff);
  assert(buff.length() == size);
  ring = new ring_buffer<char>(size, head, tail, headtime, tailtime, ringtime,
			       (char *)buff.c_str());
}

class migrate_handler_tcp : public migrate_handler {

private:

  MigrateConnection *    conn;         // Current connection
  sync_state             sync;         // Current state of the connection
  unsigned int           sndseq;       // Current sequence number 
  unsigned int           rcvseq;       // Current sequence number
  unsigned int           ackseq;       // Last successfully migrated byte 
  ring_buffer<char> *    ring;         // Ring buffer
  unsigned int           _slush;       // Largest internal buffer
  bool                   _connPending;  // Delayed connect flag

public:

  migrate_handler_tcp(init_context& ctxt) : migrate_handler(ctxt), conn(NULL),
					    sync(MIGRATE_SYNCED),
					    sndseq(0), rcvseq(0), ackseq(0),
					    ring(NULL), _slush(DEFAULT_SLUSH),
					    _connPending(false) {
    if(ctxt.serial()) {
      int connected = 0;
      *ctxt.serial() >> _magic;       // Our internal identifier
      // Check if we have more to do
      *ctxt.serial() >> connected;
      if(connected) {
	*ctxt.serial() >> session;      // Our parent session     
	*ctxt.serial() >> conn;         // Current connection
	*ctxt.serial() >> sndseq;       // Current sequence number 
	*ctxt.serial() >> rcvseq;       // Current sequence number
	*ctxt.serial() >> ackseq;       // Last successfully migrated byte 
	*ctxt.serial() >> ring;         // Ring buffer
	changeSession(session, NULL);
	session->addConn(this);
	migrate_handler::migrate_daemon->sendto(SESSION_MSG, (int)getpid(),
						(const char *)session->getOldSession(),
						sizeof(migrate_session));
      }
    }
  }

  ~migrate_handler_tcp();

  virtual int connect(address a);
  virtual int close();
  virtual void connected(flow_handler *from, bool success);
  virtual acceptret accept();

  virtual bool avail(flow_handler *from, data d);
  virtual int write(data d);
  virtual int shutdown(bool r, bool w);

  virtual address getsockname();
  virtual address getpeername();

  virtual void may_avail(bool may);

  virtual void migrateConn(int fd, MigrateSession *session);
  virtual void freezeConn();

  virtual bool save_state(oserial& out) const;

protected:

  virtual void changeSession(MigrateSession *newSession,
			     const migrate_connection *conn);
  virtual void updateConns();
  virtual void stateChange(migrate_state oldstate);

private:

  address                dest;        // Destination as specified in connect()

  // Internal functions

  void initSession(address a, migrate_state state);
  void initConn(int passive);
  void connSync();
  void initRing();

  DECLARE_HANDLER
};


#endif
