/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: migrate_handler_udp.cc,v 1.10 2002/10/02 22:10:32 snoeren Exp $
 *
 * Migrate API implementation.
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif

#include "migrate_handler_udp.hh"

void
migrate_handler_udp::updateConns() {

  migrate_handler::MigrateConnection * mc;

  // Inform daemon of connection status
  if (migrate_daemon->is_present()) {

    hash_map<const char *, MigrateConnection *,
      hash<const char *> >::iterator i;
    for(i = addr_conns.begin() ; i != addr_conns.end(); i++)
      migrate_daemon->sendto(CONNECTION_MSG, (int)getpid(),
			     (const char *)getOldConn(0, (*i).second),
			     sizeof(migrate_connection));
  }

}

migrate_handler::MigrateConnection *
migrate_handler_udp::initUDPSession(address a)
{
  migrate_handler::MigrateConnection * mc;

  // Create session and connection
  MigrateSession * session = new MigrateSession(M_AUTOCLOSE);
  session->init(a, MIGRATE_CONNECTING);
    
  address saddr = downstream[0]->getsockname();
  mc = new MigrateConnection(saddr, a, saddr, a, session, downstream[0]);
  addr_conns[a.c_str()] = mc;

  ts_debug_1("UDP connectino from %s", saddr.c_str());
  ts_debug_1("Adding connection %s to session %d", a.c_str(), session->id());
  session->addConn(this);
  
  // Notify remote end point of new connection
  if(migrate_daemon->is_present())
    migrate_daemon->sendto(CONNECTION_MSG, (int)getpid(),
			   (const char *)getOldConn(0, mc),
			   sizeof(migrate_connection));
  return mc;
}


int
migrate_handler_udp::connect(address a)
{
  int ret = downstream[0]->connect(a);

  // Initialize the session to potentially house this connection
  if (ret != 0) {
    ts_debug_1("- unable to connect: %s", strerror(errno));
  } else {
    if (ntohs(((const sockaddr_in *)a.addr())->sin_port) == 0) {
      ts_debug_1("Ignoring UDP connect to unbound port");
      _connected = OPEN;
    } else {
      _connected = TIGHT;
      initUDPSession(a);
    }
  }

  return ret;
}


void
migrate_handler_udp::connected(flow_handler *from, bool success)
{
  // Notify upstream
  get_upstream().connected(this, success);
}


bool
migrate_handler_udp::avail(flow_handler *from, data d)
{
  // Ensure we're supposed to be availing
  assert(may_avail_now());

  // If somebody else called us, we better figure out who
  if(from != downstream[0]) {
    MigrateConnection * mc = handler_conns[(int)from];
    assert(mc);
    
    // Rewrite from address
    ts_debug_2("Rewrote packet from %s", mc->daddr().c_str());
    return upstream->avail(this, data(d.bits(), d.length(), 0,
				      mc->daddr().addr(),
				      mc->daddr().addrlen()));
  }

  // Determine current destination address
  if (_connected != TIGHT) {

    address src = address((const void *)d.addr(), d.addrlen());
    migrate_handler::MigrateConnection * mc = addr_conns[src.c_str()];
    
    ts_debug_3("Received UDP from %s", src.c_str());
    if (mc == NULL) {
      ts_debug_1("Creating a UDP connection from %s", src.c_str());
      // Tighten up connection binding
      if (_connected == OPEN)
	_connected = TIGHT;
      initUDPSession(src);
    }
  }

  // Just pass it up, no rewrite necessary
  return upstream->avail(this, d);

}

int
migrate_handler_udp::write(data d)
{
  // Fastpath things that don't need to be rewritten
  if (!d.addrlen()) {
    if (_connected == TIGHT) {
      return downstream[0]->write(d);
    } else {
      // We're not connected
      return -1;
    }  
  }
  
  // We ignore multicast packets
  if(IN_MULTICAST(ntohl(((struct sockaddr_in *)d.addr())->sin_addr.s_addr))){
    ts_debug_3("Ignoring multicast packet");
    return downstream[0]->write(d);
  }

  // Otherwise, determine current destination address
  address dest = address((const void *)d.addr(), d.addrlen());
  ts_debug_3("Sending to %s", dest.c_str());

  migrate_handler::MigrateConnection * mc = addr_conns[dest.c_str()];
  if (mc == NULL) {
    ts_debug_1("Creating a UDP connection to %s", dest.c_str());
    mc = initUDPSession(dest);
  }

  // Use the address from the connection
  if (mc->downstream() != downstream[0]) {
    address da = mc->cdaddr();
    ts_debug_2("Rewriting to %s", da.c_str());
    return mc->downstream()->write(data(d.bits(), d.length()));
					
  } else {
    return downstream[0]->write(d);
  }
}

void
migrate_handler_udp::changeSession(MigrateSession *newSession,
				   const migrate_connection *conn)
{
  session = newSession;
  address a = address((const void *)&conn->saddr,
		      sizeof(struct sockaddr_in));
  addr_conns[a.c_str()]->change_session(newSession);
}


void
migrate_handler_udp::migrateConn(int fd, MigrateSession *session)
{
  if (_connected == TIGHT) {

    ts_debug_1("Migrating a connected UDP socket");

    // We never want to hear from this connection again
    downstream[0]->may_avail(false);
    downstream[0]->shutdown(true, false);

    // The socket I recieve will be identical to me
    downstream[0] = flow_handler::plumb(flow_handler::get_domain(),
					flow_handler::get_type(), fd);
  } else {

    // We better not be connected
    assert(_connected == NOT);

    ts_debug_1("Migrating NONconnection to %s", session->paddr().c_str());
    MigrateConnection * conn = addr_conns[session->paddr().c_str()];
    assert(conn);

    // If it was already on its own, close it off
    if (conn->downstream() != downstream[0]) {
      ts_debug_2("Closing up no-longer-used UDP socket");
      conn->downstream()->may_avail(false);
      conn->downstream()->shutdown(true, true);
    }

    // The socket I recieve will be identical to me
    conn->set_downstream(flow_handler::plumb(flow_handler::get_domain(),
					     flow_handler::get_type(), fd));

    // Update the address bindings
    conn->update_current(conn->downstream()->getsockname(),
			 conn->downstream()->getpeername());

    // Insert handler in hash so we can demux reads
    handler_conns[(int)conn->downstream()] = conn;
  }

}



DEFINE_HANDLER(migrateudp, migrate_handler_udp, AF_INET, SOCK_DGRAM);
HANDLER_USAGE(migrate_handler_udp,

"Migrate UDP handler v"
VERSION
"\n"
	      );
