//
// Migrate Session Layer
//
// Alex C. Snoeren <snoeren@lcs.mit.edu>
//
// Copyright (c) 2002 Massachusetts Institute of Technology.
//
// This software is being provided by the copyright holders under the GNU
// General Public License, either version 2 or, at your discretion, any later
// version. For more information, see the `COPYING' file in the source
// distribution.
//
// $Id: ring.hh,v 1.7 2002/10/02 22:10:32 snoeren Exp $
//
// Migrate API implementation.
//

#ifndef HH_RING_BUFFER
#define HH_RING_BUFFER

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <tesla/tesla.h>
#include <tesla/flow_handler.hh>

#ifndef MIN
# define MIN(a,b)	(((a)<(b))? (a):(b))
#endif

template<class T,class Allocator = std::allocator<T> >
class ring_buffer {

private:
  
  T *                  _buff;
  unsigned int         _size;
  T *                  _head;
  T *                  _tail;
  unsigned int         _headtime;
  unsigned int         _tailtime;
  unsigned int         _ringtime;

public:

  ring_buffer(unsigned int size = 0) : _buff(new T[size]), _size(size), 
    _head(_buff), _tail(_buff), _headtime(0), _tailtime(0), _ringtime(0) {}

  ring_buffer(unsigned int size, unsigned int head, unsigned int tail,
	      unsigned int headtime, unsigned int tailtime,
	      unsigned int ringtime, T * buff) :
    _buff(new T[size]), _size(size), _head(head + _buff), _tail(tail + _buff),
    _headtime(headtime), _tailtime(tailtime), _ringtime(ringtime) {
    for(int i = 0; i < size; ++i)
      _buff[i] = buff[i];
  }

  ~ring_buffer() { delete _buff; }

  T * head() const { return _head; }
  T * buff() const { return _buff; }
  T * tail() const { return _tail; }
  unsigned int headtime() const { return _headtime; }
  unsigned int tailtime() const { return _tailtime; }
  unsigned int ringtime() const { return _ringtime; }
  unsigned int size() const { return _size; }

  void resize(int size)
  {
    int copysize = MIN(size, _size);
    int headoff = _head - _buff;
    int tailoff = _tail - _buff;
    T * newbuff = new T[size];
    for(int i = 0; i < copysize; ++i)
      newbuff[i] = _buff[i];
    delete _buff;
    _buff = newbuff;
    _size = size;
    _head = newbuff + headoff;
    _tail = newbuff + tailoff;
  }

  unsigned int consecutive() const
  {
    if (_tail == _head) {
      if (_headtime < _tailtime)
	return _size;
      else
	return 0;
    } else {
      if (_tail > _head) {
	return (_tail - _head);
      } else {
	return (_size - (_head - _buff));
      }
    }
  }

  void supply(const T * data, unsigned int len)
  {
    unsigned int space;
    
    while (len) {
      space = (_buff + _size) - _tail;
      space = MIN(len, space);
      memcpy(_tail, data, space);
      _tail += space;
      assert(_tail <= (_buff + _size));
      if (_tail == (_buff + _size))
	_tail = _buff;
      data += space;
      len -= space;
    }
    _tailtime = ++_ringtime;
  }
  
  void consumed(unsigned int len)
  {
    _head += (len % _size);
    if (_head >= (_buff + _size))
      _head -= _size;
    _headtime = ++_ringtime;
  }

};

#endif


