/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 * 
 * $Id: migrate.h,v 1.50 2002/10/09 17:34:38 snoeren Exp $
 *
 * Standard Migrate include file for use by Migrate-aware apps.
 *
 */

#ifndef _MIGRATE_H
#define _MIGRATE_H

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>

#include <tesla/tesla.h>

#ifdef __cplusplus
extern "C" {
#endif

  struct fdhandle;

  typedef enum { MIGRATE_NOTCONNECTED,   /* Not yet contacted peer */
		 MIGRATE_CONNECTING,     /* Attempting to contact peer */
		 MIGRATE_ESTABLISHED,    /* In contact with peer */
		 MIGRATE_NOTSUPPORTED,   /* Peer doesn't support migrate */
		 MIGRATE_FROZEN,         /* Session is suspended */
		 MIGRATE_LOST,           /* Can't locate peer */
		 MIGRATE_PMIGRATING,     /* Changing peer IP address */
		 MIGRATE_LMIGRATING,     /* Changing local IP address */
		 MIGRATE_CLOSED          /* Going out of business */
  } migrate_state;

  typedef enum { MIGRATE_NOTSYNCED,
		 MIGRATE_HALFSYNCED,
		 MIGRATE_SYNCED } sync_state;

  struct migrate_session_t;
  typedef struct migrate_session_t migrate_session;
  struct migrate_connection_t;
  typedef struct migrate_connection_t migrate_connection;
  struct migrate_lookupfunc_t;
  typedef struct migrate_lookupfunc_t migrate_lookupfunc;

  struct migrate_lookupfunc_t {
    struct hostent *    (*func)(const char *);  /* Function to call */
    const char *        arg;                    /* Parameter to pass */
  };

  typedef void (*Continuation)(migrate_session *);

  typedef struct {
    int                 flags;
#define M_COMPLETE  0x01
    void               *fds;      /* IPC to preserve */
    void *              db;       /* Data to preserve */
    Continuation        cont;     /* Continuation function */
    /* This stuff is actually stored externally 
    char * const       *argv;
    char * const       *envp;
    char * const        cwd;
    */
  } migrate_continuation;

  typedef migrate_continuation * (*mig_handler)(migrate_session *, int flags);
#define M_INSTANT  0x01
#define M_LOCAL    0x02
#define M_REMOTE   0x04

#define M_MAXNAMESIZE 256

  struct migrate_session_t {
    int                 _fd;      /* TESLA-internal fd for ioctl */
    int                 id;       /* Local session ID */
    int                 pid;      /* ID at remote endpoint */
    migrate_state       state;
    struct sockaddr_in  laddr;    /* Local endpoint */
    struct sockaddr_in  paddr;    /* Remote endpoint */
    void *              db;       /* Attribute/value store */
    int                 flags;    /* Session flags */
#define                   M_AUTOCLOSE    0x01
#define                   M_ALWAYSLOOKUP 0x02
#define                   M_DONTMOVE     0x04
    char                dname[M_MAXNAMESIZE];    /* Local hostname */
    char                pname[M_MAXNAMESIZE];    /* Remote hostname */
    migrate_lookupfunc  newf;     /* Function to find peer */
    mig_handler         freeze;
    size_t              pbufsize; /* Default peer RCVBUF size */
  };

  struct migrate_connection_t {
    int                 fd;       /* File descriptor */
    int                 rfd;      /* Real file descriptor */
    migrate_session *   session;  /* Parent session */
    struct sockaddr_in  saddr;    /* Local endpoint */
    struct sockaddr_in  daddr;    /* Remote endpoint */
    struct sockaddr_in  csaddr;   /* Current local endpoint */
    struct sockaddr_in  cdaddr;   /* Current remote endpoint */
    int                 type;     /* SOCK_STREAM | SOCK_DGRAM */
    sync_state          sync;     /* Current state of the connection */
    unsigned int        sndseq;   /* Current sequence number */
    unsigned int        rcvseq;   /* Current sequence number */
    unsigned int        ackseq;   /* Last successfully migrated byte */
    struct _ring_t *    ring;     /* Ring buffer */
    unsigned int        refcnt;
  };

#define MIG_AVAIL          0
#define MIG_SESSION_CREATE 1
#define MIG_SESSION_CLOSE  2
#define MIG_ADD_CONN       3
#define MIG_REM_CONN       4
#define MIG_SET_FUNC       5
#define MIG_SET_NAME       6
#define MIG_GET_SESSION    7
#define MIG_NEW_HP         8
#define MIG_SET_HANDLER    9
#define MIG_MIGRATE       10
#define MIG_FREEZE        11
#define MIG_CONT          12
#define MIG_COMPCONT      13
#define MIG_SET_STORE     14

  /* Migrate Session API */

  int migrate_avail(void);

  migrate_session * migrate_session_create(int fd, int flags);
  migrate_session * migrate_get_session(int fd);

  int migrate_session_close(migrate_session *session);

  int migrate_add_connection(int fd, migrate_session *session);
  int migrate_remove_connection(int fd);

  int migrate_set_lookupfunc(migrate_session *session,
			     migrate_lookupfunc *lf);    
  int migrate_set_lookupname(migrate_session *session,
			     migrate_lookupfunc *lf);    
  int migrate_register_handler(migrate_session *session,
			       mig_handler handler);

  int migrate_migrate(migrate_session *session,
		      const struct sockaddr *addr, socklen_t addrlen);
  int migrate_freeze(migrate_session *session);

  int migrate_store(migrate_session *session, char *attr, void *value,
		    size_t len);
  int migrate_store_size(migrate_session *session, char *attr);
  void *migrate_retrieve(migrate_session *session, char *attr, void *dest);

  int migrate_return_cont(migrate_session *session,
			  migrate_continuation *cont,
			  char *const argv[], char *const envp[],
			  char *const cwd);
  
#ifdef __cplusplus
}
#endif

#define MIGRATE_SAVE(session, x) \
            migrate_store(session, #x, &x, sizeof(x))
#define MIGRATE_STRSAVE(session, x) \
            migrate_store(session, #x, x, (x ? (strlen(x) + 1) : 0))
#define MIGRATE_RESTORE(session, x) \
            migrate_retrieve(session, #x, &x)
#define MIGRATE_STRRESTORE(session, x) \
            (char *)migrate_retrieve(session, #x, \
                       malloc(migrate_store_size(session, #x)))

/* Environment variable used for migrated port number */
#define ENV_MIGRATE_PORT "MIGRATE_PORT"

/* Default port number */
#define MIGPORT 2000

#endif /* _MIGRATE_H */
