/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: mighandler.c,v 1.17 2002/10/10 17:25:16 snoeren Exp $
 *
 * Migrate API implementation.
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_ASSERT_H
# include <assert.h>
#endif

#include <migrate.h>
#include <migrate_internal.h>

extern char **environ;
extern int __migrate_store_init(migrate_session *session,
				const char *name);
extern int __migrate_store_close(migrate_session *session, int save);

void
__migrate_msg_handler(int fd, const char *handler, int msgname, void *msg,
		    int msglen)
{
  int flags = 0;

  if(!strncmp(handler, "migrate", 7)) {
#ifdef DEBUG
    ts_debug_2("Migrate message: fd %d, handler %s, msgname %d\n",
	       fd, handler, msgname);
#endif
    switch(msgname) {

    case MIG_NEW_HP: {
      migrate_lookupfunc *lf = (migrate_lookupfunc *)msg;
      struct hostent *hp = NULL;
      int len, res, retval = 0;
      
      if(msglen != sizeof(migrate_lookupfunc)) {
#ifdef DEBUG
	ts_fatal("Passed invalid lookupfunc parameter");
#endif
	assert(0);
      }

      if((hp = lf->func(lf->arg))) {
	if ((res = ts_ioctl(fd, "migrate", MIG_NEW_HP, hp->h_addr,
			    hp->h_length, &retval, (socklen_t *)&len))) {
#ifdef DEBUG
	    ts_fatal("Unable to deliver new endpoint: %s", strerror(-res));
#endif
	    assert(0);
	}
      } else {
#ifdef DEBUG
	ts_error("Supplied resolver failed");
#endif
      }
      break;
    }

    case MIG_CONT: {
      migrate_continuation * cont = (migrate_continuation *)msg;
      migrate_session * session =
	(migrate_session *)(msg + sizeof(migrate_continuation));
      if(cont->cont) {
	cont->cont(session);
      }
      break;
    }

    case MIG_MIGRATE:
      flags = M_INSTANT;
      /* Fallthrough... */

    case MIG_FREEZE: {
      migrate_session *session = (migrate_session *)msg;
      migrate_continuation *cont = NULL;

      if(msglen != sizeof(migrate_session)) {
#ifdef DEBUG
	ts_fatal("Passed invalid handler function");
#endif
	assert(0);
      }

      cont = session->freeze(session, (flags |
				       ((session->state == MIGRATE_LMIGRATING)?
					M_LOCAL : M_REMOTE)));

      /* Return continuation if one was generated */
      if(cont) {
	/* Check to see if we should immediately execute the continuation */
	if ((msgname != MIG_FREEZE) && !(cont->flags & M_COMPLETE)) {
	  if(cont->cont) {
	    cont->cont(session);
	  } else {
#ifdef DEBUG
	    ts_fatal("Invalid continuation function");
#endif
	    assert(0);
	  }
	} else
	  migrate_return_cont(session, cont, NULL, NULL, NULL);
      }

      break;
      }

    case MIG_SESSION_CLOSE:
      {
	migrate_session *session = (migrate_session *)msg;
	if(msglen != sizeof(migrate_session)) {
#ifdef DEBUG
	  ts_fatal("Passed invalid session message");
#endif
	  assert(0);
	}
	if(session->db)
	  __migrate_store_close(session, 0);
	break;
      }
    default:
#ifdef DEBUG
      ts_error("Unknown async message %d", msgname);
#endif
    }
  }
}



int
migrate_avail(void)
{
  static int checked = 0;
  static int retval = 0;
  int res = 0;
  int len = sizeof(retval);
  int fd;

  if(!checked) {
    checked = 1;
    fd = socket(AF_INET, SOCK_STREAM, 0);
    if((res = ts_ioctl(fd, "migrate", MIG_AVAIL, NULL, 0, &retval, &len)))
      retval = 0;
    close(fd);
  }
  return retval;
}


migrate_session *
migrate_session_create(int fd, int flags)
{
  migrate_session * session;
  int len = sizeof(migrate_session);
  int res = 0;
  if((session = (migrate_session *)malloc(sizeof(migrate_session)))) 
    if((res = ts_ioctl(fd, "migrate", MIG_SESSION_CREATE, &flags,
		       sizeof(int), session, (socklen_t *)&len))) {
      free(session);
      session = NULL;
      errno = -res;
    }
  session->_fd = fd;
  __migrate_store_init(session, NULL);
  return session;
}

migrate_session *
migrate_get_session(int fd)
{
  migrate_session * session;
  int len = sizeof(migrate_session);
  int res = 0;
  if((session = (migrate_session *)malloc(sizeof(migrate_session)))) 
    if((res = ts_ioctl(fd, "migrate", MIG_GET_SESSION, NULL, 0,
		       session, (socklen_t *)&len))) {
      free(session);
      session = NULL;
      errno = -res;
    }
  session->_fd = fd;
  if(!session->db)
    __migrate_store_init(session, NULL);
  return session;
}


int
migrate_session_close(migrate_session *session)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval); 

  if(!session) {
    errno = EINVAL;
    return -1;
  }
  if((res = ts_ioctl(session->_fd, "migrate", MIG_SESSION_CLOSE, NULL, 0,
		     &retval, &len))) { 
    errno = -res;
    return -1;
  }
  if(session->db)
    __migrate_store_close(session, 0);
  free(session);
  return retval;
}


int
migrate_add_connection(int fd, migrate_session *session)
{
  int res, len;

  if(!session) {
    errno = EINVAL;
    return -1;
  }
  if ((res = ts_ioctl(fd, "migrate", MIG_ADD_CONN, session,
		      sizeof(migrate_session), NULL, (socklen_t *)&len))) {
    errno = -res;
    return -1;
  }
  return 0;
}


int
migrate_remove_connection(int fd)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);
  if ((res = ts_ioctl(fd, "migrate", MIG_REM_CONN, NULL, 0,
		      &retval,(socklen_t *)&len))) {
    errno = -res;
    return -1;
  }
  return retval;
}


int
migrate_set_lookupfunc(migrate_session *session, migrate_lookupfunc *lf)
{
  int res, retval = 0;
  int len = sizeof(retval);

  if(!session) {
    errno = EINVAL;
    return -1;
  }

#ifdef DEBUG
  ts_debug_1("calling %u with %s", lf->func, lf->arg);
#endif
  if ((res = ts_ioctl(session->_fd, "migrate", MIG_SET_FUNC,
		      lf, sizeof(migrate_lookupfunc),
		      &retval,(socklen_t *)&len))) {
    errno = -res;
    return -1;
  }
  if(!retval) {
    memcpy(&session->newf, lf, sizeof(migrate_lookupfunc));
    ts_enable_messages(__migrate_msg_handler, SIGUSR2);
  }
  return retval;
}


int
migrate_set_lookupname(migrate_session *session, migrate_lookupfunc *lf)
{
  int res, retval = 0;
  int len = sizeof(retval);

  if(!session) {
    errno = EINVAL;
    return -1;
  }

  /* We can print hostnames nicely, so we differentiate */
  if(lf->func == gethostbyname) {
    lf->func = 0;
    if ((res = ts_ioctl(session->_fd, "migrate", MIG_SET_NAME, lf->arg,
			strlen(lf->arg),
			&retval,(socklen_t *)&len))) {
      errno = -res;
      return -1;
    }
  } else {
    /* Don't yet support other stuff */
    errno = ENOTSUP;
    retval = -1;
  }
  if(!retval) {
    memcpy(&session->dname, lf->arg, strlen(lf->arg));
    ts_enable_messages(__migrate_msg_handler, SIGUSR2);
  }
  return retval;
}


int
migrate_register_handler(migrate_session *session, mig_handler handler)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);
  char buf[sizeof(mig_handler) + sizeof(int)];

  if(!session) {
    errno = EINVAL;
    return -1;
  }

  memcpy(buf, &handler, sizeof(mig_handler));
  memcpy((buf + sizeof(mig_handler)), &session->_fd, sizeof(int));
  if ((res = ts_ioctl(session->_fd, "migrate", MIG_SET_HANDLER,
		      buf, (sizeof(mig_handler) + sizeof(int)),
		      &retval, &len))) {
    errno = -res;
    return -1;
  }
  if(!retval) {
    session->freeze = handler;
    ts_enable_messages(__migrate_msg_handler, SIGUSR2);
  } 
  return retval;
}


int
migrate_migrate(migrate_session *session,
		const struct sockaddr *addr, socklen_t addrlen)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);

  if(!session) {
    errno = EINVAL;
    return -1;
  }

  if ((res = ts_ioctl(session->_fd, "migrate", MIG_MIGRATE,
		      addr, addrlen, &retval, &len))) {
    errno = -res;
    return -1;
  }
  return retval;
}


int
migrate_freeze(migrate_session *session)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);

  if(!session) {
    errno = EINVAL;
    return -1;
  }

  if ((res = ts_ioctl(session->_fd, "migrate", MIG_FREEZE, NULL, 0,
		      &retval, &len))) {
    errno = -res;
    return -1;
  }
  return retval;
}


#define STWRITE(str) { size_t len = strlen((char *)str); \
                       fwrite(&len,sizeof(len),1,f); \
                       fwrite(str, len, 1, f); }

/* Chuck's argv discovery hack */

#if defined __hpux
#   define HAVE_STRINGS_FIRST
#endif

static char **__argv (void)
{
  extern long strlen (const char *);
  static char **val = (char **) 0;
  char        **p,
    *bnd_hi,
    *bnd_lo;
  
  if (!val) {
    for (p = environ; *p; p++)
      ;
#ifdef HAVE_STRINGS_FIRST   /* string data area just before pointer vectors */
    bnd_hi = environ[-2];
    bnd_lo = p[-1] + strlen (p[-1]) + 1;	/* end of strings */
    p = environ - 2;
    while ((char *)p >= bnd_lo && *p <= bnd_hi)
      p--;
#else                       /* string data area just after pointer vectors */
    bnd_hi = environ[-2];
    bnd_lo = (char *)(p + 1);		/* end of pointers */
    p = environ - 2;
    while (bnd_lo <= *p && *p <= bnd_hi)
      p--;
#endif
    p++;
    val = p;
  }
  return val;
}

static int
send_fd(int daemonfd, int sid, int fd)
{
  static mmsghdr hdr;

  hdr.type = FD_MSG;
  hdr.pid = sid;
  hdr.len = sizeof(fd);

  if(write(daemonfd, &hdr, sizeof(hdr)) != sizeof(hdr)) 
    assert(0);
    
  if(fd != -1) {

    struct cmsghdr     *cm;
    char                cbuf[CMSG_SPACE(sizeof(int))]; 
    struct msghdr       msg = {0};
    struct iovec        iov;
  
    iov.iov_base = (caddr_t) &fd; 
    iov.iov_len = sizeof(fd); 
    msg.msg_iov = &iov; 
    msg.msg_iovlen = 1; 
    
    /* Pass new file descriptor */
    msg.msg_control = cbuf;
    msg.msg_controllen = sizeof(cbuf); 
    cm = CMSG_FIRSTHDR(&msg); 
    cm->cmsg_len = CMSG_LEN(sizeof(int)); 
    cm->cmsg_level = SOL_SOCKET;
    cm->cmsg_type = SCM_RIGHTS; 
    
    *(int *)CMSG_DATA(cm) = fd;
    
    if (sendmsg(daemonfd, &msg, 0) < 0)
      assert(0);
  }

  return 0;
}

int
migrate_return_cont(migrate_session *session,
		    migrate_continuation *cont,
		    char *const argv[], char *const envp[],
		    char *const cwd)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);
  char mtemplate[] = "/tmp/migcont-XXXXXX";
  char template[] = "/tmp/tsstate-XXXXXX";
  char *teslafile = NULL;
  FILE *f = NULL;
  int fd, j;

  if(!session || !cont) {
    errno = EINVAL;
    return -1;
  }

  /* Pass continuation to Migrate daemon */
  if ((res = ts_ioctl(session->_fd, "migrate", MIG_CONT, cont,
		      sizeof(migrate_continuation), &retval, &len))) {
    errno = -res;
    return -1;
  }
  if(retval)
    return retval;

  /* If it's complete, need to pass more */
  if (cont->flags & M_COMPLETE) {

    /* Close off the database file */
    if(session->db)
      __migrate_store_close(session, 1);

#if 0
    /* First, close off all sessions but this one */
    struct _list_t * curr = migrate_sessions;
    while (curr) {
      migrate_session *cs = curr->data;
      if (!session_equals((void *)session->id, cs))
	migrate_session_close(cs);
      curr = curr->next;
    }
#endif

    /* Create a temporary file name for the TESLA state */
    if (!(teslafile = mktemp(template))) {
#ifdef DEBUG
      ts_error("Unable to mktemp!");
#endif
      assert(0);
    }


    /* Finally, send it an execve vector */   
    if (!(f = fdopen(mkstemp(mtemplate), "w"))) {
#ifdef DEBUG
      ts_error("Unable to open mcnt file %s", mtemplate);
#endif
      assert(0);
    }

    STWRITE(&template);

    /* Save TESLA file descriptor */
    fwrite(&(session->_fd), sizeof(session->_fd), 1, f);

    /* Set correct directory */
    if (!cwd) {
      char *ucwd = getcwd(NULL, 0);
      STWRITE(ucwd);
      free(ucwd);
    } else
      STWRITE(cwd);

    /* Recover environment and arguments if not passed */
    if(!argv)
      argv = __argv();
    if(!envp)
      envp = environ;

    for(j = 0; argv[j]; j++)
      STWRITE(argv[j]);
    len = 0;
    fwrite(&len,sizeof(len),1,f);


    for(j = 0; envp[j]; j++)
      STWRITE(envp[j]);
    len = 0;
    fwrite(&len,sizeof(len),1,f);

    fclose(f);

    /* Pass complete info to Migrate daemon, get daemon fd in return. */
    retval = -1;
    len = sizeof(retval);
    if ((res = ts_ioctl(session->_fd, "migrate", MIG_COMPCONT, mtemplate,
			strlen(mtemplate), &retval, &len))) {
      errno = -res;
      return -1;
    }
    if(retval == -1)
      return -1;

    fprintf(stderr, "GOT daemon fd %d\n", retval);


    /* Finally, do a global save of TESLA state */
    ts_save_state(teslafile);

    if(!cont->fds)
      fprintf(stderr, "NO fds, saving them all\n");
    else
      fprintf(stderr, "We think there are some to save\n");

    /* Now hand it all our fds */
    for (fd = 0; fd < FD_SETSIZE; fd++) {
      if((!cont->fds || FD_ISSET(fd, (fd_set *)cont->fds)) && !ts_fd_usage(fd) &&
	 (!getsockname(fd, NULL, 0) || (errno != EBADF)))
	if((fd != retval)) {
	  fprintf(stderr, "Saving fd %d\n", fd);
	  send_fd(retval, session->id, fd);
	  fprintf(stderr, "saved for session %d\n", session->id);
	}
    }
    fprintf(stderr, "we're done\n");
    close(retval);

    exit(0);
  }

  return 0;
}

