/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2002 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: store.c,v 1.7 2002/10/10 17:25:16 snoeren Exp $
 *
 * Migrate attribute/value store implementation.
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_ASSERT_H
# include <assert.h>
#endif
#ifdef HAVE_DB1_DB_H
# include <db1/db.h>
#else
# ifdef HAVE_DB_H
#  include <db.h>
# endif
#endif

#include <migrate.h>

#define DBFILENAME "/tmp/migdb-%d-%d"

extern void __migrate_msg_handler(int fd, const char *handler, int msgname,
				  void *msg, int msglen);

int
__migrate_store_close(migrate_session *session, int save)
{
  char dbname[255];
  
  if(session && session->db) {
    snprintf(dbname, 255, DBFILENAME, MIGPORT, session->id);
    if(((DB *)session->db)->close((DB *)session->db)) {
#ifdef DEBUG
      ts_error("Can't close database");
#endif
    }
    if(!save)
      unlink(dbname);
  }
  session->db = NULL;
  return 0;
}


int
__migrate_store_init(migrate_session *session, const char *name)
{
  int res, retval = 0;
  socklen_t len = sizeof(retval);
  char dbname[255];

  char *envport = getenv(ENV_MIGRATE_PORT);
  int   mport = MIGPORT;
  
  // Extract migrate port from the environment
  if (envport) {
    char *end;
    long newport = strtol(envport, &end, 0);
    if (*end || newport < 1 || newport > 65535) {
#ifdef DEBUG
      ts_error("Invalid port \"%s\" in MIGRATE_PORT\n", envport);
#endif
    } else {
      mport = newport;
    }
  }

  snprintf(dbname, 255, DBFILENAME, mport, session->id);
#ifdef DEBUG
  ts_debug_1("Initializing dbfile %s", name ? name : dbname);
#endif
  if(!(session->db = dbopen(name ? name : dbname,
			    O_CREAT|O_RDWR, S_IRWXU, DB_HASH, NULL))) {
#ifdef DEBUG
    ts_fatal("Can't init session store: %s", strerror(errno));
#endif
    assert(0);
  }

  /* Notify daemon we have a store */
  if ((res = ts_ioctl(session->_fd, "migrate", MIG_SET_STORE, &session->db,
		      sizeof(session->db), &retval, &len))) {
    __migrate_store_close(session, 0);
    assert(0);
  }

  /* Make sure we hear about session closings so we can clean up */
  ts_enable_messages(__migrate_msg_handler, SIGUSR2);
  return 0;
}



int
migrate_store(migrate_session *session, char *attr, void *value,
	      size_t len)
{
  DBT key = {attr, strlen(attr) + 1};
  DBT data = {value, len};

  if(!session) {
    errno = EINVAL;
    return -1;
  }

  if(session->db == NULL) {
    __migrate_store_init(session, NULL);
  }

  if(!len)
    return 0;

#ifdef DEBUG
  ts_debug_2("Looking for %s", attr);
#endif
  if(((DB *)session->db)->put((DB *)(session->db), &key, &data, 0)) {
#ifdef DEBUG
    ts_fatal("DB put");
#endif
    assert(0);
  }

  return 0;
}


int
migrate_store_size(migrate_session *session, char *attr)
{
  DBT key = {attr, strlen(attr)+1};
  DBT data;

  if(!session || !(session->db)) {
#ifdef DEBUG
    ts_debug_1("Database not initialized");
#endif
    return 0;
  }
  
  switch(((DB*)session->db)->get(session->db, &key, &data, 0)) {
  case -1:
#ifdef DEBUG
    ts_fatal("DB get failed");
#endif
    assert(0);
    break;
  case 1:
    return 0;
  default:
  }
  return data.size;
}


void *
migrate_retrieve(migrate_session *session, char *attr, void *dest)
{
  DBT key = {attr, strlen(attr)+1};
  DBT data;

  if(!session || !(session->db)) {
#ifdef DEBUG
    ts_debug_1("Database not initialized");
#endif
    errno = EINVAL;
    return NULL;
  }

  switch(((DB*)session->db)->get(session->db, &key, &data, 0)) {
  case -1:
#ifdef DEBUG
    ts_fatal("DB get failed");
#endif
    assert(0);
    break;
  case 1:
#ifdef DEBUG
    ts_debug_1("%s not stored!", attr);
#endif
    break;
  default:
    memcpy(dest, data.data, data.size);
  }
  return dest;
}



