/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: force.c,v 1.3 2002/09/03 18:24:18 snoeren Exp $
 */


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#  include <sys/ioctl.h>
#endif
#ifdef HAVE_SYS_UN_H
# include <sys/un.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#ifdef HAVE_NET_IF_H
#  include <net/if.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif

#include "migrated.h"

int migrated_port = MIGPORT;

static char *mpipefmt = "/tmp/.migratemonitor-%d";
static char mpipename[256];

#ifdef HAVE_GETOPT_LONG
static struct option long_options[] = {
  {"help", 0, 0, 'h'},
  {"port", 1, 0, 'p'},
  {"update", 1, 0, 'u'},
  {"version", 0, 0, 'v'},
  {0, 0, 0, 0}
};
#endif


void
usage(FILE *fd)
{
  fprintf(fd, "Usage: force [options]\n"\
	  "Options:\n" \
	  "-h"
#ifdef HAVE_GETOPT_LONG
	  ", --help\t"
#endif
	  "\tPrint this message and exit.\n");
  fprintf(fd, "-p"
#ifdef HAVE_GETOPT_LONG
	  ", --port\t"
#endif
	  "\tListen on the specified port (default: %d)\n", MIGPORT);
  fprintf(fd, "-v"
#ifdef HAVE_GETOPT_LONG
	  ", --version\t"
#endif
	  "\tPrint the version number and exit.\n" \
	  "-u if"
#ifdef HAVE_GETOPT_LONG
	  ", --update if"
#endif
	  "\tUpdate status of if.\n");
}

void
version(FILE *fd, char *ignored)
{
  fprintf(fd, "MigForce version %s, by Alex C. Snoeren.\n", VERSION);
  fprintf(fd, "Copyright (C) 2002 Massachusetts Institute of Technology.\n");
}


/* uint16_t really should be in_port_t, but I'm too lazy to write an autoconf
   test. */
typedef struct {
  struct in_addr laddr;
  struct in_addr raddr;
  uint16_t lport;
  uint16_t rport;
  short conn_up;
  short if_up;
} monitorstatemsg;

typedef struct {
  char label[16];
  struct in_addr addr;
  struct in_addr mask;
  short if_up;
} monitorifstatemsg;


short
getaddr(char *intstr, struct in_addr *addr, struct in_addr *mask)
{
    struct ifconf ifc;
    struct ifreq *ifr;
    int numreqs = 10;
    int skfd;
#ifdef GNU_LINUX
    int n;
#endif

    if ((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
	perror("socket");
	exit(1);
    }

    ifc.ifc_buf = NULL;
    numreqs = 0;

    ifc.ifc_len = sizeof(struct ifreq) * numreqs;

    while (1)
      {
	if (ifc.ifc_buf)
	  {
	    if (ioctl(skfd, SIOCGIFCONF, &ifc) < 0)
	      {
		perror("SIOCGIFCONF");
		exit(1);
	      }
	    
	    if (ifc.ifc_len < sizeof(struct ifreq) * numreqs)
		    break;
	  }
	
#ifdef GNU_LINUX
	numreqs += 10;
#else
	/* Seems we only get one shot under BSD, so let's make it */
	/* really big. */
	numreqs += 100;
#endif
	ifc.ifc_len = sizeof(struct ifreq) * numreqs;
	ifc.ifc_buf = realloc(ifc.ifc_buf, ifc.ifc_len);
      }
    
    ifr = ifc.ifc_req;

#ifdef GNU_LINUX
    for (n = 0; n < ifc.ifc_len; n += sizeof(struct ifreq), ++ifr)
#else
#define max(a,b) ((a)>(b) ? (a) : (b))
#define size(p)	max((p).sa_len, sizeof(p))
      for (; (char*)(ifr) < ((char*)ifc.ifc_buf) + ifc.ifc_len;
	   ifr = (void*)( ((char*)ifr) + sizeof(ifr->ifr_name) + size(ifr->ifr_addr)) )
#endif
	{
	  struct sockaddr_in *sin = (struct sockaddr_in *)(&ifr->ifr_addr);
	  struct ifreq ifr2, ifr3;

	  if(strcmp(ifr->ifr_name, intstr) || !sin->sin_addr.s_addr ||
	     ifr->ifr_addr.sa_family != AF_INET)
	    continue;

	  strcpy(ifr2.ifr_name, ifr->ifr_name);
	  strcpy(ifr3.ifr_name, ifr->ifr_name);
	  
	  if (ioctl(skfd, SIOCGIFFLAGS, &ifr2) < 0 ||
	      ioctl(skfd, SIOCGIFNETMASK, &ifr3) < 0) {
	    perror(strerror(errno));
	    exit(1);
	  }
	  	  
#ifdef GNU_LINUX
# define NETMASK_FIELD ifr_netmask
#else
# define NETMASK_FIELD ifr_addr
#endif
	  
	  printf("notifying interface %s %s ",
		 ifr->ifr_name,
		 inet_ntoa(sin->sin_addr));
	  printf("%s %s\n",
		 inet_ntoa(((struct sockaddr_in *)(&ifr3.NETMASK_FIELD))->sin_addr),
		 (ifr2.ifr_flags & IFF_UP) ? "up" : "down");

	  memcpy(addr, &sin->sin_addr, sizeof(struct in_addr));
	  memcpy(mask,
		 &((struct sockaddr_in *)(&ifr3.NETMASK_FIELD))->sin_addr,
		 sizeof(struct in_addr));
	  return (ifr2.ifr_flags & IFF_UP);
	}
    return -1;
}


int
main(int argc, char *argv[])
{
  int c;
  int msock;
  struct sockaddr_un uaddr;
  char *portstr = getenv("MIGRATE_PORT");
  char *intstr = NULL;
  monitorifstatemsg smsg;
  mmsghdr msg;

#ifdef HAVE_GETOPT_LONG
  int option_index = 0;
  while((c = getopt_long(argc, argv, "hp:u:v", long_options,
			 &option_index)) != -1) {
#else
  while((c = getopt(argc, argv, "hp:u:v")) != -1) {
#endif
    switch (c) {
    case 'h':
      usage(stdout);
      exit(0);
      break;
    case 'p':
      portstr = optarg;
      break;
    case 'u':
      intstr = optarg;
      break;
    case 'v':
      version(stdout, NULL);
      exit(0);
      break;
    default:
      usage(stderr);
      exit(1);
    }
  }

  if (argc - optind > 0) {
    usage(stderr);
    exit(1);
  }

  if (portstr) {
    char *end;
    long newport = strtol(portstr, &end, 0);
    if (*end || newport < 1 || newport > 65535) {
      fprintf(stderr,
	      "Invalid port specified \"%s\"\n", portstr);
    } else {
      migrated_port = newport;
    }
  }

  if(intstr) {

    if((smsg.if_up = getaddr(intstr, &smsg.addr, &smsg.mask)) != -1) {

      /* Open monitor named pipe */
      memset(&uaddr, 0, sizeof(uaddr));
      uaddr.sun_family = AF_UNIX;
      sprintf(mpipename, mpipefmt, migrated_port);
      strcpy(uaddr.sun_path, mpipename);
      
      if ((msock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	fprintf(stderr, "socket: %s\n", strerror(errno));
	goto shutdown;
      }
      if (connect(msock, (struct sockaddr *)&uaddr, sizeof(uaddr)) < 0) {
	fprintf(stderr, "Unable to open pipe: %s\n",
		strerror(errno));
	goto shutdown;
      }

      msg.type = MONITOR_IF_STATE_CHANGE;
      msg.len = sizeof(smsg);
      msg.pid = getpid();
      strncpy(smsg.label, intstr, 16);
      
      write(msock,&msg,sizeof(msg));
      write(msock,&smsg,sizeof(smsg));
    } else
      fprintf(stderr, "%s not found.\n", intstr);
    
  } else
    usage(stdout);

  shutdown:
  exit(0);
}
