#!/usr/bin/perl

#####
#
# Migrate Session Layer: Monitoring Daemon
#
# Jon Salz <jsalz@mit.edu>
# Alex C. Snoeren <snoeren@lcs.mit.edu>
#
# Copyright (c) 2001 Massachusetts Institute of Technology.
#
# This software is being provided by the copyright holders under the GNU
# General Public License, either version 2 or, at your discretion, any later
# version. For more information, see the `COPYING' file in the source
# distribution.
#
# $Id: migmonitord.pm,v 1.9 2001/12/06 19:56:18 jsalz Exp $
#
#####

use strict;

use IO::File;
use IO::Handle;
use IO::Select;
use IO::Socket;
#use IO::Socket::UNIX;
use IPC::Open2;
use Getopt::Long;
use Socket;
use File::Basename;

# Wait between /proc/net/tcp checks.
my $IDLETIME = 0.25;
# Fire off tcping when rexmits is REXMIT_THRESH or higher.
my $REXMIT_THRESH = 2;
# Window between tcpings.
my $TCPING_PERIOD = 5;

my $MIGPORT = $ENV{"MIGRATE_PORT"} || 2000;
my $HELP = 0;

sub usage {
    print STDERR "Usage: migmonitord.pm [ -p <migrate-port> ]\n";
    exit 1;
}

GetOptions('p=i' => \$MIGPORT,
	   'h' => \$HELP) || usage();
usage() if $HELP;
usage() if (@ARGV);


my $iflist = dirname($0)."/iflist";
$iflist = "iflist" if (!-e $iflist);

my $start = time();
sub logmsg
{ 
    print STDERR "+",(time() - $start),"s: ",@_,"\n";
}

my %interfaces = ();
my %addresses = ();

my $fhnum = 1;
sub newopen
{
    my $path = shift;
    my $pid = eval("open(FH$fhnum, \$path)");
    if ($pid == 0)
    {
	return ();
    }

    my $fh = eval("*FH$fhnum");
    ++$fhnum;
    return ($fh, $pid);
}

sub is_addr_up
{
    my $localaddr = shift;
    foreach my $interface (keys %interfaces)
    {
	next unless defined($addresses{$interface});
	return 1 if (defined($addresses{$interface}) &&
		     defined($addresses{$interface}->{$localaddr}));
    }
    return 0;
}

sub getline
{
    my $fh = shift;
    my $line = "";
    my $char = "";
    while (1)
    {
	my $bytes = $fh->sysread($char, 1);
	return $line if ($char eq "" || $char eq "\n" || $char eq "\r");
	$line .= $char;
    }
}

# conn -> [last_tcping_time, is_up?, is_if_up?]
my %watches = ();

# Open input pipe.
#
# Earlier versions of Perl seem to have trouble with
#
#   my $ctl = new IO::Socket::UNIX("/tmp/.migratemonitor-$MIGPORT");
#
# so we'll do the socket ops on our own.

socket(CTL, PF_UNIX, SOCK_STREAM, 0) or die "Unable to create socket: $!";
connect(CTL, sockaddr_un("/tmp/.migratemonitor-$MIGPORT")) or die "Unable to connect to migrated: $!";
my $ctl = new IO::Handle;
$ctl->fdopen(\*CTL, "w+");

my %tcpings = ();     # conn->[PID,IO::File]
my %tcpings_fd = ();  # fd->conn

sub write_state
{
    my $addr = shift;
    my $status = $watches{$addr};
    return unless (defined($status));

    logmsg "Telling migrated: ",hex_to_conn($addr)," ",($status->[1]?"UP":"DOWN")," ",($status->[2]?"IFUP":"IFDOWN");
    my ($laddr, $lport, $raddr, $rport) = hex_to_parts($addr);
    my $out = pack("IIIa4a4nnSS",
		   7,  # MONITOR_STATE_CHANGE
		   $$, # PID
		   16, # Length in bytes after header
		   inet_aton($laddr),
		   inet_aton($raddr),
		   $lport,
		   $rport,
		   $status->[1],
		   $status->[2]);
    print $ctl $out;
}

sub write_if_change
{
    my ($label, $state, $addr, $mask) = @_;

    my $out = pack("IIIa16a4a4S",
		   9,  # MONITOR_IF_STATE_CHANGE
		   $$, # PID
		   26, # Length in bytes after header
		   defined($label) ? substr($label, 0, 15) : "",
		   defined($addr) ? inet_aton($addr) : "\0\0\0\0",
		   defined($mask) ? inet_aton($mask) : "\0\0\0\0",
		   defined($state) ? $state : 0);

    print $ctl $out;
}

sub scan_available_if
{
    while (my ($conn, $status) = each %watches)
    {
	my ($laddr) = hex_to_parts($conn);

	my $was_up = $status->[2];
	my $is_now_up = 0;

	while (my ($iface) = each %interfaces)
	{
	    if (defined($addresses{$iface}) && defined($addresses{$iface}->{$laddr}))
	    {
		$is_now_up = 1;
		last;
	    }
	}

	if ($was_up != $is_now_up)
	{
	    $status->[2] = $is_now_up;
	    write_state($conn);
	}
    }
}

# Open netlink daemon.
logmsg "Opening iflist daemon";
my $iflistpid = open2(\*IFLISTR, \*IFLISTW, $iflist, "-r") or die "Opening iflist: $!";
select $ctl;
$| = 1;
select IFLISTW;
$| = 1;
select STDOUT;

# Clear interface list
write_if_change("*", 0);

sub watch
{
    my $key = shift;
    if (defined($watches{$key}))
    {
	logmsg "Warning: already watching $key";
    }
    else
    {
	$watches{$key} = [undef, 1, 1];
	logmsg "Watching ",hex_to_conn($key);
    }
}

sub unwatch
{
    my $key = shift;
    if (!defined($watches{$key}))
    {
	logmsg "Warning: not currently watching $key";
    }
    else
    {
	delete $watches{$key};
	logmsg "Unwatched ",hex_to_conn($key);
    }
}

sub conn_to_hex
{
    my $conn = shift;
    my ($laddr, $lport, $raddr, $rport) = $conn =~ /^([^:]+):(\d+)->([^:]+):(\d+)$/;
    return undef unless defined($laddr);
    return sprintf("%08X:%04X %08X:%04X",
		   unpack("L", inet_aton($laddr)),
		   $lport,
		   unpack("L", inet_aton($raddr)),
		   $rport);
}
sub hex_to_conn
{
    my $hex = shift;
    my ($laddr, $lport, $raddr, $rport) = $hex =~ /^([0-9A-F]+):([0-9A-F]+) ([0-9A-F]+):([0-9A-F]+)$/i;
    return undef unless defined($laddr);
    return inet_ntoa(pack("L", hex $laddr)) . ":" . hex($lport) . "->" .
	inet_ntoa(pack("L", hex $raddr)) . ":" . hex($rport);
}
sub hex_to_parts
{
    my $hex = shift;
    my ($laddr, $lport, $raddr, $rport) = $hex =~ /^([0-9A-F]+):([0-9A-F]+) ([0-9A-F]+):([0-9A-F]+)$/i;
    return undef unless defined($laddr);
    return (inet_ntoa(pack("L", hex $laddr)), hex($lport),
	    inet_ntoa(pack("L", hex $raddr)), hex($rport));
}

sub do_tcping
{
    my $addr = shift;
    my ($laddr, $lport, $raddr, $rport) = hex_to_parts($addr);

    $SIG{"ALRM"} = sub { print 0; exit; };
    my $cmd = "tcping -c 1 -I $laddr -p $rport $raddr";

    alarm 1;
    open(TCPING, "$cmd|") or do {
	logmsg "Unable to open tcping";
	print 1;
	exit;
    };
    1 while (<TCPING>);
    close TCPING;
    alarm 0;

    print 1;
    exit;
}

# Open multiplexor.
my $sel = new IO::Select($ctl);
my $lastifs = "."; # Force interface up/down scan
my %lastupifs = ();

# Wait for input.
MAIN: while (1)
{
    my @fhs = IO::Select::select($sel, undef, undef, $IDLETIME);
    my ($rhs, $whs, $ehs) = @fhs;
    my $ifs = "";

    # Get ifs from iflist.
    print IFLISTW "\n";
    my %upaddrs = ();
    my %upifs = ();
    my $ifchange = 0;
    while (1)
    {
	my $line = <IFLISTR>;
	if (!defined($line))
	{
	    logmsg "Iflist daemon closed. Terminating.";
	    last MAIN;
	}
	chomp $line;
	last if ($line !~ /\S/);
	
	if ($line =~ /^interface (\S+) (\S+) (\S+) (\S+)$/)
	{
	    my ($label, $addr, $mask, $state) = ($1, $2, $3, $4);
	    if ($state eq "up")
	    {
		$ifs .= "$addr ";
		++$upaddrs{$addr};
		++$upifs{"$label $addr $mask"};
	    }

	    if ($state eq "up" && !defined($lastupifs{"$label $addr $mask"}))
	    {
		logmsg "Interface gone up: $label $addr $mask";
		write_if_change($label, 1, $addr, $mask);
		++$ifchange;
	    }
	}
	else
	{
	    logmsg "Weird line from iflist: \"$line\"";
	}
    }
    foreach my $oldif (keys %lastupifs)
    {
	if (!defined($upifs{$oldif}))
	{
	    logmsg "Interface gone down: $oldif";
	    my ($if, $addr, $mask) = split(/ /, $oldif);
	    write_if_change($if, 0, $addr, $mask);
	    ++$ifchange;
	}
    }
    if ($ifchange)
    {
	write_if_change();
    }

    %lastupifs = %upifs;

    foreach my $rh (@$rhs)
    {
	if ($rh == $ctl)
	{
	    my $line = getline($rh);
	    if (length($line) == 0)
	    {
		# STDIN closed.
		$sel->remove($rh);
		logmsg "Control connection closed. Terminating.";
		sleep 5;
		last MAIN;
	    }
	    chomp $line;

	    logmsg "Data from migrated: \"$line\"";

	    if ($line =~ /^(un)?watch (\S+)/)
	    {
		my $un = $1;
		my $key = conn_to_hex($2);
		if (!defined($key))
		{
		    logmsg "Invalid watch request for connection $key";
		}
		else
		{
		    $1 eq "un" ? unwatch($key) : watch($key);
		}
	    }
	}
	else
	{
	    # It's a TCPing.
	    my $line = getline($rh);
	    chomp $line;
	    my $result = $line;

	    # Drain the filehandle.
	    1 while <$rh>;

	    my $addr = $tcpings_fd{$rh};
	    my $pid = $tcpings{$addr}->[0];
	    delete $tcpings_fd{$rh};
	    delete $tcpings{$addr};
	    $sel->remove($rh);
	    close $rh;

	    # Reap the process.
	    waitpid($pid, 0);

	    if (defined($watches{$addr}))
	    {
		my $is_now_up = $result;
		my $was_up = $watches{$addr}->[1];

		if ($is_now_up != $was_up)
		{
		    $watches{$addr}->[1] = $is_now_up;
		    write_state($addr);
		}
	    }
	}
    }

    my %conns_to_delete = %watches;

    # Loop through /proc/net/tcp, and see if anything interesting is
    # going on.
    open(TCP, "/proc/net/tcp");
    <TCP>; # Skip headers
    while(<TCP>)
    {
	my $line = $_;
	my $addr = substr($line, 6, 27);
	my $state = substr($line, 35, 2);
	next unless ($state < 4 || $state > 6);
	delete $conns_to_delete{$addr};
	next unless $watches{$addr};
	my $rexmits = hex(substr($line, 67, 8));
	if ($rexmits == 0 && !defined($tcpings{$addr}))
	{
	    # Transmission succeeded.
	    if ($watches{$addr}->[1] == 0)
	    {
		$watches{$addr}->[1] = 1;
		write_state($addr);
	    }
	}
	next unless $rexmits >= $REXMIT_THRESH;
	next if $tcpings{$addr};
	my $last_tcping = $watches{$addr}->[0];
	next if (defined($last_tcping) && time() - $last_tcping < $TCPING_PERIOD);
	next if ($watches{$addr}->[2] == 0);

	# Brand new retransmission.
	logmsg hex_to_conn($addr)," exceeding rexmit threshold ($rexmits). tcpinging.";
	$watches{$addr}->[0] = time();

	my ($fh, $pid) = newopen("-|");
	if (!$pid)
	{
	    # Child.
	    do_tcping($addr);
	    exit 0;
	}
	$tcpings{$addr} = [$pid, $fh];
	$sel->add($fh);
	$tcpings_fd{$fh} = $addr;
    }

    while (my ($conn_to_delete) = each %conns_to_delete)
    {
	logmsg "Watch ",hex_to_conn($conn_to_delete)," has gone away.";
	delete $watches{$conn_to_delete};
    }

    # Have any addresses gone up or down?
    if ($ifs ne $lastifs)
    {
	logmsg "Addresses currently up: $ifs";

	# Yes; loop through connections and update interface up/down status.
	while (my ($conn, $status) = each %watches)
	{
	    my ($laddr) = hex_to_parts($conn);

	    my $was_up = $status->[2];
	    my $is_now_up = $upaddrs{$laddr} ? 1 : 0;

	    if ($was_up != $is_now_up)
	    {
		$status->[2] = $is_now_up;
		write_state($conn);
	    }
	}	

	$lastifs = $ifs;
    }
}

$ctl->close();
close IFLIST;

