/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: cont.c,v 1.6 2002/10/10 20:47:41 snoeren Exp $
 */


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#include <list.h>

#include "migrated.h"

extern char **environ;

#define STREAD(str) fread(&len,sizeof(len),1,f); str = malloc(len +1); \
                       fread(str, 1, len, f); str[len] = '\0';

pid_t
complete_cont(session_handle *handle)
{
  char envcstr[] = "MIGRATE_CONT=XXXXXXXXXX";
  char envsstr[] = "MIGRATE_SESS=XXXXXXXXXX";
  pid_t pid;
  sigset_t sigs;
  FILE * f;
  char *argv[255];
  char *envp[255];
  char *tsfile;
  char *cwd;
  char template[64];
  size_t len;
  int i, ufd;

  if((pid = fork())) {
    
    int wpid, status;

    /* We are the parent */

    if(pid < 0)
      return -1;

    /* Wait until child has detached fully, and record new pid */
    wpid = waitpid(pid, &status, 0);
    if((wpid != pid) || !WIFEXITED(status)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR, "fork failed somehow");
      return -1;
    }
    pid = WEXITSTATUS(status);

    /* Close off our copies of the fds */
    if(handle->cont->fds) {
      struct _list_t *curr = handle->cont->fds;
      while(curr) {
	if(close(((struct fdhandle_t *)curr->data)->fd))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR, "closing fd %d: %s",
		  ((struct fdhandle_t *)curr->data)->fd, strerror(errno));
	curr = curr->next;
      }
      _delete_list((struct _list_t **)&handle->cont->fds);
    }
    free(handle->cont->db);
    free(handle->cont);
    handle->cont = NULL;

    /* Update sessions to point to correct pid */
    return pid;
  }

  /* We're the new child. Double fork. */
  if((pid = fork())) {
    /* Return the child pid to our parent */
    exit(pid);
  }

  /* We can't get back our original session, so start a new one */
  setsid(); 

  /* Pass off our copies of the fds */
  if(handle->cont->fds) {
    struct _list_t *curr = handle->cont->fds;
    while(curr) {
      struct fdhandle_t *fdh = (struct fdhandle_t *)curr->data;
      if(fdh->fdnum != fdh->fd) {
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Restoring FD %d from %d",
		fdh->fdnum, fdh->fd);
	if(dup2(fdh->fd, fdh->fdnum) != fdh->fdnum)
	  log_log(LOG_MOD_MIGRATED, LOG_ERR, "dup2: %s",
		  strerror(errno));
	assert(fdh->fd != fdh->fdnum);
	/* Make sure we close original on exec */
	if (fcntl(fdh->fd, F_SETFD, FD_CLOEXEC)) {
	  log_log(LOG_MOD_MIGRATED, LOG_ERR,
		  "fcntl (FD_CLOEXEC): %s", strerror(errno));
	}
      }
      curr = curr->next;
    }
    _delete_list((struct _list_t **)&handle->cont->fds);
  }

  /* Open continuation file */
  if (!(f = fopen((char *)handle->cont->db, "r")))
    log_log(LOG_MOD_MIGRATED, LOG_CRIT, "Unable to open cont %s",
	    (char *)handle->cont->db);

  /* Read in Telsa state file name */
  STREAD(tsfile);
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "TS FILE %s", tsfile);
  sprintf(template, "TS_RESTORE_FILE=%s", tsfile);

  /* Restore TESLA file descriptor */
  fread(&ufd, sizeof(ufd), 1, f);
  
  /* Read in current directory */
  STREAD(cwd);

  /* Read in argv */
  i = 0;
  do {
    STREAD(argv[i]);
    i++;
  } while (len);
  free(argv[--i]);
  argv[i] = NULL;

  /* Read in environment */
  i = 0;
  do {
    STREAD(envp[i]);
    i++;
  } while (len);
  free(envp[--i]);
  
  fclose(f);
  unlink((char *)handle->cont->db);
  free(handle->cont->db);
  free(handle->cont);
  handle->cont = NULL;


  /* Update environment */
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Original pid was %d", handle->pid);
  sprintf(envcstr, "MIGRATE_CONT=%d", ufd);
  envp[i++] = envcstr;
  envp[i++] = envsstr;
  envp[i++] = template;
  envp[i] = NULL;

  /* Restore privledges */
  if(setuid(handle->uid))
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "setuid: %s", strerror(errno));
  if(chdir(cwd))
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "chdir(%s): %s", strerror(errno));
  
  /* Unblock any signals that we may be handling */
  sigemptyset(&sigs);
  if(sigprocmask(SIG_SETMASK, &sigs, NULL))
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "error unmasking signals: %s",
	    strerror(errno));

  /* Execute the program */
  execve(argv[0], argv, envp);

  /* Only reached on error */
  log_log(LOG_MOD_MIGRATED, LOG_CRIT, "Exec %s failed: %s",
	  argv[0], strerror(errno));
  exit(-1);
}
