/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: interactive.c,v 1.35 2002/10/03 22:50:50 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_CDEFS_H
# include <sys/cdefs.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#ifdef HAVE_PWD_H
# include <pwd.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#ifdef HAVE_READLINE_READLINE_H
# include <readline/readline.h>
#endif
#ifdef HAVE_READLINE_HISTORY_H
# include <readline/history.h>
#endif
#ifdef HAVE_TIME_H
# include <time.h>
#endif

#include "migrated.h"

#ifndef whitespace
#define whitespace(c) (((c) == ' ') || ((c) == '\t'))
#endif

const char *migstate2ascii[] = {
  "Not connected",
  "Connecting",
  "Established",
  "Not supported",
  "Frozen",
  "Lost",
  "Peer Migrating",
  "Local Migrating",
  "Closed",
};

typedef struct {
  const char *name;
  void (*func)(FILE *, char *);
  const char *desc;
} command;

void
usage(FILE *fd)
{
  fprintf(fd, "Usage: migrated [options]\n"\
	  "Options:\n" \
	  "-d"
#ifdef HAVE_GETOPT_LONG
	  ", --debug\t"
#endif
	  "\tSet the debugging level.\n"
	  "-f"
#ifdef HAVE_GETOPT_LONG
	  ", --file\t"
#endif
	  "\tUse the specified policy rules file.\n"
	  "-h"
#ifdef HAVE_GETOPT_LONG
	  ", --help\t"
#endif
	  "\tPrint this message and exit.\n");
#ifdef HAVE_LIBREADLINE
  fprintf(fd,
	  "-i"
#ifdef HAVE_GETOPT_LONG
	  ", --interactive"
#endif
	  "\tRun in interactive mode.\n");
#endif
  fprintf(fd,
	  "-p"
#ifdef HAVE_GETOPT_LONG
	  ", --port\t"
#endif
	  "\tListen on the specified port (default: %d)\n", MIGPORT);
  fprintf(fd,
	  "-v"
#ifdef HAVE_GETOPT_LONG
	  ", --version\t"
#endif
	  "\tPrint the version number and exit.\n");
}

void defrost(FILE *fd, char *);
void freeze(FILE *fd, char *);
void help(FILE *fd, char *);
void migrate(FILE *fd, char *);
void quit(FILE *fd, char *);
void resolve(FILE *fd, char *);
void sessions(FILE *fd, char *);
void session_close(FILE *fd, char *);

command commands[] = {
  { "close", session_close, "session\t\t\tClose a session." },
  { "resolve", resolve, "session\t\t\tResolve remote end-point and migrate." },
  { "resume", defrost, "session\t\t\tResume a session." },
  { "suspend", freeze, "session\t\t\tSuspend a session." },
  { "help", help, "\t\t\t\tDisplay a list of commands."},
  { "migrate", migrate, "session address\t\tMigrate a session to new address." },
  { "quit", quit, "\t\t\t\tExit migrated." },
  { "sessions", sessions, "\t\t\tShow list of current sessions." },
  { "version", version, "\t\t\tPrint the version number." },
  { NULL, NULL, NULL }
};


static session_handle *
session_id(int sid)
{
  session_ent *cur = session_list;

  while(cur) {
    if(cur->handle->session.id == sid)
      return cur->handle;
    else
      cur = cur->next;
  }
  
  return NULL;
}


void
session_close(FILE *fd, char *args)
{
  int             bptr, eptr, sid = 0;
  char           *endptr = NULL;
  session_handle *handle = NULL;

  for (bptr = 0; args[bptr] && whitespace(args[bptr]); bptr++);
  for (eptr = bptr; args[eptr] && !whitespace(args[eptr]); eptr++);

  if(args[eptr]) {
    fprintf(fd, "Usage: close session\n");
    return;
  } else
    args[eptr] = '\0';

  sid = strtol(&args[bptr], &endptr, 10);
  if(*endptr!='\0') {
    fprintf(fd, "Invalid session ID %s\n", &args[bptr]);
    return;
  }
  
  if(!(handle = session_id(sid))) {
    fprintf(fd, "Unknown session %d\n", sid);
    return;
  }

  remove_ent(close_session(handle, 1));
  fprintf(fd, "\nClosed session %d\n", sid);

}


void
freeze(FILE *fd, char *args)
{
  int             bptr, eptr, sid = 0;
  migrate_state   state;
  char           *endptr = NULL;
  session_handle *handle = NULL;

  for (bptr = 0; args[bptr] && whitespace(args[bptr]); bptr++);
  for (eptr = bptr; args[eptr] && !whitespace(args[eptr]); eptr++);

  if(args[eptr]) {
    fprintf(fd, "Usage: suspend session\n");
    return;
  } else
    args[eptr] = '\0';

  sid = strtol(&args[bptr], &endptr, 10);
  if(*endptr!='\0') {
    fprintf(fd, "Invalid session ID %s\n", &args[bptr]);
    return;
  }
  
  if(!(handle = session_id(sid))) {
    fprintf(fd, "Unknown session %d\n", sid);
    return;
  }

  state = freeze_session(handle);
  if(state != MIGRATE_FROZEN)
    fprintf(fd, "\nSession %d suspend failed.\n", sid);

}


void
defrost(FILE *fd, char *args)
{
  int             bptr, eptr, sid = 0;
  char           *endptr = NULL;
  migrate_state   state;
  session_handle *handle = NULL;

  for (bptr = 0; args[bptr] && whitespace(args[bptr]); bptr++);
  for (eptr = bptr; args[eptr] && !whitespace(args[eptr]); eptr++);

  if(args[eptr]) {
    fprintf(fd, "Usage: defrost session\n");
    return;
  } else
    args[eptr] = '\0';

  sid = strtol(&args[bptr], &endptr, 10);
  if(*endptr!='\0') {
    fprintf(fd, "Invalid session ID %s\n", &args[bptr]);
    return;
  }
  
  if(!(handle = session_id(sid))) {
    fprintf(fd, "Unknown session %d\n", sid);
    return;
  }

  state = defrost_session(handle);
  if(state == MIGRATE_FROZEN)
    fprintf(fd, "\nSession %d resmue failed\n", sid);

}


void
migrate(FILE *fd, char *args)
{
  struct hostent     *hent = NULL;
  struct sockaddr_in  addr;
  session_handle     *handle = NULL;
  int                 bptr, eptr, sid = 0;
  migrate_state       state;
  char               *endptr = NULL;
  struct timeval      tv;

  for (bptr = 0; args[bptr] && whitespace(args[bptr]); bptr++);
  for (eptr = bptr; args[eptr] && !whitespace(args[eptr]); eptr++);

  if(!args[eptr]) {
    fprintf(fd, "Usage: migrate session addr\n");
    return;
  } else
    args[eptr] = '\0';

  sid = strtol(&args[bptr], &endptr, 10);
  if(*endptr!='\0') {
    fprintf(fd, "Invalid session ID %s\n", &args[bptr]);
    return;
  }
  
  if(!(handle = session_id(sid))) {
    fprintf(fd, "Unknown session %d\n", sid);
    return;
  }

  if((hent = gethostbyname(&args[eptr+1])) == NULL) {
    fprintf(fd, "Invalid address %s\n", &args[eptr+1]);
    return;
  }

  memset(&addr, 0, sizeof(addr));
  /* We only handle AF_INET currently */
  if((addr.sin_family = hent->h_addrtype) != AF_INET) {
    fprintf(fd, "Invalid address family %d\n", hent->h_addrtype);
    return;
  }
  memcpy(&addr.sin_addr, hent->h_addr, hent->h_length);

  gettimeofday(&tv, NULL);
  fprintf(fd, "Started migrating at (.%lu) %s\n",tv.tv_usec,ctime(&tv.tv_sec));
  state = mig_session(handle, &addr);
  if((state != MIGRATE_CONNECTING) && (state != MIGRATE_ESTABLISHED))
    fprintf(fd, "\nSession %d migrate to %s failed\n", sid,
	    inet_ntoa(addr.sin_addr));	    
}

void
help(FILE *fd, char *subtopic)
{
  int i = 0;

  for (i = 0; subtopic[i] && whitespace(subtopic[i]); i++);
  
  if (!subtopic[0]) {
    fprintf(fd, "\n");
    while(commands[i].name) {
      fprintf (fd, "%s %s\n", commands[i].name, commands[i].desc);
      i++;
    }
  } else
    fprintf(fd, "No additional help is available for %s\n.", subtopic);

}

void
quit(FILE *fd, char *ignored)
{
  migrated_done = 1;
}


void
resolve(FILE *fd, char *args)
{
  int             bptr, eptr, sid = 0;
  char           *endptr = NULL;
  migrate_state   state;
  session_handle *handle = NULL;

  for (bptr = 0; args[bptr] && whitespace(args[bptr]); bptr++);
  for (eptr = bptr; args[eptr] && !whitespace(args[eptr]); eptr++);

  if(args[eptr]) {
    fprintf(fd, "Usage: resolve session\n");
    return;
  } else
    args[eptr] = '\0';

  sid = strtol(&args[bptr], &endptr, 10);
  if(*endptr!='\0') {
    fprintf(fd, "Invalid session ID %s\n", &args[bptr]);
    return;
  }
  
  if(!(handle = session_id(sid))) {
    fprintf(fd, "Unknown session %d\n", sid);
    return;
  }

  state = resolve_session(handle);
  if(state == MIGRATE_FROZEN)
    fprintf(fd, "\nSession %d resolve failed\n", sid);

}

void
sessions(FILE *fd, char *ignored)
{
  session_ent *cur = session_list;
  migrate_connection *conn = NULL;
  struct passwd *pwent = NULL;
  struct fdhandle *lfd;
  struct _list_t *curr;

  while(cur) {
    fprintf(fd, "%u:%u\t%d\t%s->", cur->handle->session.id,
	    cur->handle->session.pid, cur->handle->pid,
	    (cur->handle->session.dname[0] ?
	     cur->handle->session.dname :
	     inet_ntoa(cur->handle->session.laddr.sin_addr)));
    fprintf(fd, "%s\t%s\t(", 
	    (cur->handle->session.pname[0] ?
	     cur->handle->session.pname :
	     inet_ntoa(cur->handle->session.paddr.sin_addr)),
	    migstate2ascii[cur->handle->session.state]);
    lfd = cur->handle->lfds;
    while(lfd) {
      fprintf(fd, "%d:%d,",(int)lfd->buff, lfd->fd);
      lfd = lfd->next;
    }
    if(cur->handle->cont && (cur->handle->cont->flags & M_COMPLETE))
      fprintf(fd, "%s ", (cur->handle->cont->db ?
			  (char *)cur->handle->cont->db : "missing"));
    fprintf(fd, "\b) %c%c%c%c%c\n",
	    (cur->handle->cont ? 'C' : ' '),
	    ((cur->handle->session.flags & M_AUTOCLOSE) ? 'A' : ' '),
	    (cur->handle->session.newf.arg ? 'R' : ' ' ),
	    ((cur->handle->session.flags & M_ALWAYSLOOKUP) ? 'L' : ' '),
	    ((cur->handle->session.flags & M_DONTMOVE) ? 'D' : ' '));

    /* Display individual connections */
    pwent = getpwuid(cur->handle->uid);
    if(pwent)
      fprintf(fd,"%s",pwent->pw_name);
    if((conn = cur->handle->anon)) {
      fprintf(fd, "ANON\t %sP\t\%s:%hu",
	      ((conn->type == SOCK_DGRAM) ? "UD" : "TC"),
	      inet_ntoa(conn->saddr.sin_addr),
	      ntohs(conn->saddr.sin_port));
      fprintf(fd, "\t%s:%hu\t\t\%d\n", inet_ntoa(conn->daddr.sin_addr),
	      ntohs(conn->daddr.sin_port), conn->refcnt);
      fprintf(fd, "\t\t(%s:%hu", inet_ntoa(conn->csaddr.sin_addr),
	      ntohs(conn->csaddr.sin_port));
      fprintf(fd, "\t%s:%hu)\n", inet_ntoa(conn->cdaddr.sin_addr),
	      ntohs(conn->cdaddr.sin_port));
    }
    for(curr = cur->handle->conns; curr; curr = curr->next) {
      conn = curr->data;
      fprintf(fd, "\t%sP\t\%s:%hu",
	      ((conn->type == SOCK_DGRAM) ? "UD" : "TC"),
	      inet_ntoa(conn->saddr.sin_addr),
	      ntohs(conn->saddr.sin_port));
      fprintf(fd, "\t%s:%hu\t\t\%d\n", inet_ntoa(conn->daddr.sin_addr),
	      ntohs(conn->daddr.sin_port), conn->refcnt);
      fprintf(fd, "\t\t(%s:%hu", inet_ntoa(conn->csaddr.sin_addr),
	      ntohs(conn->csaddr.sin_port));
      fprintf(fd, "\t%s:%hu)\n", inet_ntoa(conn->cdaddr.sin_addr),
	      ntohs(conn->cdaddr.sin_port));
    }
    fprintf(fd, "\n");
    cur = cur->next;
  }
}

void
version(FILE *fd, char *ignored)
{
  fprintf(fd, "Migrated version %s, by Alex C. Snoeren.\n", VERSION);
  fprintf(fd, "Copyright (C) 2001 Massachusetts Institute of Technology.\n");
}


command *
find_command (char *name)
{
  int i;

  for (i = 0; commands[i].name; i++)
    if (strcmp (name, commands[i].name) == 0)
      return (&commands[i]);
  return ((command *)NULL);
}

char *
command_generator (char *text, int state)
{
  static int list_index, len;
  const char *name;

  if (!state) {
    list_index = 0;
    len = strlen (text);
  } 
  
  while ((name = commands[list_index].name)) {
    list_index++;
    if (strncmp (name, text, len) == 0)
      return strdup(name);
  }
  
  return NULL;
}

#ifdef HAVE_LIBREADLINE
char **
interactive_completion(char *text, int start, int end)
{
  char **matches = (char **)NULL;

  if (start == 0)
    matches = completion_matches (text, command_generator);
  return (matches);
}
#endif

void
interactive_handler(char *line) {

  command *user_command;
  char *word;
  int i;

  if (line && line[0]) {
#ifdef HAVE_LIBREADLINE
    add_history(line);
    rl_callback_handler_remove();
#endif
    
    for (i = 0; line[i] && !whitespace (line[i]); i++);
      word = line;
      
      if (line[i])
	line[i++] = '\0';
      
      user_command = find_command (word);
      
      if (!user_command) {
	fprintf (stderr, "Unknown command: %s\n", word);
	goto done;
      }
      
      /* Get argument to command, if any. */
      for (; whitespace (line[i]); i++);
      word = line + i;
      
      /* Call the function. */
      (*(user_command->func)) (stdout, word);
  }

 done:
  free(line);
#ifdef HAVE_LIBREADLINE
  if (!migrated_done)
    rl_callback_handler_install("migrated> ", &interactive_handler);
#endif      

}
