/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: library.c,v 1.65 2002/10/10 22:09:55 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif

#include <sys/uio.h>

#include "migrated.h"

session_ent *session_list;

int
notify_libraries(session_handle *handle)
{
  struct fdhandle *lfd = NULL;
  int retval = 0;
  static mmsghdr hdr;

  hdr.type = SESSION_MSG;
  hdr.len = sizeof(migrate_session);

  /* Call continuation if we're migrating */
  if(handle->cont && ((handle->session.state == MIGRATE_LMIGRATING) ||
		      (handle->session.state == MIGRATE_PMIGRATING)) &&
     !(handle->cont->flags & M_COMPLETE)) {
    hdr.type = CONT_MSG;
    hdr.len += sizeof(migrate_continuation);
  }

  /* Notify library */
  lfd = handle->lfds;
  while (lfd) {
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG, 
	    "notify_libraries library write (%d:%d)", lfd->fd,
	    sizeof(migrate_session));

    hdr.pid = (pid_t)lfd->buff;
    if(write(lfd->fd, &hdr, sizeof(hdr)) != sizeof(hdr)) {
      /* Ignore EPIPE, we'll notice this next time we read */
      if(errno != EPIPE) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "notify_libraries: %s\n",
		strerror(errno));
	retval = errno;
      }
    }
    
    if(hdr.type == CONT_MSG) {
      if(write(lfd->fd, handle->cont, sizeof(migrate_continuation))
	 !=sizeof(migrate_continuation)) {
	/* Ignore EPIPE, we'll notice this next time we read */
	if(errno != EPIPE) {
	  log_log(LOG_MOD_MIGRATED, LOG_ERR, 
		  "continuation write (%d): %s", lfd->fd, strerror(errno));
	}
      }
    }

    if(write(lfd->fd, &handle->session, sizeof(migrate_session))
       !=sizeof(migrate_session)) {
      /* Ignore EPIPE, we'll notice this next time we read */
      if(errno != EPIPE) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, 
		"library write (%d): %s", lfd->fd, strerror(errno));
	retval = errno;
      }
    }
    lfd = lfd->next;
  }

  if(hdr.type == CONT_MSG) {
    free(handle->cont);
    handle->cont = NULL;
  }
    
  return (retval ? -1 : 0);
}


int
add_lfd(session_handle *handle, int fd, pid_t pid)
{
  struct fdhandle *new;

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	  "Adding %d:%d to session %d", pid, fd, handle->session.id);

  if(!(new = (struct fdhandle *)malloc(sizeof(*new)))) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "add_lfd (malloc): %s", strerror(errno));
    return -1;
  }
  new->fd = fd;
  new->buff = (char *)pid;
  new->next = NULL;
  if(handle->lfds) {
    struct fdhandle *fds = handle->lfds;
    do {
      if(fds->buff == new->buff) {
	log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		"Replacing library with same pid");
	fds->fd = new->fd;
	free(new);
	return 1;
      }
    }  while(fds->next && (fds = fds->next));
    fds->next = new;
  }
  else
    handle->lfds = new;

  return 0;
}


migrate_state 
connect_session(session_handle *handle)
{
  socklen_t len;

  /* Try and contact the other side */
  assert(handle);
  handle->pfd = socket(AF_INET, SOCK_STREAM, 0);
  if (handle->pfd == -1) {
    log_log(LOG_MOD_MIGRATED, LOG_CRIT, "socket: %s", strerror(errno));
    exit(1);
  }
  if(fcntl(handle->pfd, F_SETFL, O_NONBLOCK))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "unable to set socket %d non-blocking: %d, %s", handle->pfd,
	    errno, strerror(errno));
  if (fcntl(handle->pfd, F_SETFD, FD_CLOEXEC))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "fcntl (FD_CLOEXEC): %s", strerror(errno));
  handle->session.paddr.sin_port = htons(migrated_port);
  if((connect(handle->pfd, (const struct sockaddr *)&handle->session.paddr,
	      sizeof(struct sockaddr_in))==-1) && (errno != EINPROGRESS)) {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Can't connect to remote migrated on %s",
	    inet_ntoa(handle->session.paddr.sin_addr));
    close_session(handle, 1);
    set_state(handle, MIGRATE_NOTSUPPORTED);
  } else 
    set_state(handle, MIGRATE_CONNECTING);

  /* Initialize our local session endpoint */
  len = sizeof(handle->session.laddr);  
  if(handle->pfd != -1)
    if(getsockname(handle->pfd, (struct sockaddr *)&handle->session.laddr,
		   &len))
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "getsockname: %s", strerror(errno));

  return handle->session.state;  
}

session_handle *
find_session(int sessionid)
{
  session_ent *cur = session_list;

  while(cur) {    
    if(cur->handle->session.id == sessionid)
      if(cur->handle->session.state != MIGRATE_CONNECTING)
	return cur->handle;
    cur = cur->next;
  }

  return NULL;
}

session_handle *
find_anon(migrate_connection *conn, int src)
{
  session_ent *cur = session_list;

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "looking for anon %s", src?"SRC":"");

  while(cur) {    
    if((cur->handle->anon) && (!memcmp((src ? &cur->handle->anon->daddr :
					&cur->handle->anon->saddr),
				       (src ? &conn->saddr : &conn->daddr),
				       (sizeof(conn->saddr.sin_addr) +
					/* Don't check port if we're 0 */
					(ntohs(conn->saddr.sin_port) ?
					 sizeof(conn->saddr.sin_port) :
					 0)))))
      return cur->handle;
    cur = cur->next;
  }
  return NULL;
}


int
anon_conn(session_handle *handle, migrate_connection *conn)
{
  if(!(handle->anon = malloc(sizeof(*conn)))) {
    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
	    "malloc: %s", strerror(errno));
    return -1;
  }
  memcpy(handle->anon, conn, sizeof(*conn));
  conn->refcnt--;
  return 0;
}


static void
rebind_library(session_handle *handle, migrate_connection *conn,
	       session_handle *newhandle)
{
  struct fdhandle *lfd = NULL;
  static mmsghdr hdr;

  hdr.type = CONNREBIND_MSG;
  hdr.len = sizeof(migrate_connection);

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Current handle %u, conn %u new %u",
	  handle->session.id, ((session_handle*)conn->session)->session.id,
	  newhandle->session.id);

  /* Swap session pointers */
  assert((session_handle *)conn->session == handle);
  conn->session = (migrate_session *)handle->session.id;
  conn->rfd = newhandle->session.id;

  /* Update libraries */
  for(lfd = handle->lfds; lfd; lfd = lfd->next) {
    hdr.pid = (pid_t)lfd->buff;
    if(write(lfd->fd, &hdr, sizeof(hdr)) != sizeof(hdr)) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "rebind_library: %s\n",
		strerror(errno));
    } else {
      if(write(lfd->fd, conn, sizeof(migrate_connection))
	 !=sizeof(migrate_connection)) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, 
		"library write (%d): %s", lfd->fd, strerror(errno));
      }
    }
  }
  conn->session = (migrate_session *)newhandle;
}


session_handle *
session_merge(session_handle *handlea, session_handle *handleb, int full)
{
  struct _list_t *curr;
  int sid = 0;

  /* If nobody knew about this session, destroy it instead */

  if(!handlea->lfds) {
    sid = handlea->session.id;
    handlea->session.id = handleb->session.id;
  } else
    sid = handleb->session.id;
    

  /* Update session bindings for the moved connections */
  for(curr = handleb->conns; curr; curr = (struct _list_t *)curr->next) {
    rebind_library(handleb, (migrate_connection*)curr->data, handlea);
  }

  /* Move over connections */
  _list_append(&handlea->conns, handleb->conns);
  handleb->conns = NULL;

  /* XXX: Move library connections */
  if(full) assert(handlea->lfds == NULL);
  handlea->lfds = handleb->lfds;
  handleb->lfds = NULL;


  /* Move session stuff */

  assert(!handlea->uid||(handlea->uid==handleb->uid));
  handlea->uid = handleb->uid;

  if(full) {
    handlea->session.state = handleb->session.state;
  handlea->session.db = handleb->session.db;
  handlea->session.flags = handleb->session.flags;
  handlea->session.freeze = handleb->session.freeze;
  handlea->session.newf = handleb->session.newf;
  memcpy(handlea->session.dname, handleb->session.dname, M_MAXNAMESIZE);
  if(!handlea->session.pname[0])
    memcpy(handlea->session.pname, handleb->session.pname, M_MAXNAMESIZE);
 
  /* Move over handle */
  if(handlea->key == handleb->key)
    handleb->key = NULL;
  assert(!handlea->cont);
  handlea->cont = handleb->cont;
  handleb->cont = NULL;

  }

  assert(!(handlea->pid) || (handlea->pid == handleb->pid));
  handlea->pid = handleb->pid;

  /* Release second session */
  handleb->session.id = sid;
  remove_ent(close_session(handleb, 1));

  return handlea;
}


static int
conns_equal(const migrate_connection *cp1, const migrate_connection *cp2,
	    int local, int nat)
{
#define oaddr(cp, s) ( (local ^ s) ? cp->daddr : cp->saddr ) 
#ifdef DEBUG
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Comparing %s:%d",
	  inet_ntoa(cp1->saddr.sin_addr),
	  htons(cp1->saddr.sin_port));
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "to %s:%d",
	  inet_ntoa(oaddr(cp2, 1).sin_addr),
	  htons(oaddr(cp2, 1).sin_port));
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Comparing %s:%d",
	  inet_ntoa(cp1->daddr.sin_addr),
	  htons(cp1->daddr.sin_port));
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "to %s:%d",
	  inet_ntoa(oaddr(cp2, 0).sin_addr),
	  htons(oaddr(cp2, 0).sin_port));
#endif
  
  /* Some operating systems don't zero out the end of sockaddr_in, and
     don't support sin_len to let us figure out how much to compare */
  if (((!memcmp(&(cp1->saddr.sin_addr),
	       (local ? &cp2->saddr.sin_addr : &cp2->daddr.sin_addr),
	       sizeof(cp1->saddr.sin_addr))) &&
      (cp1->saddr.sin_port == oaddr(cp2, 1).sin_port)) &&
      (nat ||
       ((!memcmp(&cp1->daddr.sin_addr,
		 (local ? &cp2->daddr.sin_addr : &cp2->saddr.sin_addr),
		 sizeof(cp1->daddr.sin_addr))) &&
	(cp1->daddr.sin_port == oaddr(cp2, 0).sin_port))))
    return 1;
  else
    return 0;
}

int
conns_lequal(const void *cp1, const void *cp2)
{
  return conns_equal((migrate_connection *)cp1, (migrate_connection *)cp2,
		     1, 0);
}

int
conns_pequal(const void *lcp, const void *pcp)
{
  return conns_equal((migrate_connection *)lcp, (migrate_connection *)pcp,
		     0, 0);
}

int
conns_pNATequal(const void *lcp, const void *pcp)
{
  return conns_equal((migrate_connection *)lcp, (migrate_connection *)pcp,
		     0, 1);
}

int
conns_unbound(const void *pcp, const void *cp)
{
  /* Only allow this for UDP sockets */
  if(((migrate_connection *)cp)->type != SOCK_DGRAM)
    return 0;

  if (((((migrate_connection *)cp)->saddr.sin_addr.s_addr == INADDR_ANY) ||
       (!memcmp(&((migrate_connection*)cp)->daddr.sin_addr,
		&((migrate_connection *)pcp)->saddr.sin_addr,
		sizeof(((migrate_connection *)cp)->daddr.sin_addr)))) &&
      ((((migrate_connection*)cp)->daddr.sin_port ==
       ((migrate_connection *)pcp)->saddr.sin_port) ||
       (((migrate_connection *)pcp)->saddr.sin_port == 0)) &&
      (((migrate_connection *)cp)->saddr.sin_addr.s_addr == INADDR_ANY) &&
      ((((migrate_connection*)cp)->saddr.sin_port ==
	((migrate_connection *)pcp)->daddr.sin_port) ||
       (((migrate_connection *)cp)->saddr.sin_port == 0)))
    return 1;
  else
    return 0;
}



session_handle *
session_port(migrate_connection *conn)
{
  struct sockaddr_in *addr = &conn->daddr;
  struct sockaddr_in *saddr = &conn->saddr;
  session_ent *cur = session_list;
  struct _list_t *curr;

  assert(addr);

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	  "Looking for a session with %s:%hu",
	  inet_ntoa(addr->sin_addr), ntohs(addr->sin_port));

  /* Some operating systems don't zero out the end of sockaddr_in, and
     don't support sin_len to let us figure out how much to compare */

  /* First, try matching both local and remote ports */
  while(cur) {    
    for(curr = cur->handle->conns; curr; curr = curr->next) {
      migrate_connection *iconn = (migrate_connection *)curr->data;
      if ((!memcmp(&iconn->saddr.sin_addr, &addr->sin_addr,
		   sizeof(addr->sin_addr))) &&
	  (!memcmp(&iconn->saddr.sin_port, &addr->sin_port,
		   sizeof(addr->sin_port))) &&
	  (!memcmp(&iconn->daddr.sin_addr, &saddr->sin_addr,
		   sizeof(saddr->sin_addr))) &&
	  (!memcmp(&iconn->daddr.sin_port, &saddr->sin_port,
		   sizeof(saddr->sin_port))))
	return cur->handle;
    }
    cur = cur->next;
  }

  /* Fall back to checking only local ports in case of NAT */
  cur = session_list;
  while(cur) {    
    for(curr = cur->handle->conns; curr; curr = curr->next) {
      migrate_connection *iconn = (migrate_connection *)curr->data;
      if ((!memcmp(&iconn->saddr.sin_addr, &addr->sin_addr,
		   sizeof(addr->sin_addr))) &&
	  (!memcmp(&iconn->saddr.sin_port, &addr->sin_port,
		   sizeof(addr->sin_port))))
	return cur->handle;
    }
    cur = cur->next;
  }

  /* If we have a null local entry, match to anything */
  if (conn->type == SOCK_DGRAM) {
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	    "Checking for unbound connections on %hu", ntohs(addr->sin_port));
    cur = session_list;
    while (cur) {
      for(curr = cur->handle->conns; curr; curr = curr->next) {
	migrate_connection *iconn = (migrate_connection *)curr->data;
	if ((iconn->saddr.sin_addr.s_addr == 0) &&
	    (iconn->saddr.sin_port == addr->sin_port)) {
	  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Forcing match");
	  return cur->handle;
	}
      }
      cur = cur->next;
    }
  }

  /* If all else fails, check UDP streams from source.  This is NOT NAT
     compatible */

  if((conn->type == SOCK_DGRAM) && (ntohs(addr->sin_port) == 0)) {
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	    "Trying to match UDP on source address/port");
    
    addr = &conn->saddr;
    cur = session_list;

    while(cur) {    
      for(curr = cur->handle->conns; curr; curr = curr->next) {
	migrate_connection *iconn = (migrate_connection *)curr->data;
	if ((!memcmp(&iconn->daddr.sin_addr, &addr->sin_addr,
		     sizeof(addr->sin_addr))) &&
	    (!memcmp(&iconn->daddr.sin_port, &addr->sin_port,
		     sizeof(addr->sin_port))))
	  return cur->handle;
      }
      cur = cur->next;
    }
  }

  return NULL;
}


session_handle *
conn_session(migrate_connection *conn)
{
  session_ent *cur = session_list;

  assert(conn);

  while(cur) {    
    if(cur->handle->session.id == (int)conn->session) return cur->handle;
    cur = cur->next;
  }
  return NULL;
}

session_handle *
recv_session(int fd, pid_t mpid, void *buf)
{
  session_handle *handle = NULL;

  log_log(LOG_MOD_MIGRATED, LOG_NOTICE, "Received session %d %s->%s", 
	  ((migrate_session *)buf)->id, ((migrate_session *)buf)->dname,
	  ((migrate_session *)buf)->pname);

  /* Create a new session handle if necessary */
  if(((migrate_session *)buf)->id == -1) {

    if(!(handle = malloc(sizeof(session_handle)))) {
      log_log(LOG_MOD_MIGRATED, LOG_CRIT, "malloc: %s", strerror(errno));
      exit(1);
    }
    
    /* Initialize session */
    ((migrate_session *)buf)->id = new_sessionno();

    /* Initialize handle */
    memset(handle, 0, sizeof(session_handle));
    memcpy(&handle->session, buf, sizeof(migrate_session));
    handle->pfd = -1;
    if(add_lfd(handle, fd, mpid) < 0) {
      log_log(LOG_MOD_MIGRATED, LOG_CRIT, "Unable to add library fd");
      exit(1);
    }
    handle->pid = mpid;
    handle->key = NULL;
    
    add_entry(handle);
  }

  /* Find already open session handle */
  if (!handle && !(handle = find_session(((migrate_session *)buf)->id))) {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Can't find session %d", ((migrate_session *)buf)->id);
    return NULL;
  }

  switch (((migrate_session *)buf)->state) {

  case MIGRATE_ESTABLISHED:
  case MIGRATE_NOTCONNECTED:      

    /* Just update the parameters */
    
    handle->session.flags = ((migrate_session *)buf)->flags;
    memcpy(handle->session.dname, ((migrate_session *)buf)->dname,
	   M_MAXNAMESIZE);
    memcpy(handle->session.pname, ((migrate_session *)buf)->pname,
	   M_MAXNAMESIZE);

    /* Respond to request */
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG, 
	    "recv_session library write (%d:%d)", fd,
	    sizeof(migrate_session));
    notify_libraries(handle);
    break;
 
  case MIGRATE_CONNECTING: {
    
    
    /* First check to see if we have an anonymous session with this
       connection already laying about */
    
#if 0
    session_handle *newhandle = NULL;
    if(handle->conn &&
       (newhandle = find_anon(handle->conn, 0))) {
      
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Joining sessions %d and %d",
	      newhandle->session.id, handle->session.id);
      free(newhandle->anon);
      newhandle->anon = NULL;
      handle = session_merge(newhandle, handle);
      session_update_sess(handle);
      
    } else {
#endif
      /* This really is new, try and contact other side */
      if(handle->session.state != MIGRATE_ESTABLISHED) {
	memcpy(&handle->session, buf, sizeof(migrate_session));
	log_log(LOG_MOD_MIGRATED, LOG_INFO,
		"Connecting to remote host...");
	connect_session(handle);
      } else {
	log_log(LOG_MOD_MIGRATED, LOG_INFO,
		"Connecting already connected session.");
      }
#if 0
    }
#endif

    /* Don't notify library if we're still waiting... */
    if(handle->session.state != MIGRATE_CONNECTING) {
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG, 
	      "library (%d:%d) write:", fd, sizeof(migrate_session));
      notify_libraries(handle);
    }
    break;
  }
  
  case MIGRATE_CLOSED:
    {
      struct fdhandle *lfd, **prev;
      
      /* Remove library from this session */
      log_log(LOG_MOD_MIGRATED, LOG_INFO,
	      "Shutting down session %d for %d",
		handle->session.id, mpid);
      lfd = handle->lfds;
      prev = &handle->lfds;
      while(lfd) {
	if ((pid_t)lfd->buff == mpid) {
	  *prev = lfd->next;
	  free(lfd);
	  lfd = *prev;
	} else {
	  prev = &lfd->next;
	  lfd = lfd->next;
	}
      }
      if (!handle->lfds)
	remove_ent(close_session(handle, 0));
      handle = NULL;
      break;
    }

  case MIGRATE_FROZEN:
    {
      struct linger li; 
	
      li.l_onoff = 1; 
      li.l_linger = 0; 
      
      log_log(LOG_MOD_MIGRATED, LOG_INFO, "Receiving freeze notice for %d",
	      handle->session.id);
    
      if(handle->pfd != -1) {
	/* Say abort */
	if(setsockopt(handle->pfd, SOL_SOCKET, SO_LINGER, &li,
		      sizeof(struct linger)))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR, "Couldn't set SO_LINGER %s",
		  strerror(errno));
	if(handle->session.state != MIGRATE_FROZEN)
	  set_state(handle, MIGRATE_FROZEN);     
      }
      break;
    }

  case MIGRATE_LOST: {

    int retval = 0;

    /* First, possibly add the library to the session */
    retval = add_lfd(handle, fd, mpid);

    if(retval < 0) {
      log_log(LOG_MOD_MIGRATED, LOG_CRIT, "Unable to add library fd");
      exit(1);
    }

    /* If the app is giving us a new address to use; migrate */
    if(retval)
      mig_session(handle, &handle->session.laddr);
    else
      library_init(handle);

    break;
  }
  default:
    log_log(LOG_MOD_MIGRATED, LOG_CRIT, "Unhandled message type");
  }

  return handle;
}


int
update_conn(session_handle *handle, int fd, void *buf)
{
  migrate_connection *conn = NULL;

  assert(buf);

  /* If it doesn't already have connection, allocate one */
  if (!(conn = _list_find(handle->conns, (void *)buf, conns_lequal))) {
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "We're allocing a connection");
    if(!(conn = malloc(sizeof(migrate_connection)))) {
      log_log(LOG_MOD_MIGRATED, LOG_CRIT, "malloc: %s",
	      strerror(errno));
      exit(1);
    }
    memset(conn, 0, sizeof(migrate_connection));
    conn->refcnt = 1;
    _list_add(&handle->conns, conn);
  } 

  /* Update connection info */
  if (((migrate_connection *)buf)->rfd != -1) {

      session_handle *newhandle = NULL;

      unsigned short sport = conn->daddr.sin_port;
      unsigned int refcnt = conn->refcnt;

      /* Otherwise just update connection */
      memcpy(conn, buf, sizeof(migrate_connection));
      conn->refcnt = refcnt;
      conn->session = (migrate_session *)handle;

      /* Handle UDP binds to non-ports */
      if((conn->type == SOCK_DGRAM) && (ntohs(conn->daddr.sin_port) == 0) &&
	 (ntohs(sport) != 0)) {
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		"Overriding UDP port with %d", ntohs(sport));
	conn->daddr.sin_port = sport;
      }
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	      "Connection update: %d(%d):%d %s:%hu", handle->session.id,
	      handle->session.state, conn->fd, inet_ntoa(conn->saddr.sin_addr),
	      ntohs(conn->saddr.sin_port));
      
      /* Register connection with monitor, if address is nonzero */
      if (conn->saddr.sin_addr.s_addr)
	monitor_conn(conn, -1);

      /* Check to see if there are any anonymous conns that should be us */
      if((newhandle = find_anon(conn, 1))) {
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Joining sessions %d and %d",
		newhandle->session.id, handle->session.id);
	if(ntohs(conn->daddr.sin_port) == 0) {
	  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Taking anon's port %d",
		  ntohs(newhandle->anon->saddr.sin_port));
	  conn->daddr.sin_port = newhandle->anon->saddr.sin_port;
	}
	free(newhandle->anon);
	newhandle->anon = NULL;
	handle = session_merge(newhandle, handle, 1);
	session_update_sess(handle);
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Sending update to %d of %d:%d",
		handle->session.pid, handle->session.id, handle->session.pid);
      }

      /* Inform peer */
      if (handle->session.state == MIGRATE_ESTABLISHED)
	session_update_conn(conn);

  } else {

    log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Closing connection %d in %d, cnt %d",
	    ((migrate_connection *)buf)->fd, handle->session.id, conn->refcnt);
    /* Shutdown connection */
    if (!(--conn->refcnt)) {
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Killing connection");
      _list_del(&handle->conns, conn);
      free(conn);
    }
	
  }

  /* Tell library state of session */
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, 
	  "update_conn library write (%d:%d)", fd, sizeof(migrate_session));
  notify_libraries(handle);
  return 0;
}

migrate_state
freeze_session(session_handle *handle)
{
  if(handle->session.state==MIGRATE_ESTABLISHED) {
    set_state(handle, MIGRATE_FROZEN);
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Suspending session %d", handle->session.id);
    notify_libraries(handle);
  } else {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "We don't know how to suspend non-established states!");
  }
  return handle->session.state;
}

migrate_state
defrost_session(session_handle *handle)
{
  if(handle->session.state == MIGRATE_FROZEN) {
    set_state(handle, MIGRATE_ESTABLISHED);
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Resuming session %d", handle->session.id);
    notify_libraries(handle);
  } else {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "Session %d not frozen", handle->session.id);
  }
  return handle->session.state;
}

migrate_state
resolve_session(session_handle *handle)
{
  if(handle->session.newf.arg || handle->session.pname[0]) {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Invoking %s lookup function for session %d",
	    (handle->session.newf.arg ? "application" : "default"),
	    handle->session.id);
    set_state(handle, MIGRATE_LOST);
  } else {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "No lookup function defined, suspending %d", handle->session.id);
    set_state(handle, MIGRATE_FROZEN);
  }

  notify_libraries(handle);
  return handle->session.state;
}


migrate_state 
mig_session(session_handle *handle, struct sockaddr_in *addr)
{
  int sock;
  socklen_t len;

  /* Try and connect to the other side from our new location */
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock == -1) {
    log_log(LOG_MOD_MIGRATED, LOG_CRIT, "socket: %s", strerror(errno));
    exit(1);
  }
  if(fcntl(sock, F_SETFL, O_NONBLOCK))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "unable to set socket %d non-blocking: %s", sock,
	    strerror(errno));
  if(fcntl(sock, F_SETFD, FD_CLOEXEC))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "fcntl (FD_CLOEXEC): %s", strerror(errno));
  addr->sin_port = 0; /* We don't care what port we get */
  if(bind(sock, (struct sockaddr *)addr, sizeof(struct sockaddr_in))) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "Unable to migrate to %s:%u: %s", inet_ntoa(addr->sin_addr),
	    ntohs(addr->sin_port),
	    strerror(errno));
    close(sock);
    return MIGRATE_NOTCONNECTED;
  }

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	  "Initiating migration to %s", inet_ntoa(addr->sin_addr));
  swap_fds(handle, sock);

  /* XXX: Do we need to re-resolve peer's address first? */
  if (handle->session.flags & M_ALWAYSLOOKUP) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "M_ALWAYSLOOKUP not yet implemented");
  }

  /* We need to connect to the listening port */
  handle->session.paddr.sin_port = htons(migrated_port);
  if (connect(handle->pfd, (const struct sockaddr *)&handle->session.paddr,
	      sizeof(struct sockaddr_in))==-1)
    if(errno!=EINPROGRESS) {
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	      "Can't connect to remote migated on %s: %s",
	      inet_ntoa(handle->session.paddr.sin_addr), strerror(errno));
      close_session(handle, 1);
      switch (errno) {
      case EINVAL:
      case ENETUNREACH:
      case ETIMEDOUT:
	return (set_state(handle, MIGRATE_FROZEN));
      default:
	return (set_state(handle, MIGRATE_NOTSUPPORTED));
      }
    }

  len = sizeof(struct sockaddr_in);
  if(getsockname(handle->pfd, (struct sockaddr *)&handle->session.laddr,
		 &len)) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "getsockname: %s", strerror(errno));
    memcpy(&handle->session.laddr, &addr, sizeof(struct sockaddr_in));
  }

  return (set_state(handle, MIGRATE_CONNECTING));
}


int
update_library(migrate_connection *iconn, session_handle *handle)
{
  migrate_connection  conn;
  struct cmsghdr     *cm;
  char                cbuf[CMSG_SPACE(sizeof(int))]; 
  struct msghdr       msg = {0};
  struct iovec        iov;
  struct fdhandle    *lfd;
  static mmsghdr hdr;

  hdr.type = CONNECTION_MSG;
  hdr.len = 0;

  /* We may not be able to update yet, as continuation is still running */
  if(!handle->lfds) {
    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	    "Holding new connection until continuation completes");
    _list_add(&handle->update_pending, iconn);
    return 1;
  }

  assert((handle->lfds != NULL) && (iconn));

  memcpy(&conn, iconn, sizeof(conn));
  conn.session = (migrate_session *)handle->session.id;

  iov.iov_base = (caddr_t) &conn; 
  iov.iov_len = sizeof(conn); 
  msg.msg_iov = &iov; 
  msg.msg_iovlen = 1; 
  
  /* Pass new file descriptor */
  msg.msg_control = cbuf;
  msg.msg_controllen = sizeof(cbuf); 
  cm = CMSG_FIRSTHDR(&msg); 
  cm->cmsg_len = CMSG_LEN(sizeof(int)); 
  cm->cmsg_level = SOL_SOCKET;
  cm->cmsg_type = SCM_RIGHTS; 
    
  lfd = handle->lfds;
  while(lfd) {
    
    /* Create new fds for each library to avoid race conditions */
    if((*(int *)CMSG_DATA(cm) = dup(conn.rfd)) < 0) {
      log_log(LOG_MOD_MIGRATED, LOG_CRIT,
	      "dup failed: %s", strerror(errno));
      exit(1);
    }

    log_log(LOG_MOD_MIGRATED, LOG_DEBUG, 
	    "Updating conn on %d to %d(%d)\n",
	    lfd->buff, conn.rfd, *(int *)CMSG_DATA(cm));
    
    hdr.pid = (pid_t)lfd->buff;
    if(write(lfd->fd, &hdr, sizeof(hdr)) != sizeof(hdr)) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "update_library: %s\n",
		strerror(errno));
    }
    

    if (sendmsg(lfd->fd, &msg, 0) < 0) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "Unable to pass file descriptor: %s", strerror(errno));
      return -1;
    }
    close(*(int *)CMSG_DATA(cm));
    lfd = lfd->next;
  }

  close(conn.rfd);
  iconn->rfd = -1;

  return 0;
}

int
library_disconnect(int fd)
{
  session_ent *cur = session_list;
  struct fdhandle *lfd, **prev;

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Disconnecting library %d.", fd);

  /* Disconnect all sessions opened by this library */
  while(cur) {    
    prev = &cur->handle->lfds;
    lfd = cur->handle->lfds;
    while(lfd) {
      if(lfd->fd == fd) {
	log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		"Closing session %d for %d:%d", cur->handle->session.id,
		(int)lfd->buff, lfd->fd);
	*prev = lfd->next;
	free(lfd);
	lfd = *prev;
      } else {
	prev = &lfd->next;
	lfd = lfd->next;
      }
    }
    if((cur->handle->lfds == NULL) &&
       !((cur->handle->session.state == MIGRATE_FROZEN) && cur->handle->cont &&
      	 (cur->handle->cont->flags & M_COMPLETE))) {
      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	      "Deleting no-longer used session %d",
	      cur->handle->session.id);
      cur = remove_ent(close_session(cur->handle, 1));
    } else
      cur = cur->next;
  }

  return 0;
}



/* This function is ugly and NOT THREAD SAFE */

int
library_init(session_handle *handle)
{
  struct _list_t *curr;
  if(handle->mcp_pending) {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE, "Doing pending conns prepare");
    mcp(handle);
  }
  while((curr = handle->update_pending)) {
    migrate_connection *conn = (migrate_connection *)curr->data;
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE, "Doing pending library update");
    update_library(conn, handle);
    _list_del(&handle->update_pending, conn);
  }
  
  return 0;
}


