/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * Copyright (c) 1999 BBN Corporation
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice and this permission
 * appear in all copies and in supporting documentation, and that the
 * name of BBN Corporation not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission. BBN makes no representations about the
 * suitability of this software for any purposes.  It is provided "AS
 * IS" without express or implied warranties.
 *
 * This software and its documentation was written by BBN Corporation
 * under sponsorship by the Defense Advanced Research Projects Agency.
 *
 * $Id: log.c,v 1.5 2002/05/31 17:46:20 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_TIME_H
# include <time.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_FILE_H
# include <sys/file.h>
#endif
#ifdef HAVE_READLINE_READLINE_H
# include <readline/readline.h>
#endif



#include "log.h"

/* Log variables */
char *log_ident = NULL;
char *log_fname = NULL;
int log_method = LOG_METH_STDERR;
int log_module_levels[sizeof(int) * 8];

#undef EXAMPLE_USAGE
#ifdef EXAMPLE_USAGE

int main(int argc, char *argv[])
{
	char dir[1024], name[1024];
	int i;

	/* put together directory name */
	sprintf(dir, "%s", ".");

	/* construct log file name */
	sprintf(name, "%s/messages", dir);

	/* set up logging module */
	log_init(argv[0], name, LOG_METH_APPL_LOG_FILE | LOG_METH_STDERR,
							LOG_WARNING);
	log_setlevel(LOG_MOD_LDTP, LOG_CRIT);
	log_setlevel(LOG_MOD_AUTHEN, LOG_INFO);

	/* do some logging */
	for (i = LOG_EMERG; i <= LOG_DEBUG; i++) {
		char buf[20];
		sprintf(buf, "%d %d", getpid(), i);
		log_log(LOG_MOD_LDTP, i, buf);
		log_log(LOG_MOD_AUTHEN, i, buf);
	}

	printf("filename is %s\n", log_getfilename());
	printf("method is %s\n", log_method2ascii(log_getmethod()));
	for (i = 1; i != 0; i <<= 1) {
		printf("%x: %s\n", i, log_level2ascii(log_getlevel(i)));
	}
}
#endif


/* Initialize the logging module */
void log_init(const char *ident, const char *fname, int method, int level)
{
	int i, num = sizeof(int) * 8;

	/* Do some cleanup in case the class has been initialized
	 * multiple times. */
	if (log_ident) {
		free(log_ident);
		log_ident = NULL;
	}
	if (log_fname) {
		free(log_fname);
		log_fname = NULL;
	}

	/* Initialize various variables */
	if (ident) log_ident = strdup(ident);
	if (fname) log_fname = strdup(fname);
	log_method = method;
	for (i = 0; i < num; i++) log_module_levels[i] = level;
}


/* Wrapper function for printf style logging; get a va_list and
 * call vlog which will do the real work */
void log_log(int mod, int priority, const char *msg, ...)
{
	va_list vargs;
	va_start(vargs, msg);
	log_vlog(mod, priority, msg, vargs);
	va_end(vargs);
}

/* This function does the real work for logging */
void log_vlog(int mod, int priority, const char *msg, va_list vargs)
{
	int num = sizeof(int) * 8, module, mod_test, rval, len = 1024;
	char *buf = NULL, *buf2 = NULL, *t, timebuf[20];
	const char *pname, *mname;
	time_t now;

	/* see if we want to log this */
	for (module = 0; module < num; module++) {
		mod_test = 1 << module;
		if ((mod & mod_test) && priority <= log_module_levels[module])
			break;
	}
	if (module == num) return;

	/* make buffer for constructing strings */
	buf = malloc(len);
	
	/* try putting message together with buffer we've got */
	rval = vsnprintf(buf, len, msg, vargs);

	/* if buffer is too small, reallocate and rebuild the string */
	if (rval >= len) {
		len = rval + 10;
		free(buf);
		buf = malloc(len);
		rval = vsnprintf(buf, len, msg, vargs);
	}

	/* get the time stamp */
	now = time(NULL);
	t = ctime(&now);
	strncpy(timebuf, t + 4, 15);
	timebuf[15] = '\0';

	/* get the priority name */
	pname = log_level2ascii(priority);

	/* get the module names */
	mname = log_mod2ascii(mod);

	/* get a large enough 2nd buffer */
	len = rval + strlen(pname) + strlen(timebuf) + strlen(mname) + 10;
	if (log_ident) len += strlen(log_ident);
	buf2 = malloc(len);
	
	/* put together the final string */
	rval = sprintf(buf2, "%s %s %s: %s", timebuf, pname,
					(log_ident ? log_ident : ""), buf);

	/* add a newline if needed */
	if (buf2[rval - 1] != '\n') {
		buf2[rval++] = '\n';
		buf2[rval] = '\0';
	}

	/* if we're logging to an application log file, do it */
	if (log_fname && log_method & LOG_METH_APPL_LOG_FILE) {

		/* lock the file */
		int fd = open(log_fname, O_APPEND | O_CREAT | O_WRONLY, 0664);
		if (fd != -1) {
			flock(fd, LOCK_EX);
			write(fd, buf2, rval);
			flock(fd, LOCK_UN);
			close(fd);
		}
	}

	/* if we're putting things out to stderr, do it */
	if (log_method & LOG_METH_STDERR) fprintf(stderr, "%s", buf2);

	/* if we're doing syslog, do it */
	if (log_method & LOG_METH_SYSLOG) syslog(priority, buf);

#ifdef HAVE_LIBREADLINE
	/* if we're logging to readline console, do it */
	if (log_method & LOG_METH_CONSOLE) {
	  rl_save_prompt();
	  rl_message("%s",buf2);
	  rl_forced_update_display();	  
	  rl_restore_prompt();
	  rl_forced_update_display();	  
	}
#endif

	/* cleanup */
	if (buf) {
		free(buf);
		buf = NULL;
	}
	if (buf2) {
		free(buf2);
		buf2 = NULL;
	}
}

/* This function sets the verbosity level */
void log_setlevel(int mod, int pri)
{
	int module, mod_test, num = sizeof(int) * 8;
	
	/* find out which modules we're setting to the specified level
	 * and then set the level */
	for (module = 0; module < num; module++) {
		mod_test = 1 << module;
		if (mod & mod_test) log_module_levels[module] = pri;
	}
}

/* add text to a buffer */
static int add_entry(int, int, char *, const char *);
static int add_entry(int num, int len, char *buf, const char *txt)
{
	if (num) buf[len++] = '|';
	strcpy(&buf[len], txt);
	return strlen(buf);
}

/* put together the module name(s) */
const char *log_mod2ascii(int mod)
{
	static char buf[1024];
	int num = 0, len;

	buf[0] = '\0';
	len = 0;

	if (mod & LOG_MOD_MIGRATED) {
		len = add_entry(num++, len, buf, "migrated");
	}
	if (mod & LOG_MOD_CRYPTO) {
		len = add_entry(num++, len, buf, "crypto");
	}
	return buf;
}

/* this routine converts an ascii log level to a log level value
 * if the ascii isn't recognized, this routine will return -1 */
int log_ascii2level(const char *level)
{
	if (!strncmp(level, "alert", 5)) return LOG_ALERT;
	else if (!strncmp(level, "crit", 4)) return LOG_CRIT;
	else if (!strncmp(level, "debug", 5)) return LOG_DEBUG;
	else if (!strncmp(level, "emerg", 5)) return LOG_EMERG;
	else if (!strncmp(level, "err", 3)) return LOG_ERR;
	else if (!strncmp(level, "info", 4)) return LOG_INFO;
	else if (!strncmp(level, "noti", 4)) return LOG_NOTICE;
	else if (!strncmp(level, "panic", 5)) return LOG_EMERG;
	else if (!strncmp(level, "warn", 4)) return LOG_WARNING;
	else return -1;
}

/* This routine converts a log level value to its ascii equivalent */
const char *log_level2ascii(int level)
{
	static char buf[32];
	switch(level) {
		case LOG_EMERG:		return "emerg";
		case LOG_ALERT:		return "alert";
		case LOG_CRIT:		return "crit";
		case LOG_ERR:		return "err";
		case LOG_WARNING:	return "warn";
		case LOG_NOTICE:	return "notice";
		case LOG_INFO:		return "info";
		case LOG_DEBUG:		return "debug";
		default:
			sprintf(buf, "unknown(%d)", level);
			return buf;
	};
}

/* this routine converts an ascii logging method to a LOG_METH_ value
 * if the ascii isn't recognized, this routine will return 
 * LOG_METH_UNKNOWN
 * Method can be specified as any combination (in any order) of efs.
 *	e signifies stderr
 *	f signifies a project log file
 *	s signifies syslog
 */
int log_ascii2method(const char *mstr)
{
	int i, len = strlen(mstr);
	int method = 0;

	for (i = 0; i < len; i++) {
		switch(mstr[i]) {
			case 'e':
				method |= LOG_METH_STDERR;
				break;
			case 'f':
				method |= LOG_METH_APPL_LOG_FILE;
				break;
			case 's':
				method |= LOG_METH_SYSLOG;
				break;
			default:
				return LOG_METH_UNKNOWN;
		}
	}

	return method;
}

/* this routine converts a LOG_METH_ value to its ascii equivalent
 * if the value isn't recognized, this routine will return "unknown (%d)"
 */
const char *log_method2ascii(int method)
{
	static char buf[128];
	int index = 0;

	if (method & LOG_METH_STDERR) buf[index++] = 'e';
	if (method & LOG_METH_APPL_LOG_FILE) buf[index++] = 'f';
	if (method & LOG_METH_SYSLOG) buf[index++] = 's';
	buf[index] = '\0';
	if (index == 0) {
		sprintf(buf, "unknown (%d)", method);
	}
	return buf;
}

/* this routine returns the log file name */
const char *log_getfilename(void)
{
	return log_fname;
}

/* this routine returns the logging method */
int log_getmethod(void)
{
	return log_method;
}

/* this routine returns the level for the specified module (it will return
 * the level for the first module bit it finds turned on)
 */
int log_getlevel(int mod)
{
	int module, mod_test, num = sizeof(int) * 8;
	
	/* find out which modules we're setting to the specified level
	 * and then set the level */
	for (module = 0; module < num; module++) {
		mod_test = 1 << module;
		if (mod & mod_test) return log_module_levels[module];
	}

	/* return level for undefined level -> this will have whatever
	 * the module was initialized to */
	return log_module_levels[23];
}
