/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * Copyright (c) 1999 BBN Corporation
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice and this permission
 * appear in all copies and in supporting documentation, and that the
 * name of BBN Corporation not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission. BBN makes no representations about the
 * suitability of this software for any purposes.  It is provided "AS
 * IS" without express or implied warranties.
 *
 * This software and its documentation was written by BBN Corporation
 * under sponsorship by the Defense Advanced Research Projects Agency.
 *
 * $Id: log.h,v 1.3 2001/10/01 15:21:42 snoeren Exp $
 */

/*
 * File:	log.h
 * Author:	Bev Schwartz
 * Date:	July 15, 1999
 *
 *	Logging Module
 *
 *	The logging module is initialied with the function log_init.
 *
 *	Verbosity level for _all_ modules is set to the default given to
 *	log_init when log_init is called.  After that, modules can have
 *	their verbosity level set with log_setlevel.
 *
 *	Use log_log for printf style logging; log_vlog for
 *	vprintf style logging.
 *
 *	See documentation below for information about arguments to
 *	all of the functions mentioned above.
 *
 *	Notes:
 *	- if the module isn't initialized, logging will be done for
 *	  warnings and higher, and the messages will go to stderr
 *	- if the module is initialized more than once, the module will
 *	  proceed with the new settings; module verbosity levels will
 *	  be reset to the level given to log_init
 *	- since modules are indicated with bit fields, any of the 
 *	  module arguments can take multiple modules or'd together
 *	- if log_init is not given a file name for an application log file,
 *	  but is told to do logging to an application log file, it will not
 *	  do put logging messages in a local application file.  Logging to
 *	  an application log file will happen only if a valid file name
 *	  is given _and_ logging to an application log file is requested.
 *	- flock is used to get exclusive write privileges for the
 *	  application log file
 */

#ifndef _LOG_H
#define _LOG_H

#if HAVE_CONFIG_H
# include <config.h>
#endif
#if STDC_HEADERS
# include <stdarg.h>
#endif
#if HAVE_SYSLOG_H
# include <syslog.h>
#endif

/* different modules in Migrate */
#define LOG_MOD_UNKNOWN  0x0
#define LOG_MOD_MIGRATED 0x01
#define LOG_MOD_CRYPTO   0x02
#define LOG_MOD_MONITOR  0x04
#define LOG_MOD_ALL      0xffffffff

/* different logging methods in Migrate */
#define LOG_METH_UNKNOWN       0x0
#define LOG_METH_APPL_LOG_FILE 0x01
#define LOG_METH_SYSLOG        0x02
#define LOG_METH_STDERR        0x04
#define LOG_METH_CONSOLE       0x08

/*
 * Function: void log_init(const char *, const char *, int, int)
 * Parameters:
 *	const char *;	- an executable identifier, e.g. argv[0]
 *	const char *;	- a fully qualified file name where stuff gets logged
 *	int method;	- where messages get logged (see LOG_METH_ #defines)
 *	int level;	- the default level which all modules will be logged
 *			  (each module can be changed individually later);
 *			  logging levels come from syslog.h.
 * Return: void
 * Purpose:
 *	Initialize the logging module.  If the logging module isn't
 *	initialized, defaults will be used
 */
void log_init(const char *, const char *, int, int);

/*
 * Functions: void log_log(int, int, const char *, ...);
 *	      void log_vlog(int, int, const char *, va_list);
 * Parameters:
 *	int;		- the module(s) to which this log message applies
 *			  (see LOG_MOD #defines above)
 *	int;		- the priority level (from syslog.h)
 *	const char *;	- a printf style string
 *	...;		- the rest of the printf arguments
 *	or
 *	va_list;	- vprintf arguments
 * Return: void
 * Purpose:
 *	A message can be logged to stderr, syslog, or an application log
 *	file, depending on which method was indicated when log_init was
 *	called.
 *	The message will be logged only if the logging level passed to
 *	this function is as high as or higher than the level set for
 *	the module(s) indicated in the module argument.
 */
void log_log(int, int, const char *, ...);
void log_vlog(int, int, const char *, va_list);
        
/*
 * Function: void log_setlevel(int, int);
 * Parameters:
 *	int;		- the module(s) whose level is being set (see
 *			  (LOG_MOD #defines above)
 *	int;		- the priority level (from syslog.h)
 * Return: void
 * Purpose:
 *	Set the priority level for what level logging messages should
 *	be reported.  Only message at this level or higher will be reported.
 */
void log_setlevel(int, int);

/*
 * Function: int log_ascii2level(const char *);
 * Parameters:
 *	const char *;	- an ascii level to be converted to a LOG_ level
 *			  from syslog.h
 * Return: int
 *	A LOG_ level if ascii is recognized; -1 otherwise.
 * Purpose:
 *	Convert from human readable log level to LOG_ level as defined
 *	in syslog.h.  Function is case insensitive.
 */
int log_ascii2level(const char *);

/*
 * Function: const char *log_level2ascii(int);
 * Parameters:
 *	int;		- a LOG_ level from syslog.h
 * Return: const char *
 *	Return an ascii representation of the LOG_ level.  If unknown,
 *	"unknown(%d)" is returned, where %d is the value provided to
 *	the routine.
 * Purpose:
 *	Convert from LOG_ value from syslog.h to a human readable form.
 */
const char *log_level2ascii(int);
       
/*
 * Function: int log_ascii2method(const char *);
 * Parameters:
 *	const char *;	- an ascii representation of logging method
 * Return: int
 *	A LOG_METH_ value (or a number or'd together) as defined above.
 *	If not successful, returns LOG_METH_UNKNOWN.
 * Purpose:
 *	A logging method can be expressed in any combination (in any order)
 *	of the letters "efs", where 'e' indicates stderr, 'f' indicates
 *	an application log file, and 's' indicates syslog.
 */
int log_ascii2method(const char *);

/*
 * Function: const char * log_method2ascii(int);
 * Parameters:
 *	int;		- a numeric representation of the logging method
 * Return: const char *
 *	Ascii representation of the logging method.  If unknown,
 *	"unknown(%d)" is returned, where %d is the value provided to
 *	the routine.
 * Purpose:
 *	Convert from method bit map to "efs" ascii encoding.
 */
const char *log_method2ascii(int);

/*
 * Function: const char *log_mod2ascii(int);
 * Parameters:
 *	int;		- a bit map of LOG_MOD_ values, as defined above
 * Return: const char *
 *	Return a string representing the different modules represented
 *	in the argument passed to the function.
 * Purpose:
 *	Convert a bit map of LOG_MOD_ values to something human readable.
 */
const char *log_mod2ascii(int);

/*
 * Function: int log_ascii2mod(const char *);
 * Parameters:
 *	const char *;	- an ascii representation of a single module
 * Return: int
 *	Return a LOG_MOD_ value if the ascii is recognized, LOG_MOD_UNKNOWN
 *	otherwise.
 * Purpose:
 *	Convert an human readable module to a LOG_MOD_ value.
 */
int log_ascii2mod(const char *);

/*
 * Function: const char * log_getfilename(void);
 * Parameters: none
 * Return: char *
 *	The name of the log file.
 * Purpose:
 *	An accessor which returns the name of the log file.
 */
const char *log_getfilename(void);

/*
 * Function: int log_getmethod(void);
 * Parameters: none
 * Return: int
 *	current logging method
 * Purpose:
 *	An accessor which returns the current logging method.
 */
int log_getmethod(void);

/*
 * Function: int log_getlevel(int);
 * Parameters:
 *	int mod;		which module 
 * Return: int
 *	the syslog LOG_ logging level for the module
 * Purpose:
 *	An accessor which returns the logging level for a specified module.
 */
int log_getlevel(int);

#endif




