/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: migrated.c,v 1.78 2002/10/10 21:50:21 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#ifdef HAVE_SYS_UN_H
# include <sys/un.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_READLINE_READLINE_H
# include <readline/readline.h>
#endif

#include <list.h>
#include <fdpass.h>
#include "migrated.h"

static char *lpipefmt = "/tmp/.migratepipe-%d";
static char *mpipefmt = "/tmp/.migratemonitor-%d";

static char lpipename[256];
static char mpipename[256];

static int debuglevel = LOG_NOTICE;

#ifdef HAVE_GETOPT_LONG
static struct option long_options[] = {
  {"debug", 1, 0, 'd'},
  {"file", 1, 0, 'f'},
  {"help", 0, 0, 'h'},
#ifdef HAVE_LIBREADLINE
  {"interactive", 0, 0, 'i'},
#endif
  {"port", 1, 0, 'p'},
  {"version", 0, 0, 'v'},
  {0, 0, 0, 0}
};
#endif

static int interactive = 0;
int migrated_done = 0;

int migrated_port = MIGPORT;

static session_handle *netfds[FD_SETSIZE];
static fd_set afds, ofds, cfds, lfds, nfds, mfds, tfds;
static unsigned int _sessionno = 1;
static int nsock = -1;
static int lsock = -1;
static int msock = -1;

unsigned int
new_sessionno(void)
{
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Created session %d", _sessionno);
  return _sessionno++;
}

static size_t
rcvbufsize(int fd)
{
  int bufsize;
  size_t len = sizeof(bufsize);

  /* Determine standard TCP buffer size for this end point */
  if (getsockopt(fd, SOL_SOCKET, SO_RCVBUF, &bufsize, &len)) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "getsockopt: %s, using %d\n", strerror(errno),
	    DEFAULT_RCVBUF);
    bufsize = DEFAULT_RCVBUF;
  } else {
    log_log(LOG_MOD_MIGRATED, LOG_INFO,
	    "Setting default RCVBUF size to %u", bufsize);
  }
  return bufsize;
}

static void
init_session(int fd, session_handle *handle)
{
  size_t bufsize;
  int len = sizeof(bufsize);

  memset(handle, 0, sizeof(session_handle));
  handle->session.id = new_sessionno();
  handle->lfds = NULL;
  handle->pfd = fd;
  FD_SET(fd, &ofds);
  FD_SET(fd, &nfds);
  netfds[fd] = handle;
  handle->lbufsize = rcvbufsize(fd);

  /* Identify endpoints */
  len = sizeof(struct sockaddr_in);
  if(getsockname(fd, (struct sockaddr *)&handle->session.laddr, &len))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "getsockname: %s", strerror(errno));
  len = sizeof(struct sockaddr_in);
  if(getpeername(fd, (struct sockaddr *)&handle->session.paddr, &len))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "getpeername: %s", strerror(errno));
}

/* Add session to session list */
session_ent *
add_entry(session_handle *handle)
{
  session_ent *ent, *cur = session_list;

  if(!(ent = (session_ent *)malloc(sizeof(session_ent)))) {
    log_log(LOG_MOD_MIGRATED, LOG_CRIT, "malloc: %s", strerror(errno));
    exit(1);
  }
  ent->handle = handle;
  ent->next = NULL;
  
  if(!session_list)
    session_list = ent;
  else {
    while(cur->next) cur = cur->next;
    cur->next = ent;
  }

  /* log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
     "Registering session %d", handle->session.id); */

  return ent;
}

/* Remove handle for session list and free datastructures */
session_ent *
remove_ent(session_handle *handle)
{
  session_ent **prev = &session_list;
  session_ent *cur = session_list;

  if (!handle)
    return NULL;

  while(cur) {
    if(cur->handle == handle)  break;
    prev = (session_ent **)&cur->next;
    cur = cur->next;
  }
  
  if(!cur) {
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Can't find session %d", handle->session.id);
    return NULL;
  }

  handle = cur->handle;
  *prev = cur->next;
      
    /* Close off our copies of the fds */
  if(handle->conns) {
    struct _list_t *curr = handle->conns;
    while(curr) {
      free(curr->data);
      curr->data = NULL;
      curr = (struct _list_t *)curr->next;
    }
    _delete_list(&handle->conns);
  }

  /* Nuke any continuation data */
  if(handle->cont) {
    struct _list_t *curr = (struct _list_t *)handle->cont->fds;
    while(curr) {
      struct fdhandle_t *fdh = (struct fdhandle_t *)(curr->data);
      close(fdh->fd);
      curr = (struct _list_t *)curr->next;
    }
    _delete_list((struct _list_t **)&handle->cont->fds);
    if(handle->cont->flags & M_COMPLETE) {
      /* XXX: Remove TS file and db file */
      unlink((char *)handle->cont->db);
      free((void *)handle->cont->db);
    }
    free(handle->cont);
  }

  /* Free crypto stuff */
  if (handle->key) {
    free(handle->key);
    handle->key = NULL;
  }

  free(handle);
  handle = NULL;
  cur->handle = NULL;
  free(cur);  

  return *prev;
}

session_handle *
close_session(session_handle *handle, int force)
{
  struct _list_t *curr = NULL;

  /* Don't close sessions somebody else is going to want */
  if (!force)
    for(curr = handle->conns; curr; curr = (struct _list_t *)curr->next)
      if(((migrate_connection *)curr->data)->refcnt)
	return NULL;

  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Closing session %d", handle->session.id);

  if (handle->pfd != -1) {
    close(handle->pfd);
    FD_CLR(handle->pfd, &cfds);
    FD_CLR(handle->pfd, &ofds);
    FD_CLR(handle->pfd, &nfds);	      
    handle->pfd = -1;
    netfds[handle->pfd] = NULL;
  }
  return handle;
}


migrate_state
set_state(session_handle *handle, migrate_state state)
{
  /* Ignore duplicate calls */
  if(state == handle->session.state)
    return state;

  switch(handle->session.state) {
  case MIGRATE_FROZEN:
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Moving session %d out of freeze", handle->session.id);
    if(handle->cont && (handle->cont->flags & M_COMPLETE)) {
      complete_cont(handle);
    }
    break;
  default:
    break;
  }
  
  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	  "Moved session %d:%d from %s to %s", handle->session.id,
	  handle->session.pid, migstate2ascii[handle->session.state],
	  migstate2ascii[state]);
  
  return handle->session.state = state;
}


static void
migrated_cleanup(int ignored)
{
  log_log(LOG_MOD_MIGRATED, LOG_NOTICE, "Migrated exiting.");

  if (nsock != -1) close(nsock);
  if (lsock != -1) close(lsock);
  if (msock != -1) close(msock);
 
  unlink(lpipename);
  unlink(mpipename);

  exit(0);
}


int
manage_conns(void) {
  
  struct sockaddr_in iaddr;
  struct sockaddr_un uaddr;
  fd_set rfds, wfds;
  socklen_t len;
  int readlen;
  int selnum = 0;
  int on = 1;
  int err, fd = -1;
  char buf[BUFLEN];
  mmsghdr msg;

  /* Open network listening socket */
  memset(&iaddr, 0, sizeof(iaddr));
  iaddr.sin_family = AF_INET;
  iaddr.sin_addr.s_addr = INADDR_ANY;
  iaddr.sin_port = htons(migrated_port);

  if ((nsock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "socket: %s", strerror(errno));
    goto shutdown;
  }
  if (setsockopt(nsock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to set SO_REUSEADDR: %s",
	    strerror(errno));
  }
  if (fcntl(nsock, F_SETFD, FD_CLOEXEC)) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "fcntl (FD_CLOEXEC): %s", strerror(errno));
  }
  if (bind(nsock, (struct sockaddr *)&iaddr, sizeof(iaddr)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to bind to port %d: %s",
	    migrated_port, strerror(errno));
    goto shutdown;
  }
  if (fcntl(nsock, F_SETFL, O_NONBLOCK))
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "Unable to set listening port non-blocking: %s", strerror(errno));
  if (listen(nsock, 1) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to listen on port %d: %sn",
	    migrated_port, strerror(errno));
    goto shutdown;
  }
  memset(netfds, 0, FD_SETSIZE*sizeof(int));

  /* Open library named pipe */
  memset(&uaddr, 0, sizeof(uaddr));
  uaddr.sun_family = AF_UNIX;

  sprintf(lpipename, lpipefmt, migrated_port);
  strcpy(uaddr.sun_path, lpipename);
  
  if ((lsock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "socket: %s", strerror(errno));
    goto shutdown;
  }
  if (fcntl(lsock, F_SETFD, FD_CLOEXEC)) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "fcntl (FD_CLOEXEC): %s", strerror(errno));
  }
  if (bind(lsock, (struct sockaddr *)&uaddr, sizeof(uaddr)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to open pipe: %s",
	    strerror(errno));
    goto shutdown;
  }
  /* Set permissions */
  if (chmod(lpipename, (S_IRWXG|S_IRWXO|S_IRWXU))) 
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "fchmod: %s",
	    strerror(errno));
  if (listen(lsock, 1) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "listen: %s",
	    strerror(errno));
    goto shutdown;
  }

  /* Open monitor named pipe */
  memset(&uaddr, 0, sizeof(uaddr));
  uaddr.sun_family = AF_UNIX;
  sprintf(mpipename, mpipefmt, migrated_port);
  strcpy(uaddr.sun_path, mpipename);
  
  if ((msock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "socket: %s", strerror(errno));
    goto shutdown;
  }
  if (fcntl(msock, F_SETFD, FD_CLOEXEC)) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR,
	    "fcntl (FD_CLOEXEC): %s", strerror(errno));
  }
  if (bind(msock, (struct sockaddr *)&uaddr, sizeof(uaddr)) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to open pipe: %s",
	    strerror(errno));
    goto shutdown;
  }
  /* Set permissions */
  if (chmod(mpipename, (S_IRWXG|S_IRWXO|S_IRWXU))) 
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "fchmod: %s",
	    strerror(errno));
  if (listen(msock, 1) < 0) {
    log_log(LOG_MOD_MIGRATED, LOG_ERR, "listen: %s",
	    strerror(errno));
    goto shutdown;
  }

  FD_ZERO(&afds);
  FD_ZERO(&rfds);
  FD_ZERO(&wfds);
  FD_ZERO(&ofds);
  FD_ZERO(&cfds);
  FD_ZERO(&lfds);
  FD_ZERO(&nfds);
  FD_ZERO(&mfds);
  FD_ZERO(&tfds);
  FD_SET(nsock, &rfds);
  FD_SET(lsock, &rfds);
  FD_SET(msock, &rfds);
#ifdef HAVE_LIBREADLINE
  if (interactive) {
    log_init("migrated", NULL, LOG_METH_CONSOLE, debuglevel);    
    FD_SET(STDIN_FILENO, &rfds);
    rl_attempted_completion_function = (char**(*)(void))interactive_completion;
    rl_callback_handler_install("migrated> ", (void(*)())&interactive_handler);
  } else
#endif
    log_init("migrated", NULL, LOG_METH_SYSLOG, debuglevel);
 
  log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	  "Migrated %s on port %d", VERSION, migrated_port);

  while (!migrated_done) {
    selnum = select(FD_SETSIZE, &rfds, &wfds, NULL, 0);
    if (selnum < 0) {
      if(errno != EINTR) {
	log_log(LOG_MOD_MIGRATED, LOG_WARNING, "select: %s", strerror(errno));
	assert(0);
      }
      continue;
    }
    /* Accept new local library connections */
    if (FD_ISSET(lsock, &rfds)) {
      len = sizeof(uaddr);
      if ((fd = accept(lsock, (struct sockaddr *) &uaddr, &len)) < 0 ) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "library connection accept: %s",
		strerror(errno));
	exit(0);
      } else {
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		"Accepted a library connection (%d)...", fd);
	if (fcntl(fd, F_SETFD, FD_CLOEXEC))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR,
		  "fcntl (FD_CLOEXEC): %s", strerror(errno));
	FD_SET(fd, &ofds);
	FD_SET(fd, &lfds);
      }
      FD_CLR(lsock, &rfds);
      selnum--;
    }

    /* Accept new local monitor connections */
    if (FD_ISSET(msock, &rfds)) {
      len = sizeof(uaddr);
      if ((fd = accept(msock, (struct sockaddr *) &uaddr, &len)) < 0 ) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "accept: %s",
		strerror(errno));
      } else {
	if(fcntl(fd, F_SETFD, FD_CLOEXEC))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR,
		  "fcntl (FD_CLOEXEC): %s", strerror(errno));
	log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		"Accepted a monitor connection...");

	if (!monitor_open(fd))
	{
	  log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		  "Denied monitor connection.");
	  close(fd);
	}
	else
	{
	  session_ent *cur;

	  FD_SET(fd, &ofds);
	  FD_SET(fd, &mfds);

	  for (cur = session_list; cur; cur = cur->next) {
	    struct _list_t *curr = cur->handle->conns;
	    for (curr = cur->handle->conns; curr; curr = (struct _list_t *)curr->next)
	      monitor_conn((migrate_connection *)curr->data, fd);
	  }
	}
      }
      FD_CLR(msock, &rfds);
      selnum--;
    }

    /* Accept new network connections */
    if (FD_ISSET(nsock, &rfds)) {

      session_handle *handle;
      len = sizeof(uaddr);
      if ((fd = accept(nsock, (struct sockaddr *) &uaddr, &len)) < 0 ) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR, "accept: %s",
		strerror(errno));
      } else {
	log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		"Accepted a network connection");

	/* Adjust socket options */
	if (fcntl(fd, F_SETFL, O_NONBLOCK))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR,
		  "Unable to set network socket non-blocking: %s",
		  strerror(errno));
	if (fcntl(fd, F_SETFD, FD_CLOEXEC))
	  log_log(LOG_MOD_MIGRATED, LOG_ERR,
		  "fcntl (FD_CLOEXEC): %s", strerror(errno));

	/* Allocate new session handle */
	if(!(handle = (session_handle *)malloc(sizeof(session_handle)))) {
	  log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		  "malloc: %s", strerror(errno));
	  close(fd);
	} else {

	  /* Set up session handle */
	  init_session(fd, handle);

	  /* Add to session list */
	  set_state(handle, MIGRATE_CONNECTING);
	  add_entry(handle);
	}
      }
      FD_CLR(nsock, &rfds);
      selnum--;
    }

#ifdef HAVE_LIBREADLINE
    /* Handle user input */
    if (FD_ISSET(STDIN_FILENO, &rfds)) {
      rl_callback_read_char();
      FD_CLR(STDIN_FILENO, &rfds);
      selnum--;
    }
#endif
    
    /* Deal with clients */
    for (fd = 0; ((selnum > 0) && (fd < FD_SETSIZE)); fd++) {
      if (FD_ISSET(fd, &rfds)) {

	/* Handle accepting connections */
	if (FD_ISSET(fd, &afds)) {

	  migrate_connection *conn = (migrate_connection *)netfds[fd];
	  struct sockaddr_in addr;
	  int newfd = -1;

	  log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Accepting...");

	  len = sizeof(addr);
	  switch(conn->type) {
	    
	  case SOCK_STREAM:
	    
	    if ((newfd = accept(fd, (struct sockaddr *) &addr, &len)) == -1) {
	      log_log(LOG_MOD_MIGRATED, LOG_ERR, "accept: %s",
		      strerror(errno));
	      goto wrongaddr;
	    }
	    if (fcntl(newfd, F_SETFD, FD_CLOEXEC))
	      log_log(LOG_MOD_MIGRATED, LOG_ERR,
		      "fcntl (FD_CLOEXEC): %s", strerror(errno));
	    break;
	    
	  case SOCK_DGRAM: {
	    char buf[1024];
	    size_t buflen = 1024;
	    int msglen = -1;

	    if ((msglen = recvfrom(fd, buf, buflen, 0,
				   (struct sockaddr *)&addr, &len)) < 0) {
	      log_log(LOG_MOD_MIGRATED, LOG_ERR, "recvfrom: %s",
		      strerror(errno));
	      goto wrongaddr;
	    }
	    if (memcmp(buf, migrate_hello, strlen(migrate_hello) + 1)) {
	      buf[buflen] = '\0';
	      log_log(LOG_MOD_MIGRATED, LOG_INFO,
		      "Bogus UDP %d from %s:%d: %s", fd,
		      inet_ntoa(addr.sin_addr), ntohs(addr.sin_port), buf);
	      goto wrongaddr;
	    }
	    newfd = fd;
	    break;
	  }
	  default:
	    assert(0);
	  }

	  /* Verify this is the right host */	      
	  if(memcmp(&addr.sin_addr,
		    &(((session_handle *)conn->session)->session.paddr.sin_addr),
		    sizeof(addr.sin_addr))) {
	    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
		    "Connecting host %s not peer,",
		    inet_ntoa(addr.sin_addr));
	    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
		    "Expecting %s",
		    inet_ntoa(((session_handle *)conn->session)->session.paddr.sin_addr));
	    if (newfd != -1) close(newfd);
	    goto wrongaddr;
	  }
	    
	  /* Patch into previous connection */
	  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		  "Accepting new migrating connection");
	  /* All Tesla sockets are non-blocking */
	  if (fcntl(newfd, F_SETFL, O_NONBLOCK))
	    log_log(LOG_MOD_MIGRATED, LOG_ERR,
		    "Unable to set network socket non-blocking: %s",
		    strerror(errno));
	  
	  conn->rfd = newfd;
	  len = sizeof(conn->cdaddr);
	  memcpy(&conn->cdaddr, &addr, len);

	  switch (conn->type) {

	  case SOCK_STREAM:
	    if (close(fd))
	      log_log(LOG_MOD_MIGRATED, LOG_ERR,
		      "close: %s", strerror(errno));
	    break;

	  case SOCK_DGRAM:
	    if (connect(fd, (struct sockaddr *)&addr, len))
	      log_log(LOG_MOD_MIGRATED, LOG_ERR,
		      "connect (%d): %s", fd, strerror(errno));
	    break;	    

	  default:
	    break;
	  }

	  /* Notify library */
	  update_library(conn, (session_handle *)conn->session);

	  /* Monitor it */
	  monitor_conn(conn, -1);

	  /* Clean up listening connection */
	  FD_CLR(fd, &afds);
	  FD_CLR(fd, &ofds);
	  netfds[fd] = NULL;
	
	  /* Possibly update state */
	  if((--((session_handle *)conn->session)->ofds) == 0) {
	    set_state((session_handle *)conn->session, MIGRATE_ESTABLISHED);
	    notify_libraries((session_handle *)conn->session);
	  }

	wrongaddr:
	  /* Keep listening for somebody else... */
	}


	/* Handle monitor connections */
	if (FD_ISSET(fd, &mfds)) {
	
	  if((readlen = read(fd, &msg, sizeof(msg))) == -1) {
	    log_log(LOG_MOD_MIGRATED, LOG_ERR,
		    "Unable to read from monitor: %s", strerror(errno));
	  } else if(!readlen) {
	    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
		    "Disconnecting monitor.");
	    monitor_close(fd);
	    close(fd);
	    FD_CLR(fd, &ofds);
	    FD_CLR(fd, &mfds);
	  } else {

	    /* XXX: We really need to do something better */
	    assert(readlen == sizeof(msg));
	    readlen = read(fd, buf, msg.len);
	    if(readlen < 0) {
	      log_log(LOG_MOD_MIGRATED, LOG_CRIT,
		      "monitor read: %s", strerror(errno));
	      break;
	    }
	    assert(readlen == msg.len);
	    recv_monitor(&msg, buf);

	  }

	/* Handle library connections */
	} else if (FD_ISSET(fd, &lfds)) {
	  
	  errno = 0;
	  readlen = read(fd, &msg, sizeof(msg));
	  if (readlen < (int)sizeof(msg)) {
	    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		    "Unable to read from library (%d, expected %d): %s",
		    readlen, sizeof(msg), strerror(errno));
	    library_disconnect(fd);
	    close(fd);
	    FD_CLR(fd, &ofds);
	    FD_CLR(fd, &lfds);
	  } else {

	    assert(readlen == sizeof(msg));
	    if (msg.type > LAST_MSG_TYPE) {
	      log_log(LOG_MOD_MIGRATED, LOG_ERR,
		      "Received bogosity from app");
	      library_disconnect(fd);
	      close(fd);
	      FD_CLR(fd, &ofds);
	      FD_CLR(fd, &lfds);
	    } else {

	      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		      "About to read type %d from %d:%d", msg.type,
		      msg.pid, fd);

	      /* XXX: We really need to do something better */
	      if(msg.type != FD_MSG) {
		if((readlen = read(fd, buf, msg.len)) < 0) {
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			"library read: %s", strerror(errno));
		  exit(-1);
		}
		assert(readlen == msg.len);
	      }
	      
	      switch(msg.type) {
		
	      case SESSION_MSG: {
		session_handle *handle = NULL;
		assert(readlen==sizeof(migrate_session));
		if((handle = recv_session(fd, msg.pid, buf))) {
		  if(handle->pfd != -1) {
		    netfds[handle->pfd] = handle;		  
		  }
		  switch (handle->session.state) {
		  case MIGRATE_NOTSUPPORTED:
		  case MIGRATE_NOTCONNECTED:
		  break;
		  case MIGRATE_CONNECTING:
		    FD_SET(handle->pfd, &cfds);
		    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		    "Adding fd %d to cfds", handle->pfd);
		    /* Fallthrough... */
		  case MIGRATE_ESTABLISHED:
		    FD_SET(handle->pfd, &nfds);
		    /* XXX: Notify library */
		    break;
		  case MIGRATE_FROZEN:
		    /* XXX: Abort our current connection */
		    if(handle->pfd != -1) {
		      log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
			      "Shutting down pfd socket %d", handle->pfd);
		      close(handle->pfd);
		      FD_CLR(handle->pfd, &ofds);
		      FD_CLR(handle->pfd, &nfds);	      
		      netfds[handle->pfd] = NULL;
		      handle->pfd = -1;
		    }
		    break;
		default:
		  /* Nothing to do... */
		  }
		}
		break;
	      }
	      
	      case CONNECTION_MSG: {
		
		session_handle *handle = NULL;
		
		/* Lookup appropriate session */
		log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
			"RECIEVED conn in session %d:",
			(int)((migrate_connection *)buf)->session);
		if(!(handle = conn_session((migrate_connection*)buf))) {
		  log_log(LOG_MOD_MIGRATED, LOG_CRIT,
			  "Unable to find session %d",
			  ((migrate_connection *)buf)->session);
		  /* Just ignore this packet */
		} else {
		  update_conn(handle, fd, (migrate_connection *)buf);
		}
		break;
	      }
	      
	      case CONT_MSG: {
		session_handle *handle = NULL;
		assert(readlen==sizeof(migrate_continuation));
		if((handle = find_session(msg.pid))) { 
		  assert(handle->session.state == MIGRATE_FROZEN);
		  assert(!handle->cont);
		  handle->cont = (migrate_continuation *)
		    malloc(sizeof(migrate_continuation));
		  memcpy(handle->cont, buf, sizeof(migrate_continuation));
		  /* Nuke the fds section, we will rebuild it */
		  handle->cont->fds = NULL;
		  log_log(LOG_MOD_MIGRATED, LOG_INFO,
			  "Processing continuation %u for session %d",
			  handle->cont->cont,msg.pid);
		} else
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "Received cont for bogus session %d", msg.pid);
		break;
	      }

	      case COMPCONT_MSG: {
		session_handle *handle = NULL;
		if((handle = find_session(msg.pid)) && 
		   (handle->session.state == MIGRATE_FROZEN) &&
		   (handle->cont && (handle->cont->flags & M_COMPLETE))) {
		  /* We can hijack the db field since we'll rewrite it again
		     when we restart anyway */
		  handle->cont->db = malloc(readlen+1);
		  memcpy(handle->cont->db, buf, readlen);
		  ((char *)handle->cont->db)[readlen] = '\0';
		  log_log(LOG_MOD_MIGRATED, LOG_INFO,
			  "Complete continuation file %s",
			  handle->cont->db);
		} else
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "Received compcont for bogus session %d", msg.pid);
		break;
	      }

	      case FD_MSG: {
		unsigned int len;
		struct fdhandle_t *fdh = (struct fdhandle_t *)malloc(sizeof(struct fdhandle_t));
		session_handle *handle = NULL;
		fdh->fd = rcvfd(fd, &fdh->fdnum, &len);
		assert(len==sizeof(fdh->fdnum));
		if((handle = find_session(msg.pid)) && 
		   (handle->session.state == MIGRATE_FROZEN) &&
		   handle->cont && (handle->cont->flags & M_COMPLETE)) {
		  _list_add_tail((struct _list_t **)&handle->cont->fds,
				 (void *)fdh);
		  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
			  "Received %d as %d", fdh->fd, fdh->fdnum);
		} else {
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "Received fd for inappropriate session %d", msg.pid);
		  close(fdh->fd);
		  free(fdh);
		}
		break;
	      }

	      default:
	      /* Something random */
		assert(0);
	      }
	    }
	  }
	/* Handle network connections */
	} else if (FD_ISSET(fd, &nfds)) {
	  session_handle *handle = netfds[fd];
	  assert(handle);

	  /* If there's no pending data, read message header */
	  if(!handle->msghdr.len) {
	    errno = 0;
	    if((readlen = read(fd, &handle->msghdr, sizeof(msg))) !=
	       sizeof(handle->msghdr)) {
	      log_log(LOG_MOD_MIGRATED, LOG_INFO,
		      "Unable to read from network: %s", strerror(errno));
	      /* If things appear to have closed normally, close.  Otherwise,
	         if peer just disappeared, we need to suspend. */

	      set_state(handle,
			(readlen ? MIGRATE_FROZEN : MIGRATE_NOTCONNECTED));
	      close_session(handle, 1);
	      log_log(LOG_MOD_MIGRATED, LOG_NOTICE, "%s session %d.",
		      (readlen ? "Freezing" : "Disconnected" ),
		      handle->session.id);
	      notify_libraries(handle);
	      if(handle->lfds == NULL) {
		assert(!readlen);
		remove_ent(handle);
	      }
	      goto noread;
	    } else {
	      /* XXX: We really need to do something better */
	      assert(readlen == sizeof(handle->msghdr));
	      handle->msghdr.len = ntohl(handle->msghdr.len);
	      assert(handle->msghdr.len < BUFLEN);
#ifdef DEBUG
	      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		      "Reading a %d type message of length %d on %d",
		      handle->msghdr.type, handle->msghdr.len, fd);
#endif
	    }
	  }
#ifdef DEBUG
	  log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		  "Reading %d, %d already received",
		   (handle->msghdr.len - handle->buflen), handle->buflen);
#endif
	  if((readlen = read(fd, &handle->buf[handle->buflen],
			 (handle->msghdr.len - handle->buflen))) == -1) {
	    if(errno != EAGAIN) {
	      log_log(LOG_MOD_MIGRATED, LOG_ERR,
		      "network read: %s", strerror(errno));
	      assert(0);
	    }
	    goto noread;
	  }
	  if(!readlen) {
	    log_log(LOG_MOD_MIGRATED, LOG_ERR,
		    "EOF received?");
	    exit(1);
	  }

	  handle->buflen += readlen;
	    
	  if(handle->buflen == handle->msghdr.len) {
	    
	    switch(handle->msghdr.type) {
	      
	    case SESSION_EST:

	      /* Ensure we're still connecting */
	      assert(handle->session.state == MIGRATE_CONNECTING);

	      if (!recv_net(handle)) {

		/* Send connection info */
		session_update_conns(handle);
	      }
	      break;

	    case SESSION_RESUME_REQUEST:

	      if (session_crypto_challenge(handle)||session_sendmsg(handle)) {
		log_log(LOG_MOD_CRYPTO, LOG_ERR,
			"Unable to generate and send challenge");
		remove_ent(close_session(handle, 1));
	      }
	      break;

	    case SESSION_RESUME_CHALLENGE:

	      /* Make sure we're in connecting state */
	      assert(handle->session.state == MIGRATE_CONNECTING);

	      session_crypto_auth(handle, handle->buf, handle->buflen);
	      if(session_sendmsg(handle))
		log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
			"Ignoring invalid session resumption request.");
	      else {
		log_log(LOG_MOD_MIGRATED, LOG_INFO,
			"Challenge decrypted.");
		set_state(handle, MIGRATE_LMIGRATING);
		notify_libraries(handle);
	      }
	      break;

	    case SESSION_RESUME_RESPONSE: {
	      session_handle *oldhandle;

	      /* Make sure we're in connecting state */
	      assert(handle->session.state == MIGRATE_CONNECTING);

	      if (!(oldhandle = session_crypto_finalize(handle))) {
		log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
			"Invalid response to challenge.");
		/* Don't free shared crypto keys */
		handle->key = NULL;
		remove_ent(close_session(handle, 1));
	      } else {
		log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
			"Successfully migrated %d",
			handle->session.id);
		session_merge(handle, oldhandle, 1);
    		set_state(handle, MIGRATE_PMIGRATING);
		notify_libraries(handle);

		/* Update peer */
		session_update_sess(handle);

		/* Prepare to migrate connections */
		mcp(handle);
	      }
	      break;
	    }

	    case SESSION_MSG:

	      /* Session update from the other side.  Get new pid. */
	      handle->session.pid = ((migrate_session *)handle->buf)->id;
	      log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		      "Updating session (%d:%d) in state %d to %d",
		      handle->session.id, handle->session.pid,
		      handle->session.state,
		      ((migrate_session *)handle->buf)->state);
	      break;

	    case CONNECTION_MSG: {

	      session_handle *newhandle;

	      assert(ntohl((int)((migrate_connection *)handle->buf)->session)
		     == handle->session.id);
	      
	      switch(handle->session.state) {
	      
	      case MIGRATE_ESTABLISHED:

		/* Find connection in anonymous session */
              if (!(newhandle = session_port(
				   (migrate_connection *)handle->buf))) {
                /* Add it as an anonymous connection */
                log_log(LOG_MOD_MIGRATED, LOG_ERR,
                        "Can't find connection--going anon");
                anon_conn(handle, (migrate_connection *)handle->buf);
              } else {
                if(handle != newhandle) {
		  switch(newhandle->session.state) {
		  case MIGRATE_CONNECTING:
		    /* Double connect race.  Don't merge. */
		    log_log(LOG_MOD_MIGRATED, LOG_ALERT,
			    "Double connect race %d:%d and %d:%d",
			    handle->session.id, handle->session.pid,
			    newhandle->session.id, newhandle->session.pid);
		    exit(-1);
		  case MIGRATE_NOTCONNECTED:
		    /* Merge sessions together */
		    log_log(LOG_MOD_MIGRATED, LOG_INFO,
			    "Merged session %d:%d and %d:%d",
			    handle->session.id, handle->session.pid,
			    newhandle->session.id, newhandle->session.pid);
		    session_merge(handle, newhandle, 0);
		    set_state(handle, MIGRATE_ESTABLISHED); 
		    notify_libraries(handle);
		    break;
		  case MIGRATE_ESTABLISHED:
		    /* We've got a double-connect.  Pick one. */
		    log_log(LOG_MOD_MIGRATED, LOG_INFO,
			    "Double-connect session %d:%d and %d:%d",
			    handle->session.id, handle->session.pid,
			    newhandle->session.id, newhandle->session.pid);
		    session_merge(handle, newhandle, 0);
		    break;
		  default:
		    log_log(LOG_MOD_MIGRATED, LOG_ERR,
			    "Merging session in state: %s",
			    migstate2ascii[newhandle->session.state]);
		    exit(-1);
		  }
		}
		/* Update peer */
		session_update_sess(handle);  
	      }
	      break;
	      
	      case MIGRATE_LMIGRATING: {

		int newsock = -1;
		migrate_connection *conn = NULL;
		
		/* XXX: This has problems if there are two connections
		 * both going to the same remote port */

		/* First compare exact local and remote addresses */
		if(!(conn = _list_find(handle->conns, handle->buf,
				       conns_pequal)) &&
		   /* Fall back to just remote address in case of NAT*/
		   (!(conn = _list_find(handle->conns, handle->buf,
					conns_pNATequal))))
		  
		  /* Fall back to unbound remote */
		  if(!(conn = _list_find(handle->conns, handle->buf,
					 conns_unbound))) 
		    {
		      /* XXX: Huh? */
		      log_log(LOG_MOD_MIGRATED, LOG_CRIT,
			      "Can't find requested connection in session");
		      assert(0);
		    }
		if ((newsock = migrate_conn(conn, handle)) != -1) {
		  /* Record the outstanding connection */
		  ++handle->ofds;
		  FD_SET(newsock, &cfds);
		  FD_SET(newsock, &tfds);
		  netfds[newsock] = (session_handle *)conn;
		}		
		break;
	      }
	      
	      default:
		/* XXX: HUH? */
		assert(0);
	      }
	      break;
	    }
	    
	    default:
	      log_log(LOG_MOD_MIGRATED, LOG_DEBUG, "Fubar mesg on network");

	      /* Something random */
	      assert(0);
	    }
	    memset(&handle->msghdr, 0, sizeof(handle->msghdr));
	    handle->buflen = 0;
	  }
	noread:
	}
	FD_CLR(fd, &rfds);
	selnum--;
      }
    }
 

    /* Deal with connecting network connections */
    for (fd = 0; ((selnum > 0) && (fd < FD_SETSIZE)); fd++) {
      if (FD_ISSET(fd, &wfds)) {
	
	if (FD_ISSET(fd, &cfds)) {
	  len = sizeof(err);
	  if(getsockopt(fd, SOL_SOCKET, SO_ERROR, &err, &len)) {
	    /* XXX: What do we do? */
	    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
		    "Failed getsockopt: %s", strerror(errno));
	    assert(0);
	  }

	  FD_CLR(fd, &cfds);
	  if(err) {
	    /* The other side lacks Migrate support */
	    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
		    "connect: %s", strerror(err));
	    close(fd);
	    FD_CLR(fd, &nfds);
	    assert(netfds[fd]);
	    
	    if(FD_ISSET(fd, &tfds)) {
	      /* Transport connection failed to migrate */ 
	      migrate_connection *conn = (migrate_connection *)netfds[fd];
	      session_handle *handle = (session_handle *)conn->session;
	      FD_CLR(fd, &tfds);
	      log_log(LOG_MOD_MIGRATED, LOG_INFO,
		      "Connection %d on session %d failed",
		      conn->fd, handle->session.id);
	      --handle->ofds;
	      assert(handle->session.state == MIGRATE_LMIGRATING);
	      set_state(close_session(handle, 1), MIGRATE_LOST);
	      resolve_session(handle);
	      notify_libraries(handle);
	    } else {
	      log_log(LOG_MOD_MIGRATED, LOG_INFO,
		      "Unable to contact remote Migrated on %s",
		      inet_ntoa(netfds[fd]->session.paddr.sin_addr));
	      switch(netfds[fd]->session.state) {
	      case MIGRATE_CONNECTING:
		set_state(close_session(netfds[fd], 1), MIGRATE_NOTSUPPORTED);
		notify_libraries(netfds[fd]);
		break;
	      default:
		/* We shouldn't be connecting in another state */
		log_log(LOG_MOD_MIGRATED, LOG_CRIT,
			"Connect failed for session %d in state %s",
			netfds[fd]->session.id,
			migstate2ascii[netfds[fd]->session.state]);
		assert(0);
	      }
	    }
	  } else {

	    FD_SET(fd, &ofds);
	    
	    assert(netfds[fd]);

	    if(FD_ISSET(fd, &tfds)) {
	      migrate_connection *conn = (migrate_connection *)netfds[fd];
	      FD_CLR(fd, &tfds);
	      
	      /* Check to make sure something didn't happen in the mean time */
	      if(((session_handle *)conn->session)->session.state !=
		 MIGRATE_LMIGRATING) {
		log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
			"Reconnected connection for session in invalid state");
	      }	else {
	      
		log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
			"Reconnected migrating connection");
		conn->rfd = fd;

		/* Get current local port */
		len = sizeof(struct sockaddr_in);
		if(getsockname(fd,
			       (struct sockaddr *)&conn->csaddr,
			       &len))
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "getsockname: %s", strerror(errno));
		
		/* All TESLA sockets are non-blocking */
		if (fcntl(fd, F_SETFL, 0))
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "Unable to set network socket non-blocking: %s",
			  strerror(errno));
		
		/* Notify library */
		update_library(conn, (session_handle *)conn->session);
		
		/* Monitor it */
		monitor_conn(conn, -1);
		
		/* Possibly update state */
		if((--((session_handle *)conn->session)->ofds) == 0) {
		  set_state((session_handle *)conn->session,
			     MIGRATE_ESTABLISHED);
		  notify_libraries((session_handle *)conn->session);
		}
	      }
	      
	      FD_CLR(fd, &ofds);

	      netfds[fd] = NULL;

	    } else {
	    switch(netfds[fd]->session.state) {
	      
	    case MIGRATE_CONNECTING:

	      if(netfds[fd]->key) {
		
		/* We're resuming a previous session */
		log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
			"Requesting migrate of session %d",
			netfds[fd]->session.id);
		
		if ((err = session_crypto_resume(netfds[fd])))
		  log_log(LOG_MOD_CRYPTO, LOG_ERR,
			  "Problem generating request message");
		if ((err = session_sendmsg(netfds[fd]))) {
		  log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
			  "Unable to send session migrate request");
		  close(netfds[fd]->pfd);
		  FD_CLR(fd, &ofds);
		  FD_CLR(fd, &nfds);	      
		  netfds[fd]->pfd = -1;
		  set_state(netfds[fd], MIGRATE_NOTSUPPORTED);
		  netfds[fd] = NULL;
		}
	      } else {
	      
		/* We're connecting a whole new session */

		/* Now that we're connected, get default buffer size */
		netfds[fd]->lbufsize = rcvbufsize(fd);

		/* Attempt crypto initialization */
		session_crypto_init(netfds[fd]);
		if((err = session_sendmsg(netfds[fd]))) {
		  log_log(LOG_MOD_MIGRATED, LOG_ERR,
			  "Can't negotiate keys: %s", strerror(err));
		  set_state(close_session(netfds[fd],1),MIGRATE_NOTSUPPORTED);
		  notify_libraries(netfds[fd]);
		  /* XXX: Notify library */
		  /* return MIGRATE_NOTSUPPORTED; */
		}
	      }
	      break;
	      
	    default:
	      /* XXX: Huh? */
	      log_log(LOG_MOD_MIGRATED, LOG_EMERG,
		      "Connecting socket %d did something weird", fd);
	      assert(0);
	    }
	    }
	  }
	} else {
	  
	  /* We shouldn't be selecting on writes... */
	  log_log(LOG_MOD_MIGRATED, LOG_EMERG,
		  "Socket %d ready for writing but not connection?",fd);
	  /* assert(0); */
	  
	}

	FD_CLR(fd, &wfds);
	selnum--;
      }
    }
    
    /* We should have now handled everything */
    if (selnum)
      log_log(LOG_MOD_MIGRATED, LOG_EMERG,
	      "Selnum is %d", selnum);
    assert(selnum==0);

    /* Reset file descriptors */
    memcpy(&rfds, &ofds, sizeof(fd_set));
    memcpy(&wfds, &cfds, sizeof(fd_set));
    FD_SET(nsock, &rfds);
    FD_SET(lsock, &rfds);    
    FD_SET(msock, &rfds);    
    if (interactive) FD_SET(STDIN_FILENO, &rfds);
  }
  
 shutdown:

#ifdef HAVE_LIBREADLINE
  rl_callback_handler_remove();
#endif

  if (fd != -1) close(fd);

  migrated_cleanup(0);

  /* Never reached */
  return 0;
}


/* Trade out old session fd for new, connecting one */
int
swap_fds(session_handle *handle, int newfd)
{
  assert(newfd != -1);
  if(handle->pfd != -1) {
    /* Do NOT close previous file descriptor, but ignore it. */
    netfds[handle->pfd] = NULL;
    FD_CLR(handle->pfd, &cfds);
    FD_CLR(handle->pfd, &nfds);
    FD_CLR(handle->pfd, &ofds);
    handle->pfd = -1;
  }
  netfds[newfd] = handle;
  FD_SET(newfd, &nfds);
  FD_SET(newfd, &cfds);
  handle->pfd = newfd;

  return 0;
}


int
main(int argc, char *argv[])
{
  int c;
  int err = 0;
  char *portstr = getenv(ENV_MIGRATE_PORT);

#ifdef HAVE_GETOPT_LONG
  int option_index = 0;
  while((c = getopt_long(argc, argv, "d:hf:ip:v", long_options,
			 &option_index)) != -1) {
#else
  while((c = getopt(argc, argv, "d:hf:ip:v")) != -1) {
#endif
    switch (c) {
    case 'd':
      debuglevel = atoi(optarg);
      break;
    case 'f':
      monitor_policy_file = optarg;
      break;
    case 'h':
      usage(stdout);
      exit(0);
      break;
#ifdef HAVE_LIBREADLINE
    case 'i':
      interactive = 1;
      break;
#endif
    case 'p':
      portstr = optarg;
      break;
    case 'v':
      version(stdout, NULL);
      exit(0);
      break;
    default:
      usage(stderr);
      exit(1);
    }
  }

  if (argc - optind > 0) {
    usage(stderr);
    exit(1);
  }
  
  /* Kick off Bev's logging subsystem */
  log_init("migrated", NULL, LOG_METH_STDERR, debuglevel);
  
  if (portstr) {
    char *end;
    long newport = strtol(portstr, &end, 0);
    if (*end || newport < 1 || newport > 65535) {
      log_log(LOG_MOD_MIGRATED, LOG_WARNING,
	      "Invalid port specified \"%s\"", portstr);
    } else {
      migrated_port = newport;
    }
  }
  
  /* Kick off the monitor */
  monitor_init();
    
  /* Make sure we can grab named pipes */
  if (unlink(lpipename) < 0) 
    if (errno != ENOENT) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to create pipe %s: %s",
	      lpipename, strerror(errno));
      exit(1);
    }
  if (unlink(mpipename) < 0) 
    if (errno != ENOENT) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to create pipe %s: %s",
	      mpipename, strerror(errno));
      exit(1);
    }

  /* XXX: Don't close stderr for debugging purposes */
  if (!interactive) {
    if (daemon(0,1) == -1) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR, "Unable to daemonize");
    }
    close(0);
    close(1);
  }

  /* Initialize session hash tables */
  session_list = NULL;

  /* Initialized crypto library */
  if ((err=crypto_init())) {
    log_log(LOG_MOD_MIGRATED, LOG_CRIT,
	    "Crypto initialization failed: %s", strerror(err));
    exit(1);
  }

  /* Clean up nicely on exit */
  signal(SIGINT, migrated_cleanup);
  signal(SIGQUIT, migrated_cleanup);
  signal(SIGKILL, migrated_cleanup);
  signal(SIGTERM, migrated_cleanup);
  
  /* Never returns */
  return manage_conns();
}


int
mcp(session_handle *handle)
{
  int csock = -1;
  struct _list_t *curr;

  if(handle->session.state == MIGRATE_FROZEN)
    return (handle->mcp_pending = 1);

  for(curr = handle->conns; curr; curr = (struct _list_t *)curr->next) {
    if ((csock = migrate_conn_prepare((migrate_connection *)curr->data,
				      handle)) != -1) {
      /* Record the outstanding connection */
      ++handle->ofds;
      FD_SET(csock, &afds);
      FD_SET(csock, &ofds);
      netfds[csock] = (session_handle *)curr->data;
    }
  }
  handle->mcp_pending = 0;
  return 0;
}
