/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: migrated.h,v 1.44 2002/10/10 21:50:21 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_ASSERT_H
# include <assert.h>
#endif

#include <migrate_internal.h>
#include <list.h>
#include "log.h"

#define BUFLEN 1024
#define DEFAULT_RCVBUF 87380         /* Standard Linux > FreeBSD */

extern const char *migstate2ascii[];
extern unsigned int sessionno;

typedef struct {
  migrate_session       session;
  int                   pid;         /* Process id */
  uid_t                 uid;         /* User who owns this session */
  int                   pfd;
  struct fdhandle      *lfds;
  struct _list_t       *conns;
  migrate_connection   *anon;        /* One anon per session */
  void *                key;         /* Crypto stuff */
  void *                nonce; 
  char                 *challenge;   /* Random challenge */
  int                   buflen;  
  mmsghdr               msghdr;
  migrate_continuation *cont;
  struct _list_t       *update_pending;
  int                   mcp_pending;
  int                   ofds;        /* Number of outstanding conns */
  size_t                lbufsize;    /* Default local RCVBUF size */
  char                  buf[BUFLEN]; /* Message assembly area */
} session_handle;


struct session_ent_t;
typedef struct session_ent_t session_ent;

struct session_ent_t {
  session_handle *handle;
  session_ent *next;
};

struct fdhandle_t {
  int fd;
  int fdnum;
};

extern session_ent *session_list;
extern int migrated_done;

extern char migrate_hello[];
extern int migrated_port;

extern char *monitor_policy_file;

/* From netio.c */

int session_sendmsg(session_handle *handle);
int recv_net(session_handle *handle);
int session_update_conn(migrate_connection *conn);
int session_update_conns(session_handle *handle);
int session_update_sess(session_handle *handle);
int migrate_conn(migrate_connection *conn, session_handle *handle);
int migrate_conn_prepare(migrate_connection *conn, session_handle *handle);

/* From crypto.c */

int crypto_init(void);
void * session_handshake(session_handle *handle);
void * finish_handshake(session_handle *handle);
int session_crypto_resume(session_handle *handle);
int session_crypto_init(session_handle *handle);
int session_crypto_challenge(session_handle *handle);
int session_crypto_auth(session_handle *handle, char *buf, int len);
session_handle *session_crypto_finalize(session_handle *handle);

/* From interactive.c */

void usage(FILE *fd);
void version(FILE *fd, char *);
void interactive_handler(char *line);
char ** interactive_completion(char *text, int start, int end);

/* From library.c */

int conns_lequal(const void *cp1, const void *cp2);
int conns_pequal(const void *lcp, const void *pcp);
int conns_pNATequal(const void *lcp, const void *pcp);
int conns_unbound(const void *cp, const void *pcp);

session_handle *session_port(migrate_connection *conn);
session_handle *conn_session(migrate_connection *conn);
session_handle * recv_session(int fd, pid_t pid, void *buf);
migrate_state freeze_session(session_handle *session);
migrate_state defrost_session(session_handle *session);
migrate_state resolve_session(session_handle *session);
migrate_state mig_session(session_handle *session, struct sockaddr_in *addr);
session_handle *session_merge(session_handle *newh, session_handle *oldh,
			      int full);
int anon_conn(session_handle *handle, migrate_connection *conn);
int update_conn(session_handle *handle, int fd, void *buf);
int update_library(migrate_connection *conn, session_handle *handle);
int notify_libraries(session_handle *handle);
int library_init(session_handle *handle);
int library_disconnect(int fd);

/* From monitor.c */

int monitor_init();
int recv_monitor(mmsghdr *msg, char *buf);
int monitor_conn(migrate_connection *conn, int fd);
int monitor_open(int fd);
void monitor_close(int fd);

/* From migrated.c */

unsigned int new_sessionno(void);
session_ent *add_entry(session_handle *handle);
session_ent *remove_ent(session_handle *handle);
migrate_state set_state(session_handle *handle, migrate_state state);
session_handle * close_session(session_handle *handle, int force);
int swap_fds(session_handle *handle, int newfd);
session_handle *find_session(int sessionid);
int mcp(session_handle *handle);

/* From cont.c */

pid_t complete_cont(session_handle *handle);

