char *monitor_init_tcl = ""
    "\n"
    "###########\n"
    "###\n"
    "### Migrate Session Layer\n"
    "###\n"
    "### Alex C. Snoeren <snoeren@lcs.mit.edu>\n"
    "###\n"
    "### Copyright (c) 2001 Massachusetts Institute of Technology.\n"
    "###\n"
    "### This software is being provided by the copyright holders under the GNU\n"
    "### General Public License, either version 2 or, at your discretion, any later\n"
    "### version. For more information, see the `COPYING' file in the source\n"
    "### distribution.\n"
    "###\n"
    "### $Id: monitor-init.tcl,v 1.3 2001/12/03 20:01:16 jsalz Exp $\n"
    "###\n"
    "###########\n"
    "\n"
    "### N.B.: This file is not configurable. You should be editing monitor-rules.tcl.\n"
    "\n"
    "#####\n"
    "#\n"
    "# Global variables:\n"
    "#\n"
    "# services: An associative array mapping service names to port\n"
    "#   numbers. E.g., $services(tcp/http) is 80; $services(udp/talk)\n"
    "#   is 517.\n"
    "#\n"
    "# conn: An associative array containing connection properties. Set\n"
    "#   when monitor-rules.tcl is invoked.\n"
    "#\n"
    "#     $conn(l-port) = local port\n"
    "#     $conn(r-port) = remote port\n"
    "#     $conn(l-addr) = local address (as 8 hexits)\n"
    "#     $conn(r-addr) = remote address (as 8 hexits)\n"
    "#\n"
    "# interfaces: An array wherein each element is a (label,\n"
    "#   address, mask) tuple. E.g., $interfaces(eth0 121F0047 FFFFFF00)\n"
    "#   might be set to indicate an address of 18.31.0.71 in a class-C\n"
    "#   network.\n"
    "#\n"
    "#####\n"
    "\n"
    "global debug\n"
    "global env\n"
    "set debug [expr { [info exists env(MIGMONITOR_DEBUG)] && \\\n"
    "\t![string equal $env(MIGMONITOR_DEBUG) 0] && ![string equal $env(MIGMONITOR_DEBUG) \"\"] }]\n"
    "\n"
    "proc debug { msg } {\n"
    "    global debug\n"
    "    if { $debug } {\n"
    "\tputs \"*** (monitor) $msg\"\n"
    "    }\n"
    "}\n"
    "\n"
    "proc clear-interfaces {} {\n"
    "    global interfaces\n"
    "    set interfaces [list]\n"
    "    debug \"clearing interface list in Tcl interpreter\"\n"
    "}\n"
    "\n"
    "# Invoked when an interface goes up or down. State is 1 for up,\n"
    "# or 0 for down.\n"
    "proc interface { label state addr mask } {\n"
    "    debug \"interface [list $label $state $addr $mask]\"\n"
    "    global interfaces\n"
    "\n"
    "    set elt \"$label $addr $mask\"\n"
    "    set index [lsearch -exact $interfaces $elt]\n"
    "\n"
    "    if { $state } {\n"
    "\tif { $index < 0 } {\n"
    "\t    lappend interfaces $elt\n"
    "\t}\n"
    "    } else {\n"
    "\tif { $index >= 0 } {\n"
    "\t    set interfaces [lreplace $interfaces $index $index]\n"
    "\t}\n"
    "    }\n"
    "\n"
    "    debug \" - now interfaces is $interfaces\"\n"
    "}\n"
    "\n"
    "proc score-interface { ifglob score } {\n"
    "    global scores\n"
    "    global interfaces\n"
    "\n"
    "    foreach if $interfaces {\n"
    "\tset label [lindex $if 0]\n"
    "\tset addr [lindex $if 1]\n"
    "\tset mask [lindex $if 2]\n"
    "\t\n"
    "\tif { [string match $ifglob $label] } {\n"
    "\t    debug \"- scoring $if as $score\"\n"
    "\t    set scores($if) $score\n"
    "\t}\n"
    "    }\n"
    "}\n"
    "\n"
    "proc warn { str } {\n"
    "    debug \"Warning: $str\"\n"
    "}\n"
    "\n"
    "proc mask-parse { str } {\n"
    "    if { ![regexp {^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)(/([0-9]+))?$} $str \\\n"
    "\t    {} a b c d {} bits] } {\n"
    "\twarn \"Illegal IP address \\\"$str\\\"\"\n"
    "\treturn {}\n"
    "    }\n"
    "    \n"
    "    if { ![info exists bits] } {\n"
    "\tset bits 32\n"
    "    }\n"
    "    if { $bits > 32 } {\n"
    "\twarn \"Invalid number of bits $bits in IP address \\\"$str\\\"\"\n"
    "    }\n"
    "\n"
    "    set mask [expr { (~0) << (32 - $bits) }]\n"
    "    set addr [expr { $mask & (($a << 24) + ($b << 16) + ($c << 8) + $d) }]\n"
    "\n"
    "    return [list $addr $mask]\n"
    "}\n"
    "\n"
    "proc mask-match { ip mask } {\n"
    "    if { ![scan $ip \"%x\" ipint] } {\n"
    "\twarn \"Invalid IP address \\\"$ip\\\" in mask-match\"\n"
    "\treturn 0\n"
    "    }\n"
    "\n"
    "    return [expr { ($ipint & [lindex $mask 1]) == [lindex $mask 0] }]\n"
    "}\n"
    "\n"
    "proc conn { prop } {\n"
    "    global conn\n"
    "    if { ![info exists $conn($prop)] } {\n"
    "\twarn \"Invalid property \\\"$prop\\\"\"\n"
    "\treturn \"\"\n"
    "    }\n"
    "    return $conn($prop)\n"
    "}\n"
    "\n"
    "proc on { prop value args } {\n"
    "    global conn\n"
    "    global services\n"
    "\n"
    "    set match 0\n"
    "    set proto $conn(proto)\n"
    "\n"
    "    set lport [expr { [string equal $prop \"port\"] || [string equal $prop \"l-port\"] }]\n"
    "    set pport [expr { [string equal $prop \"port\"] || [string equal $prop \"p-port\"] }]\n"
    "\n"
    "    if { [string equal $prop \"eval\"] } {\n"
    "\tset match [uplevel expr $value]\n"
    "    } elseif { [string equal $prop \"proto\"] } {\n"
    "\tset match [string equal $conn(proto) $value]\n"
    "    } elseif { $lport || $pport } {\n"
    "\tforeach p [split $value \",\"] {\n"
    "\t    if { [regexp {^([0-9]+)(-([0-9]+))?$} $p \"\" p1 \"\" p2] } {\n"
    "\t\t# It's either a port number or a range of port numbers.\n"
    "\t\tif { $p2 == \"\" } {\n"
    "\t\t    set p2 $p1\n"
    "\t\t}\n"
    "\t    } elseif { [info exists services($proto/$p)] } {\n"
    "\t\t# Not a port number or range, but it's listed in services.\n"
    "\t\tset p1 $services($proto/$p)\n"
    "\t\tset p2 $p1\n"
    "\t    } else {\n"
    "\t\t# Who the heck knows?\n"
    "\t\twarn \"Unknown service $proto/$p\"\n"
    "\t\tcontinue\n"
    "\t    }\n"
    "\n"
    "\t    if { ($lport && ($conn(l-port) >= $p1 && $conn(l-port) <= $p2)) || \\\n"
    "\t\t    ($pport && ($conn(p-port) >= $p1 && $conn(p-port) <= $p2)) } {\n"
    "\t\tset match 1\n"
    "\t\tbreak\n"
    "\t    }\n"
    "\t}\n"
    "    } elseif { [string equal $prop \"remote-ip\"] } {\n"
    "\tset mask [mask-parse $value]\n"
    "\tset match [mask-match $conn(l-addr) $mask]\n"
    "    } else {\n"
    "\twarn \"Unknown property $prop\"\n"
    "    }\n"
    "\n"
    "    if { $match } {\n"
    "\tdebug \"- ON $prop $value matched\"\n"
    "\tif { [llength $args] == 1 } {\n"
    "\t    uplevel 1 eval $args\n"
    "\t} else {\n"
    "\t    uplevel 1 $args\n"
    "\t}\n"
    "    } else {\n"
    "\tdebug \"- ON $prop $value didn't match\"\n"
    "    }\n"
    "}\n"
    "\n"
    "proc score-init {} {\n"
    "    global conn\n"
    "    global scores\n"
    "\n"
    "    if { [info exists scores] } { unset scores }\n"
    "    if { [info exists conn] } { unset conn }\n"
    "\n"
    "    array set scores {}\n"
    "    set conn(migrate) threshold\n"
    "}\n"
    "\n"
    "proc migrate { when } {\n"
    "    global conn\n"
    "    set conn(migrate) $when\n"
    "    debug \"MIGRATE $when\"\n"
    "}\n"
    "\n"
    "proc migrate-decision {} {\n"
    "    global conn\n"
    "    global scores\n"
    "\n"
    "    if { [info exists conn(migrate)] } {\n"
    "\tset migrate $conn(migrate)\n"
    "    } else {\n"
    "\tset migrate \"threshold\"\n"
    "    }\n"
    "\n"
    "    debug \"Making migration decision...\"\n"
    "\n"
    "    if { [string equal $migrate \"never\"] } {\n"
    "\tdebug \" - never migrate\"\n"
    "\treturn \"\"\n"
    "    }\n"
    "\n"
    "    # Figure out current score and highest score\n"
    "    foreach { if score } [array get scores] {\n"
    "\tset label [lindex $if 0]\n"
    "\tset addr [lindex $if 1]\n"
    "\tset mask [lindex $if 2]\n"
    "\n"
    "\tif { [string equal $addr $conn(l-addr)] } {\n"
    "\t    set current_score $score\n"
    "\t    set current_if $if\n"
    "\t}\n"
    "\tif { ![info exists best_score] || $score > $best_score } {\n"
    "\t    set best_score $score\n"
    "\t    set best_if $if\n"
    "\t}\n"
    "    }\n"
    "\n"
    "    debug \" - variables:\"\n"
    "    foreach s { current_score current_if best_score best_if } {\n"
    "\tdebug \"   - $s is [expr { [info exists $s] ? [set $s] : \"(none)\" }]\"\n"
    "    }\n"
    "\n"
    "    if { [string equal $migrate \"if-dead\"] } {\n"
    "\tif { ![info exists current_score] && [info exists best_score] } {\n"
    "\t    debug \" - it's dead; migrate to $best_if\"\n"
    "\t    return $best_if\n"
    "\t} else {\n"
    "\t    debug \" - it's not dead; not migrating\"\n"
    "\t    return \"\"\n"
    "\t}\n"
    "    }\n"
    "\n"
    "    if { [string equal $migrate \"threshold\"] } {\n"
    "\tif { [info exists best_score] && \\\n"
    "\t\t(![info exists current_score] || double($best_score) / double($current_score) >= 1.25) } {\n"
    "\n"
    "\t    set current_pretty [expr { [info exists current_score] ? $current_score : \"(none)\" }]\n"
    "\t    debug \" - $best_score beats current score of $current_pretty; migrating to $best_if\"\n"
    "\t    return $best_if\n"
    "\t} else {\n"
    "\t    debug \" - doesn't beat threshold; not migrating\"\n"
    "\t    return \"\"\n"
    "\t}\n"
    "    }\n"
    "\n"
    "    debug \" - not migrating\"\n"
    "    return \"\"\n"
    "}\n"
    "\n"
    "proc monitor-policy { scr } {\n"
    "    global script\n"
    "    set script [info script]\n"
    "\n"
    "    proc monitor-policy-real {} $scr\n"
    "}\n"
    "\n"
    "proc monitor-policy-eval {} {\n"
    "    global script\n"
    "    global debug\n"
    "    global conn\n"
    "\n"
    "    if { $debug } {\n"
    "\tdebug \"POLICY: ($conn(proto) $conn(l-addr):$conn(l-port)->$conn(p-addr):$conn(p-port))\"\n"
    "\n"
    "\t# When debugging, reload monitor-init.tcl and monitor-policy.tcl every time to catch\n"
    "\t# updates. This is a semi-hack to make development easier.\n"
    "\n"
    "\tcatch { source \"monitor-init.tcl\" }\n"
    "\tcatch { source $script }\n"
    "    }\n"
    "\n"
    "    monitor-policy-real\n"
    "}    \n"
    "\n"
    "global inited\n"
    "if { ![info exists inited] } {\n"
    "    set inited 1\n"
    "    clear-interfaces\n"
    "}\n"
    ;
