
###########
###
### Migrate Session Layer
###
### Alex C. Snoeren <snoeren@lcs.mit.edu>
###
### Copyright (c) 2001 Massachusetts Institute of Technology.
###
### This software is being provided by the copyright holders under the GNU
### General Public License, either version 2 or, at your discretion, any later
### version. For more information, see the `COPYING' file in the source
### distribution.
###
### $Id: monitor-init.tcl,v 1.3 2001/12/03 20:01:16 jsalz Exp $
###
###########

### N.B.: This file is not configurable. You should be editing monitor-rules.tcl.

#####
#
# Global variables:
#
# services: An associative array mapping service names to port
#   numbers. E.g., $services(tcp/http) is 80; $services(udp/talk)
#   is 517.
#
# conn: An associative array containing connection properties. Set
#   when monitor-rules.tcl is invoked.
#
#     $conn(l-port) = local port
#     $conn(r-port) = remote port
#     $conn(l-addr) = local address (as 8 hexits)
#     $conn(r-addr) = remote address (as 8 hexits)
#
# interfaces: An array wherein each element is a (label,
#   address, mask) tuple. E.g., $interfaces(eth0 121F0047 FFFFFF00)
#   might be set to indicate an address of 18.31.0.71 in a class-C
#   network.
#
#####

global debug
global env
set debug [expr { [info exists env(MIGMONITOR_DEBUG)] && \
	![string equal $env(MIGMONITOR_DEBUG) 0] && ![string equal $env(MIGMONITOR_DEBUG) ""] }]

proc debug { msg } {
    global debug
    if { $debug } {
	puts "*** (monitor) $msg"
    }
}

proc clear-interfaces {} {
    global interfaces
    set interfaces [list]
    debug "clearing interface list in Tcl interpreter"
}

# Invoked when an interface goes up or down. State is 1 for up,
# or 0 for down.
proc interface { label state addr mask } {
    debug "interface [list $label $state $addr $mask]"
    global interfaces

    set elt "$label $addr $mask"
    set index [lsearch -exact $interfaces $elt]

    if { $state } {
	if { $index < 0 } {
	    lappend interfaces $elt
	}
    } else {
	if { $index >= 0 } {
	    set interfaces [lreplace $interfaces $index $index]
	}
    }

    debug " - now interfaces is $interfaces"
}

proc score-interface { ifglob score } {
    global scores
    global interfaces

    foreach if $interfaces {
	set label [lindex $if 0]
	set addr [lindex $if 1]
	set mask [lindex $if 2]
	
	if { [string match $ifglob $label] } {
	    debug "- scoring $if as $score"
	    set scores($if) $score
	}
    }
}

proc warn { str } {
    debug "Warning: $str"
}

proc mask-parse { str } {
    if { ![regexp {^([0-9]+)\.([0-9]+)\.([0-9]+)\.([0-9]+)(/([0-9]+))?$} $str \
	    {} a b c d {} bits] } {
	warn "Illegal IP address \"$str\""
	return {}
    }
    
    if { ![info exists bits] } {
	set bits 32
    }
    if { $bits > 32 } {
	warn "Invalid number of bits $bits in IP address \"$str\""
    }

    set mask [expr { (~0) << (32 - $bits) }]
    set addr [expr { $mask & (($a << 24) + ($b << 16) + ($c << 8) + $d) }]

    return [list $addr $mask]
}

proc mask-match { ip mask } {
    if { ![scan $ip "%x" ipint] } {
	warn "Invalid IP address \"$ip\" in mask-match"
	return 0
    }

    return [expr { ($ipint & [lindex $mask 1]) == [lindex $mask 0] }]
}

proc conn { prop } {
    global conn
    if { ![info exists $conn($prop)] } {
	warn "Invalid property \"$prop\""
	return ""
    }
    return $conn($prop)
}

proc on { prop value args } {
    global conn
    global services

    set match 0
    set proto $conn(proto)

    set lport [expr { [string equal $prop "port"] || [string equal $prop "l-port"] }]
    set pport [expr { [string equal $prop "port"] || [string equal $prop "p-port"] }]

    if { [string equal $prop "eval"] } {
	set match [uplevel expr $value]
    } elseif { [string equal $prop "proto"] } {
	set match [string equal $conn(proto) $value]
    } elseif { $lport || $pport } {
	foreach p [split $value ","] {
	    if { [regexp {^([0-9]+)(-([0-9]+))?$} $p "" p1 "" p2] } {
		# It's either a port number or a range of port numbers.
		if { $p2 == "" } {
		    set p2 $p1
		}
	    } elseif { [info exists services($proto/$p)] } {
		# Not a port number or range, but it's listed in services.
		set p1 $services($proto/$p)
		set p2 $p1
	    } else {
		# Who the heck knows?
		warn "Unknown service $proto/$p"
		continue
	    }

	    if { ($lport && ($conn(l-port) >= $p1 && $conn(l-port) <= $p2)) || \
		    ($pport && ($conn(p-port) >= $p1 && $conn(p-port) <= $p2)) } {
		set match 1
		break
	    }
	}
    } elseif { [string equal $prop "remote-ip"] } {
	set mask [mask-parse $value]
	set match [mask-match $conn(l-addr) $mask]
    } else {
	warn "Unknown property $prop"
    }

    if { $match } {
	debug "- ON $prop $value matched"
	if { [llength $args] == 1 } {
	    uplevel 1 eval $args
	} else {
	    uplevel 1 $args
	}
    } else {
	debug "- ON $prop $value didn't match"
    }
}

proc score-init {} {
    global conn
    global scores

    if { [info exists scores] } { unset scores }
    if { [info exists conn] } { unset conn }

    array set scores {}
    set conn(migrate) threshold
}

proc migrate { when } {
    global conn
    set conn(migrate) $when
    debug "MIGRATE $when"
}

proc migrate-decision {} {
    global conn
    global scores

    if { [info exists conn(migrate)] } {
	set migrate $conn(migrate)
    } else {
	set migrate "threshold"
    }

    debug "Making migration decision..."

    if { [string equal $migrate "never"] } {
	debug " - never migrate"
	return ""
    }

    # Figure out current score and highest score
    foreach { if score } [array get scores] {
	set label [lindex $if 0]
	set addr [lindex $if 1]
	set mask [lindex $if 2]

	if { [string equal $addr $conn(l-addr)] } {
	    set current_score $score
	    set current_if $if
	}
	if { ![info exists best_score] || $score > $best_score } {
	    set best_score $score
	    set best_if $if
	}
    }

    debug " - variables:"
    foreach s { current_score current_if best_score best_if } {
	debug "   - $s is [expr { [info exists $s] ? [set $s] : "(none)" }]"
    }

    if { [string equal $migrate "if-dead"] } {
	if { ![info exists current_score] && [info exists best_score] } {
	    debug " - it's dead; migrate to $best_if"
	    return $best_if
	} else {
	    debug " - it's not dead; not migrating"
	    return ""
	}
    }

    if { [string equal $migrate "threshold"] } {
	if { [info exists best_score] && \
		(![info exists current_score] || double($best_score) / double($current_score) >= 1.25) } {

	    set current_pretty [expr { [info exists current_score] ? $current_score : "(none)" }]
	    debug " - $best_score beats current score of $current_pretty; migrating to $best_if"
	    return $best_if
	} else {
	    debug " - doesn't beat threshold; not migrating"
	    return ""
	}
    }

    debug " - not migrating"
    return ""
}

proc monitor-policy { scr } {
    global script
    set script [info script]

    proc monitor-policy-real {} $scr
}

proc monitor-policy-eval {} {
    global script
    global debug
    global conn

    if { $debug } {
	debug "POLICY: ($conn(proto) $conn(l-addr):$conn(l-port)->$conn(p-addr):$conn(p-port))"

	# When debugging, reload monitor-init.tcl and monitor-policy.tcl every time to catch
	# updates. This is a semi-hack to make development easier.

	catch { source "monitor-init.tcl" }
	catch { source $script }
    }

    monitor-policy-real
}    

global inited
if { ![info exists inited] } {
    set inited 1
    clear-interfaces
}
