/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: monitor.c,v 1.17 2002/08/30 18:54:12 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_SYS_CDEFS_H
# include <sys/cdefs.h>
#endif
#ifdef HAVE_CTYPE_H
# include <ctype.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif

#ifdef HAVE_TCL
# include <tcl.h>
#endif /* HAVE_TCL */

#include "migrated.h"
#include <list.h>

static int monitorfd = -1;

char *monitor_policy_file = 0;

/* monitor-init.c */
extern char *monitor_init_tcl;

#ifdef HAVE_TCL
static Tcl_Interp *interp = 0;

static char *monitor_policy_search[] = {
  SYSCONFDIR "/monitor-policy.tcl",
  "/etc/monitor-policy.tcl",
  "/usr/local/etc/monitor-policy.tcl",
  "./monitor-policy.tcl",
  0
};
#endif

/* uint16_t really should be in_port_t, but I'm too lazy to write an autoconf
   test. */
typedef struct {
  struct in_addr laddr;
  struct in_addr raddr;
  uint16_t lport;
  uint16_t rport;
  short conn_up;
  short if_up;
} monitorstatemsg;

typedef struct {
  char label[16];
  struct in_addr addr;
  struct in_addr mask;
  short if_up;
} monitorifstatemsg;

int
recv_monitor(mmsghdr *msg, char *buf)
{
  log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	  "Received a %d message from monitor", msg->type);
  assert(msg->type == MONITOR_STATE_CHANGE ||
	 msg->type == MONITOR_IF_STATE_CHANGE);

  if (msg->type == MONITOR_STATE_CHANGE)
  {
    monitorstatemsg *mmsg = (void*)buf;
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - source is %s:%hu", inet_ntoa(mmsg->laddr), ntohs(mmsg->lport));
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - dest is %s:%hu", inet_ntoa(mmsg->raddr), ntohs(mmsg->rport));
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - conn is %s; interface is %s",
	    (mmsg->conn_up ? "up" : "down"),
	    (mmsg->if_up ? "up" : "down"));	  
    
    return 0;
  }
  else if (msg->type == MONITOR_IF_STATE_CHANGE)
  {
    monitorifstatemsg *mmsg = (void*)buf;
    char *ch;

    if (mmsg->label[0] == 0)
    {
#ifdef HAVE_TCL
      session_ent *cur;
      for (cur = session_list; cur; cur = cur->next)
      {
	struct _list_t *curr = NULL;
	for(curr = cur->handle->conns; curr; curr = curr->next) {
   
	  migrate_connection *c = (migrate_connection *)curr->data;
	char buf[128];

	if (!c->saddr.sin_addr.s_addr)
	  continue;

	/* Set up the Tcl environment to consider migration. */
	if (Tcl_Eval(interp, "score-init") != TCL_OK)
	{
	  log_log(LOG_MOD_MONITOR, LOG_ERR,
		  "Error evaluating score-init: %s", Tcl_GetStringResult(interp));
	  continue;
	}

	if (!Tcl_SetVar2(interp, "conn", "proto",
			 c->type == SOCK_STREAM ? "tcp" : "udp",
			 TCL_GLOBAL_ONLY))
	  goto tcl_error;

	/* Notice we're using the "virtual" port numbers (the ones the
	   application thinks the connection is on) but the *actual*
	   current endpoint IP addresses. */

	sprintf(buf, "%hu", ntohs(c->saddr.sin_port));
	if (!Tcl_SetVar2(interp, "conn", "l-port", buf, TCL_GLOBAL_ONLY))
	  goto tcl_error;
	sprintf(buf, "%hu", ntohs(c->daddr.sin_port));
	if (!Tcl_SetVar2(interp, "conn", "p-port", buf, TCL_GLOBAL_ONLY))
	  goto tcl_error;

	sprintf(buf, "%08X", (unsigned int)ntohl(c->csaddr.sin_addr.s_addr));
	if (!Tcl_SetVar2(interp, "conn", "l-addr", buf, TCL_GLOBAL_ONLY))
	  goto tcl_error;
	sprintf(buf, "%08X", (unsigned int)ntohl(c->cdaddr.sin_addr.s_addr));
	if (!Tcl_SetVar2(interp, "conn", "p-addr", buf, TCL_GLOBAL_ONLY))
	  goto tcl_error;

	if (Tcl_Eval(interp, "monitor-policy-eval") != TCL_OK)
	{
	  log_log(LOG_MOD_MONITOR, LOG_ERR,
		  "Error evaluating monitor policy: %s",
		  Tcl_GetStringResult(interp));
	  continue;
	}

	if (Tcl_Eval(interp, "migrate-decision") != TCL_OK) {
	  log_log(LOG_MOD_MONITOR, LOG_ERR,
		  "Error evaluating migrate-decision: %s", Tcl_GetStringResult(interp));
	  continue;
	} else {
	  /* Woohoo! What's the decision? */
	  char *result = Tcl_GetStringResult(interp);
	  int argc;
	  char **argv = 0;

	  log_log(LOG_MOD_MONITOR, LOG_DEBUG, "Migrate decision: \"%s\"", result);

	  if (Tcl_SplitList(interp, result, &argc, &argv) != TCL_OK ||
	      (argc != 0 && argc != 3)) {
	    log_log(LOG_MOD_MONITOR, LOG_ERR, "Invalid list returned from migrate-decision");
	    goto end;
	  }

	  if (argc == 3) {
	    int cnt;
	    unsigned long val;
	    struct sockaddr_in addr;

	    if (!(cnt = sscanf(argv[1], "%lx", &val))) {
	      log_log(LOG_MOD_MONITOR, LOG_ERR, "Invalid address \"%s\"", argv[1]);
	      goto end;
	    }

	    memset(&addr, 0, sizeof addr);
	    addr.sin_family = AF_INET;
	    addr.sin_addr.s_addr = htonl(val);
	    addr.sin_port = 0;
	    mig_session(cur->handle, &addr);

	    /*
	    {
	      char buf[128];
	      strcpy(buf, "0 18.31.0.72");
	      migrate(stderr, buf);
	    }
	    */
	  }

	 end:
	  if (argv) Tcl_Free((char *)argv);
	}
      }
      }
#endif /* HAVE_TCL */

      return 0;
    }

    /* Make sure the string is zero-terminated */
    mmsg->label[sizeof(mmsg->label) - 1] = 0;

    if (!strcmp(mmsg->label, "*")) {
      /* Reset interface list. */
#ifdef HAVE_TCL
      if (Tcl_Eval(interp, "clear-interfaces") != TCL_OK)
      {
	log_log(LOG_MOD_MONITOR, LOG_ERR,
		"Error clearing interfaces: %s", Tcl_GetStringResult(interp));
      }

      return 0;
#endif
    }

    for (ch = mmsg->label; *ch; ++ch)
      if (!isalnum(*ch) && *ch != ':')
      {
	log_log(LOG_MOD_MONITOR, LOG_ERR, "Invalid label \"%s\" in "
		"interface state change struct from monitor",
		mmsg->label);
	return 1;
      }

    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - label is %s", mmsg->label);
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - addr is %s", inet_ntoa(mmsg->addr));
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - mask is %s", inet_ntoa(mmsg->mask));
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    " - state is %s", mmsg->if_up ? "up" : "down");

#ifdef HAVE_TCL
    if (interp)
    {
      char cmd[128];
      int ret;

      sprintf(cmd,
	      "interface \"%s\" %d %08X %08X",
	      mmsg->label,
	      (int)mmsg->if_up,
	      (unsigned int)ntohl(mmsg->addr.s_addr),
	      (unsigned int)ntohl(mmsg->mask.s_addr));

      ret = Tcl_Eval(interp, cmd);
      if (ret != TCL_OK)
      {
	log_log(LOG_MOD_MONITOR, LOG_ERR,
		"Error when invoking Tcl command \"%s\" from monitor: %s",
		cmd, Tcl_GetStringResult(interp));
      }
    }
#endif /* HAVE_TCL */

    return 0;
  }

#ifdef HAVE_TCL
 tcl_error:
  log_log(LOG_MOD_MONITOR, LOG_ERR,
	  "Unexpected Tcl error: %s", Tcl_GetStringResult(interp));
#endif
  return 1;
}

#ifdef HAVE_TCL
static int
init_interp()
{
  struct servent *ent;
  char *script;

  interp = Tcl_CreateInterp();
  if (!interp)
    MIGRATE_FATAL("Unable to create TCL interpreter");

  script = strdup(monitor_init_tcl);
  if (Tcl_Eval(interp, script) != TCL_OK) {
    free(script);
    log_log(LOG_MOD_MONITOR, LOG_ERR,
	    "Evaluation of monitor-init.tcl failed: %s",
	    Tcl_GetStringResult(interp));
    return 1;
  }
  free(script);

  /* Load the configuration file. */
  if (!monitor_policy_file) {
    struct stat buf;
    char **search;

    for (search = monitor_policy_search; *search; ++search)
      if (stat(*search, &buf) == 0)
	break;

    if (!*search) {
      log_log(LOG_MOD_MONITOR, LOG_ERR,
	      "Unable to locate migrate-policy.tcl");
      return 1;
    }

    monitor_policy_file = *search;
  } else {
    struct stat buf;

    if (stat(monitor_policy_file, &buf)) {
      log_log(LOG_MOD_MONITOR, LOG_ERR,
	      "Monitor policy file %s does not exist", monitor_policy_file);
      return 1;
    }
  }    

  if (Tcl_EvalFile(interp, monitor_policy_file) != TCL_OK) {
    log_log(LOG_MOD_MONITOR, LOG_ERR,
	    "Evaluation of %s failed: %s",
	    monitor_policy_file,
	    Tcl_GetStringResult(interp));
    return 1;
  }

  /* Load service entries. */
  while ((ent = getservent()) != 0)
  {
    char name[128];
    char port[16];
    char **alias;

    if (strcmp(ent->s_proto, "tcp"))
      continue;

    snprintf(port, sizeof port, "%hu", ntohs(ent->s_port));
    snprintf(name, sizeof name, "%s/%s", ent->s_proto, ent->s_name);
    name[sizeof name - 1] = '\0';

    if (!Tcl_SetVar2(interp, "services", name, port, TCL_GLOBAL_ONLY))
      goto tcl_error;

    for (alias = ent->s_aliases; *alias; ++alias) {
      snprintf(name, sizeof name, "%s/%s", ent->s_proto, *alias);
      name[sizeof name - 1] = '\0';
      if (!Tcl_SetVar2(interp, "services", name, port, TCL_GLOBAL_ONLY) !=
	  TCL_OK)
	goto tcl_error;
    }
  }
  return 0;

 tcl_error:
  log_log(LOG_MOD_MONITOR, LOG_ERR,
	  "Evaluation of interp init Tcl code failed: %s",
	  Tcl_GetStringResult(interp));
  return 1;
  }

#endif /* HAVE_TCL */

int
monitor_init()
{
  int ret = 0;

#ifdef HAVE_TCL
  ret = init_interp();
  if (ret) return ret;
#endif

  return ret;
}

/* Returns 1 to accept the connection, 0 to close. */
int
monitor_open(int fd)
{
  if (monitorfd < 0)
  {
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    "No FD yet. Accepting %d.", fd);
    monitorfd = fd;

    return 1;
  }
  log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	  "Had FD %d. Rejecting.", monitorfd);
  return 0;
}

void
monitor_close(int fd)
{
  if (monitorfd == fd)
    monitorfd = -1;
}

int
monitor_conn(migrate_connection *conn, int fd)
{
  char buf[128];
  int len, totallen;

  if (fd < 0)
    fd = monitorfd;
  if (fd < 0)
    return 0;

  /* Don't monitor listening sockets. */
  if (!conn->saddr.sin_addr.s_addr)
    return 0;

  /* Ask monitor to watch this connection */
  log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	  "Asking monitor to watch %s:%hu", inet_ntoa(conn->saddr.sin_addr),
	  ntohs(conn->saddr.sin_port));

  len = totallen = snprintf(buf, sizeof(buf), "watch %s:%hu->", 
			    inet_ntoa(conn->csaddr.sin_addr),
			    ntohs(conn->csaddr.sin_port));
  assert(len >= 0 && totallen < (int)sizeof(buf));

  len = snprintf(buf + len, sizeof(buf) - len, "%s:%hu\n",
		 inet_ntoa(conn->cdaddr.sin_addr),
		 ntohs(conn->cdaddr.sin_port));
  totallen += len;
  assert(len >= 0 && totallen < (int)sizeof(buf));

  buf[totallen - 1] = '\0';
  log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	  "Telling monitor: \"%s\"", buf);
  buf[totallen - 1] = '\n';

  if (write(monitorfd, buf, totallen) < 0)
  {
    log_log(LOG_MOD_MONITOR, LOG_DEBUG,
	    "Unable to write to monitor: %s", strerror(errno));
    close(monitorfd);
    monitorfd = -1;
  }

  return 0;
}
