/*
 * Migrate Session Layer
 *
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: netio.c,v 1.23 2002/10/10 21:50:21 snoeren Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif

#include "migrated.h"

char migrate_hello[] = "Migrate Hello";

int
session_sendmsg(session_handle *handle) {

  int len = handle->msghdr.len;

  handle->msghdr.len = htonl(handle->msghdr.len);
  assert(handle->pfd != -1);

  if(write(handle->pfd, &handle->msghdr, sizeof(handle->msghdr)) !=
     sizeof(handle->msghdr))
    return errno;
  if(write(handle->pfd, handle->buf, len) != len)
    return errno;

  handle->buflen = 0;
  handle->msghdr.len = 0;

  return 0;
}


int
recv_net(session_handle *handle)
{
  void *key;

  if(handle->key) {
    if(!(key = finish_handshake(handle)))
      return -1;
    free(handle->key);
    handle->key = NULL;
  } else {
    if(!(key = session_handshake(handle)))
      return -1;
    session_sendmsg(handle);
  }
  set_state(handle, MIGRATE_ESTABLISHED);  
  handle->key = key;

  if (notify_libraries(handle))
    assert(0);

  return 0;
}


int
session_update_conn(migrate_connection *conn)
{
  session_handle * handle = (session_handle *)conn->session;
  int retval = 0;

  /* Don't notify other side of non-fixed port conns */
  if(conn->daddr.sin_addr.s_addr != INADDR_ANY) {
    handle->msghdr.type = CONNECTION_MSG;
    handle->msghdr.len = sizeof(migrate_connection);
    handle->buflen = sizeof(migrate_connection);
    memcpy(handle->buf, conn, handle->buflen);
    ((migrate_connection *)handle->buf)->session = 
      (migrate_session *)htonl(handle->session.pid);
    if((retval = session_sendmsg(handle)))
      return retval;
  }
  return 0;
}

int
session_update_conns(session_handle *handle)
{
  migrate_connection *conn = NULL;
  struct _list_t *curr = NULL;

  for(curr = handle->conns; curr; curr = curr->next) {
    conn = (migrate_connection *)curr->data;
    session_update_conn(conn);
  }
  return 0;
}

int
session_update_sess(session_handle *handle)
{
  handle->msghdr.type = SESSION_MSG;
  handle->msghdr.len = handle->buflen = sizeof(migrate_session);
  memcpy(handle->buf, &handle->session, handle->buflen);
  return session_sendmsg(handle);
}


int
migrate_conn(migrate_connection *conn, session_handle *handle)
{

    /* Update local & destination addresses */
    memcpy(&conn->cdaddr,
	   &((migrate_connection*)handle->buf)->csaddr,
	   sizeof(conn->cdaddr));
    /* We only want the new IP address; we'll nuke port later */
    memcpy(&conn->csaddr,
	   &handle->session.laddr,
	   sizeof(conn->csaddr));

    if ((conn->rfd = socket(AF_INET, conn->type, 0)) == -1) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "socket: %s", strerror(errno));
      return -1;
    }
    if (fcntl(conn->rfd, F_SETFL, O_NONBLOCK)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "fcntl: %s", strerror(errno));
      close(conn->rfd);
      return (conn->rfd = -1);
    }
    if (fcntl(conn->rfd, F_SETFD, FD_CLOEXEC)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "fcntl (FD_CLOEXEC): %s", strerror(errno));
      close(conn->rfd);
      return (conn->rfd = -1);
    }
    /* Nail down to appropriate address.  We don't care what port. */
    conn->csaddr.sin_port = 0;
    if (bind(conn->rfd, (struct sockaddr *)&conn->csaddr,
	     sizeof(conn->csaddr))) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "bind: %s", strerror(errno));
      close(conn->rfd);
      return (conn->rfd = -1);
    }
    log_log(LOG_MOD_MIGRATED, LOG_NOTICE,
	    "Connecting to %s on port %d", inet_ntoa(conn->cdaddr.sin_addr),
	    ntohs(conn->cdaddr.sin_port));
    if ((connect(conn->rfd,
		 (const struct sockaddr *)&conn->cdaddr,
		 sizeof(struct sockaddr_in))) == -1)
      if(errno!=EINPROGRESS) {
	log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
		"connect: %s", strerror(errno));
	close(conn->rfd);
	conn->rfd = -1;
	return -1;
      }

    switch (conn->type) {

    case SOCK_STREAM:
      /* Nothing else to do but wait for the connect */
      break;

    case SOCK_DGRAM: {
      int len = (strlen(migrate_hello) + 1);

/* Not all OSes support NOSIGNAL */
#ifndef MSG_NOSIGNAL
#  define MSG_NOSIGNAL 0
#endif
      /* We need to send a hello to bind the address... */
      if (send(conn->rfd, migrate_hello, len, MSG_NOSIGNAL) < len) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR,
		"send: %s", strerror(errno));
	close(conn->rfd);
	conn->rfd = -1;
      }
      break;
	       }
    default:
      /* We don't yet handle this type of socket */
      assert(0);
    }

    return conn->rfd;
}

int
migrate_conn_prepare(migrate_connection *conn, session_handle *handle)
{
  int len, csock = -1;

    /* Update local address */
    len = sizeof(conn->csaddr);
    memcpy(&conn->csaddr, &handle->session.laddr, len);

    /* Create a new socket to use instead */
    if ((csock = socket(AF_INET, conn->type, 0)) == -1) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "socket: %s", strerror(errno));
      return -1;
    }
    if (fcntl(csock, F_SETFD, FD_CLOEXEC)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "fcntl (FD_CLOEXEC): %s", strerror(errno));
      close(csock);
      return -1;
    }
    conn->csaddr.sin_port = 0;
    if (bind(csock, (struct sockaddr *)&conn->csaddr, len)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "bind: %s", strerror(errno));
      close(csock);
      return -1;
    }
    if (getsockname(csock, (struct sockaddr *)&conn->csaddr, &len)) {
      log_log(LOG_MOD_MIGRATED, LOG_ERR,
	      "getsockname: %s", strerror(errno));
      close(csock);
      return -1;
    }

    switch(conn->type) {
      
    case SOCK_STREAM: 

      if (fcntl(csock, F_SETFL, O_NONBLOCK)) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR,
		"Unable to set listening port non-blocking: %s",
		strerror(errno));
	close(csock);
	return -1;
      }
      if (listen(csock, 1) < 0) {
	log_log(LOG_MOD_MIGRATED, LOG_ERR,
		"listen: %s", strerror(errno));
	close(csock);
	return -1;
      }
      /* Fallthrough... */
   
    case SOCK_DGRAM:
      conn->rfd = csock;
      break;

    default:
      /* We don't yet handle this type of socket */
      assert(0);
    }
    
    /* Notify peer of listening socket */
    handle->buflen = sizeof(*(conn));
    assert(handle->buflen <= BUFLEN);
    memcpy(handle->buf, conn, handle->buflen);
    ((migrate_connection *)handle->buf)->session = 
      (migrate_session *)htonl(handle->session.pid);
    handle->msghdr.type = CONNECTION_MSG;
    handle->msghdr.len = handle->buflen;
    session_sendmsg(handle);

    log_log(LOG_MOD_MIGRATED, LOG_DEBUG,
	    "Opening port %d for listening on %s",
	    ntohs(conn->csaddr.sin_port), inet_ntoa(conn->csaddr.sin_addr));
  
  return csock;
}

