 #include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include "hash.h"

struct hashtype *
hash_create(int hash_buckets, hash_free_t freeproc)
{
	struct hashtype *newhash;
	newhash = malloc(sizeof(*newhash));
	if (!newhash) {
		return NULL;
	}
	bzero(newhash, sizeof(*newhash));

	newhash->hash_nbuckets = hash_buckets;
	newhash->freeproc = freeproc;
	newhash->hash_size = 0;
	newhash->contents = malloc(sizeof(struct htable_entry *) * hash_buckets);
	if (!newhash->contents) {
		goto cleanup;
	}
	bzero(newhash->contents, sizeof(struct htable_entry *) * hash_buckets);
	
	return newhash;
	
 cleanup:
	if (newhash->contents != NULL) {
		free(newhash->contents);
	}
	free(newhash);
	return NULL;
}

void
hash_empty(struct hashtype *A)
{
	int i;
	struct htable_entry *tb;
	
	for (i = 0; i < A->hash_nbuckets; i++) {
		tb = A->contents[i];
		while (tb != NULL) {
			struct htable_entry *freeme;
			freeme = tb;
			tb = tb->next;
			A->freeproc(freeme->item);
			free(freeme);
		}
		A->contents[i] = NULL;
	}
}

void
hash_destroy(struct hashtype *A)
{
	hash_empty(A);
	free(A->contents);
	free(A);
}

/*
 * XXX, might need to fix this eventually or make it
 * a user-supplyable function.
 */
int hkey_to_bucket(struct hashtype *A, hkey_t key)
{
	return (key % A->hash_nbuckets);
}

stored_t *
hash_find(struct hashtype *A, hkey_t key)
{
	int bucket;
	struct htable_entry *buck;

	if (A->contents == NULL) { return NULL; }
	
	bucket = hkey_to_bucket(A, key);
	buck = A->contents[bucket];

	if (buck == NULL) { return NULL; }

	while (buck != NULL) {
		if (buck->key == key) { return buck->item; }
		buck = buck->next;
	}
	return NULL;
}

int
hash_insert(struct hashtype *A, hkey_t key, stored_t *newitem)
{
	int bucket;
	struct htable_entry *buck;
	if (hash_find(A, key)) {
		errno = EEXIST;
		return -1;
	}
	buck = malloc(sizeof(*buck));
	if (!buck) {
		return -1;
	}
	bucket = hkey_to_bucket(A, key);
	buck->next = A->contents[bucket];
	buck->item = newitem;
	buck->key = key;
	A->contents[bucket] = buck;
	A->hash_size++;
	return 0;
}

stored_t *
hash_remove(struct hashtype *A, hkey_t key)
{
	int bucket;
	stored_t *item;
	struct htable_entry *buck, *prev;
	bucket = hkey_to_bucket(A, key);
	buck = A->contents[bucket];
	if (buck == NULL) { return NULL; }
	if (buck->key == key) {
		A->contents[bucket] = buck->next;
		item = buck->item;
		free(buck);
		A->hash_size--;
		return item;
	}
	prev = buck;
	buck = buck->next;
	while (buck != NULL) {
		if (buck->key == key) {
			prev->next = buck->next;
			item = buck->item;
			free(buck);
			A->hash_size--;
			return item;
		}
		prev = buck;
		buck = buck->next;
	}
	return NULL;
}

int
hash_size(struct hashtype *A)
{
	return A->hash_size;
}


#ifdef TEST_HASH
#include <assert.h>
#include <stdio.h>

void
freeproc(struct htest *item)
{
	free(item);
}

#define NITEMS 10
int
main()
{
	struct hashtype *h;
	struct htest **hashes;
	struct htest *tmp;
	int i;

	hashes = malloc(sizeof(struct htest *) * NITEMS);
	assert(hashes != NULL);
	
	for (i = 0; i < NITEMS; i++) {
		hashes[i] = malloc(sizeof(struct htest));
		assert(hashes[i] != NULL);
		sprintf(hashes[i]->stuff, "Hash-%d", i);
	}

	h = hash_create(3, freeproc);
	assert(h != NULL);

	for (i = 0; i < NITEMS; i++) {
		hash_insert(h, i, hashes[i]);
	}

	assert(hash_size(h) == NITEMS);
	
	for (i = 0; i < NITEMS; i++) {
		tmp = hash_find(h, i);
		assert(tmp != NULL);
		printf("Hash %d:  %s\n", i, tmp->stuff);
		free(hash_remove(h, i));
		assert(hash_size(h) == NITEMS-i-1);
	}
	exit(0);
}
#endif
