/*
 * This program runs on a machine and binds to a port, listens
 * for incoming requests on the port, then redirects that request
 * to a webserver running on the same machine via a TCP splice.
 *
 * $Id: wedge.c,v 1.11 2001/04/26 19:15:03 dga Exp $
 *
 * Architecture:
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/signal.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <linux/unistd.h>
#include <assert.h>
#include <string.h>
#include <netdb.h>
#include "wedge.h"
#include "debug.h"

#include "hash.h"

#ifdef DEBUG
unsigned int debug = 0;
#endif

int local_port = DEFAULT_LOCAL_TCP_PORT;
int remote_port = DEFAULT_REMOTE_PORT;

struct timeval now;  /* Update whenever necessary */

struct sockaddr_in localhost_sock; /* Cached copy for local connections */

struct con_info *cons = NULL;	/* The local connection table */
struct server *servers = NULL;	/* The remote servers we know about */

struct hashtype *all_cons = NULL; /* A hash of all (local,remote) cons */

/*
 * Misc. little typecasting functions.  Dave was lazy in writing
 * the linked lists for the connection tracking
 */

inline struct listel *con_to_list(struct con_info *a) {
    return (struct listel *)a;
}

inline struct con_info *list_to_con(struct listel *a) {
    return (struct con_info *)a;
}

inline struct listel *remcon_to_list(struct remcon *a) {
    return (struct listel *)a;
}

inline struct remcon *list_to_remcon(struct listel *a) {
    return (struct remcon *)a;
}

inline struct listel *serv_to_list(struct server *a) {
    return (struct listel *)a;
}

inline struct server *list_to_serv(struct listel *a) {
    return (struct server *)a;
}

/* The basic hashing function: turn a remote connection into a key */

int remcon_to_key(struct remcon *cons) {
    /*
     * XXX - this is a temporary hash function;  we should probably
     * make it more complete for the real implementation
     */
    return (cons->remhost.sin_addr.s_addr + cons->remhost.sin_port);
}

/*
 * Interface to the kernel migrate code
 */

_syscall2(int, socketcall, int, call, unsigned long *, args);

int
getsockstate(int s, struct sockstate *state, int len)
{
    unsigned long buff[3];
    
    buff[0] = s;
    buff[1] = (unsigned long)state;
    buff[2] = (unsigned long)&len;
    
    return socketcall(18, buff);
}

int
setsockstate(int s, struct sockstate *state, int len)
{
    unsigned long buff[3];
    
    buff[0] = s;
    buff[1] = (unsigned long)state;
    buff[2] = len;
    
    return socketcall(19, buff);
}


/*
 * If this ever becomes a bottleneck, we can replace this
 * with a hash lookup, but with low numbers of servers, it
 * shouldn't be a problem;  it's called for every announcement
 * UDP packet received.
 */
struct server *
find_server(struct in_addr remaddr)
{
    struct server *tmp;
    tmp = servers;
    while (tmp != NULL) {
	if (tmp->sockaddr.sin_addr.s_addr == remaddr.s_addr) {
	    return tmp;
	}
	tmp = list_to_serv(tmp->head.next);
    }
    return NULL;
}

struct server *
add_server(struct sockaddr_in sockaddr)
{
    struct server *newserver;

    newserver = malloc(sizeof(*newserver));
    if (!newserver) {
	perror("could not allocate server struct");
	return NULL;
    }
    newserver->con_list = NULL;
    newserver->con_tail = NULL;
    newserver->sockaddr = sockaddr;
    newserver->head.prev = NULL;
    newserver->head.next = serv_to_list(servers);
    if (servers) {
	servers->head.prev = serv_to_list(newserver);
    }
    servers = newserver;
    return newserver;
}

int
read_servers(char *server_filename)
{
    FILE *infile;
    char inbuf[512];
    struct hostent *he;
    struct sockaddr_in sockaddr;
    char *retpos;

    infile = fopen(server_filename, "r");
    if (!infile) {
	perror(server_filename);
	return -1;
    }
    while (fgets(inbuf, sizeof(inbuf)-1, infile) != NULL) {
	bzero(&sockaddr, sizeof(sockaddr));
	retpos = strchr(inbuf, '\n');
	if (retpos) *retpos = '\0';
	
	if (!inet_aton(inbuf, &(sockaddr.sin_addr))) {
	    he = gethostbyname(inbuf);
	    if (!he) {
		herror(inbuf);
		continue; /* Just skip the name... */
	    }
	    memcpy(&(sockaddr.sin_addr), he->h_addr, he->h_length);
	}
	sockaddr.sin_port = htons(ANNOUNCE_PORT);
	sockaddr.sin_family = AF_INET;
#ifndef NO_SIN_LEN
	sockaddr.sin_len = sizeof(sockaddr);
#endif
	/* Okay, resolved */
	add_server(sockaddr);
    }
    return 0;
}

/*
 * Decide if we should try to take over the connection when
 * the remote server fails.
 *
 * returns 1 if we should, 0 otherwise.
 */

int takeoverp(struct remcon *rem)
{
    return 1; /* grin */
}

long timediff(const struct timeval *time_now,
              const struct timeval *time_prev)
{
        long usecs_now;
        
        if (time_prev->tv_sec == 0 &&
            time_prev->tv_usec == 0) {
                return 0;
        }
        
        usecs_now = (time_now->tv_sec - time_prev->tv_sec) * 1000000;
        usecs_now += time_now->tv_usec;

        return usecs_now - time_prev->tv_usec;
}

void
timeval_add(struct timeval *tv, unsigned int add_us)
{
    int add_sec;

    add_sec = add_us / 1000000;
    add_us %= 1000000;
    
    if ((tv->tv_usec + add_us) > 1000000) {
	add_us -= 1000000;
	add_sec++;
    }
    tv->tv_usec += add_us;
    tv->tv_sec += add_sec;
}

int
insert_remcon(struct server *server, struct remcon *newcon)
{
    newcon->server = server;
    newcon->expire_time = now;
    timeval_add(&(newcon->expire_time), CONNECTION_CACHE_TIMEOUT_USEC);

    newcon->head.prev = NULL;
    newcon->head.next = remcon_to_list(server->con_list);
    if (server->con_list) {
	server->con_list->head.prev = remcon_to_list(newcon);
    } else {
	server->con_tail = newcon;
    }
    server->con_list = newcon;
    return hash_insert(all_cons, remcon_to_key(newcon), newcon);
}

/* This will DEEP COPY the remcon passed in.
 * it _will_ allocate new memory for the
 * URL field
 */
int
add_remcon(struct server *server, struct remcon *remcon)
{
    struct remcon *newcon;
    newcon = malloc(sizeof(*newcon));
    if (!newcon) {
	perror("add_remcon: malloc");
	return -1;
    }
    newcon->head.type = TYPE_REMOTE;
    newcon->remhost = remcon->remhost;
    newcon->server = server;
    newcon->ss = remcon->ss;
    newcon->headerlen = remcon->headerlen;
    newcon->con_seqno = remcon->con_seqno;
    newcon->url = strdup(remcon->url);
    if (!newcon->url) {
	perror("add_remcon: strdup");
	free(newcon);
	return -1;
    }
    return insert_remcon(server, newcon);
}

void
free_remcon(struct remcon *remcon)
{
    if (remcon->url) free(remcon->url);
    free(remcon);
}

/*
 * This removes from the list, but DOES NOT FREE, the remcon
 */

void
remove_remcon(struct server *server, struct remcon *remcon)
{
    int key;

    key = remcon_to_key(remcon);
    if (!hash_remove(all_cons, key)) {
	fprintf(stderr, "removing remcon: key %d wasn't in hash\n", key);
	/* Oh well, life goes on */
    }

    /* Sanity checking */
    if (server->con_list == NULL) {
	fprintf(stderr, "removing remcon:  server had null con list\n");
	return;
    }
    
    /* Case 1:  It's the only connection on the server */
    if (server->con_list == remcon && server->con_tail == remcon) {
	server->con_list = NULL;
	server->con_tail = NULL;
	return;
    }
    /* Case 2:  It's the tail, but not the head */
    if (server->con_tail == remcon) {
	server->con_tail = list_to_remcon(remcon->head.prev);
	remcon->head.prev->next = NULL;
	return;
    }
    /* Case 3:  It's the head, but not the tail */
    if (server->con_list == remcon) {
	server->con_list = list_to_remcon(remcon->head.next);
	remcon->head.next->prev = NULL;
	return;
    }
    /* Case 4:  It's in the middle */
    remcon->head.prev->next = remcon->head.next;
    remcon->head.next->prev = remcon->head.prev;
    return;
}

void
refresh_remcon(struct server *server, struct remcon *remcon)
{
    /* Laziness.  We could save a hash update here if we
     * wanted to */
    remove_remcon(server, remcon);
    insert_remcon(server, remcon);
}

/*
 * Add a local connection
 */

struct con_info *
add_connection(int cli_fd)
{
    struct con_info *newcon;

    newcon = malloc(sizeof(*newcon));

    if (!newcon) {
	perror("could not allocate new connection struct");
	return NULL;
    }

    newcon->cli_fd = cli_fd;
    newcon->state = PENDING_CON;
    newcon->serv_fd = 0;
    newcon->head.prev = NULL;
    newcon->head.next = con_to_list(cons);
    newcon->head.type = TYPE_LOCAL;
    newcon->url = NULL;
    newcon->headerlen = 0;

    if (cons) {
	cons->head.prev = con_to_list(newcon);
    }

    cons = newcon;
    
    return newcon;
}

/*
 * Removes a local connection
 *
 * Before calling rem_connection, both fds must be closed
 */

void
rem_connection(struct con_info *clicon)
{
    int cli_fd;

    cli_fd = clicon->cli_fd;
    if (!clicon) {
	fprintf(stderr, "attempt to rem connection for invalid con\n");
	return;
    }
    if (!hash_remove(all_cons, remcon_to_key((struct remcon *)clicon))) {
	fprintf(stderr, "removing local con:  key %d wasn't in hash\n",
		remcon_to_key((struct remcon *)clicon));
    }
    if (clicon->head.prev) {
	clicon->head.prev->next = clicon->head.next;
    } else {
	cons = list_to_con(clicon->head.next);
    }
    if (clicon->url != NULL) {
	free(clicon->url);
    }
    free(clicon);
}

/*
 * Send an announcement packet to all of our peer servers
 */

void
send_to_servers(int announce_sock, char *pkt)
{
    struct wedge_header *hdr;
    int pktsize;
    struct server *serv;
    
    hdr = (struct wedge_header *)pkt;
    pktsize = sizeof(struct wedge_header) +
	ntohl(hdr->numcons) * sizeof(struct onecon);
    DPRINTF(DEBUG_ANNOUNCE,
	    "Numcons: %d\n"
	    "Sending packet of size %d\n",
	    (int)ntohl(hdr->numcons), pktsize);

    for (serv = servers; serv != NULL; serv = list_to_serv(serv->head.next))
    {
	DPRINTF(DEBUG_ANNOUNCE, "  .. to %s\n",
		inet_ntoa(serv->sockaddr.sin_addr));
	
	sendto(announce_sock, pkt, pktsize, 0,
	       (struct sockaddr *)&(serv->sockaddr),
	       sizeof(struct sockaddr_in));
    }
    
}

/*
 * Build the periodic update packets to send to remote servers
 */

void
periodic_update(int announce_sock)
{
    struct con_info *con;
    char pkt[1400];
    struct wedge_header *hdr = (struct wedge_header *)pkt;
    struct onecon *pktcon;

    int ncons = 0;
    int n_in_pkt = 0;
    int max_in_pkt = 0;

    max_in_pkt = sizeof(pkt) - sizeof(struct wedge_header);
    max_in_pkt /= sizeof(struct onecon);

    /* Build an announcement to send to the remote servers */
    for (con = cons; con != NULL; con = list_to_con(con->head.next))
    {
	if (con->state == READ_GET)
	    ncons++;
    }
    DPRINTF(DEBUG_ANNOUNCE, "Sending periodic announcement\n");

    bzero(pkt, sizeof(pkt));
    
    /* Build an announcement to send to the remote servers */
    pktcon = (struct onecon *) (pkt + sizeof(struct wedge_header));
    for (con = cons; con != NULL; con = list_to_con(con->head.next))
    {
	/* We only deal with fully formed connections */
	if (con->state != READ_GET) { continue; }
	DPRINTF(DEBUG_ANNOUNCE, "  %16.16s  %d -> %d  %s\n",
		inet_ntoa(con->remhost.sin_addr),
		con->cli_fd, con->serv_fd,
		con->url);
	pktcon->remhost = con->remhost.sin_addr.s_addr;
	pktcon->remport = con->remhost.sin_port;
	pktcon->ss      = con->ss;
	pktcon->headerlen = htonl(con->headerlen);
	pktcon->con_seqno = htons(con->con_seqno);
	memcpy(pktcon->url, con->url,
	       max(strlen(con->url), sizeof(pktcon->url)));
	n_in_pkt++;
	if (n_in_pkt >= max_in_pkt) {
	    hdr->numcons = htonl(n_in_pkt);
	    send_to_servers(announce_sock, pkt);
	    n_in_pkt = 0;
	    bzero(pkt, sizeof(pkt));
	    pktcon = (struct onecon *)(pkt + sizeof(struct wedge_header));
	} else {
	    pktcon = (struct onecon *)(pkt + sizeof(struct wedge_header)
				       + (sizeof(struct onecon)*(n_in_pkt-1)));
	}
	
    }
    if (n_in_pkt > 0) {
	hdr->numcons = htonl(n_in_pkt);
	send_to_servers(announce_sock, pkt);
    }
    DPRINTF(DEBUG_ANNOUNCE, "\n");
}

#ifdef DEBUG
/* Purely for debugging */
void
print_allcons()
{
    struct server *server;

    for (server = servers; server != NULL;
	 server = (struct server *)server->head.next) {
	/* Run through their lists nuking connections until
	 * we hit one that hasn't yet expired */
	long left;
	struct remcon *con, *nextcon;

	con = server->con_list;
	printf("remote connections on %s:\n",
	       inet_ntoa(server->sockaddr.sin_addr));

	while (1) {
	    if (con == NULL) break;
	    nextcon = (struct remcon *)con->head.next;

	    left = timediff(&(con->expire_time), &now);
	    printf("   %s %d - %s - %ld microseconds remaining\n",
		   inet_ntoa(con->remhost.sin_addr),
		   ntohs(con->remhost.sin_port),
		   con->url,
		   left);
	    con = nextcon;
	}
    }
}
#endif

void
check_connection_cache()
{
    struct server *server;

    for (server = servers; server != NULL;
	 server = (struct server *)server->head.next) {
	/* Run through their lists nuking connections until
	 * we hit one that hasn't yet expired */
	long left;
	struct remcon *con, *nextcon;

	con = server->con_tail;

	while (1) {
	    if (con == NULL) break;
	    nextcon = (struct remcon *)con->head.prev;

	    left = timediff(&(con->expire_time), &now);
	    if (left < 1000) {
		remove_remcon(server, con);
		free_remcon(con);
	    } else {
		break;
	    }
	    con = nextcon;
	}
    }
}

/*
 * Determine the current offset based on the TCP sequence numbers
 */

/*
 * Cheezeball hack to get the struct in here - Alex, at some
 * point, you should give a real header or library function
 * so programs can access certain bits of the exported
 * state without doing what I'm doing here.  But, as they
 * say, one layer violation begets another.
 */
struct my_tcp_opt {
    int     tcp_header_len; /* Bytes of tcp header to send          */
    u_int32_t   pred_flags;
    u_int32_t   rcv_nxt;        /* What we want to receive next         */
    u_int32_t   snd_nxt;        /* Next sequence we send                */
    u_int32_t   snd_una;        /* First byte we want an ack for        */
    u_int32_t   rcv_tstamp;     /* timestamp of last received packet    */
};

unsigned int
compute_offset(struct remcon *con, int remfd)
{
    struct sockstate curstate;
    struct my_tcp_opt *cur, *old;

    getsockstate(remfd, &curstate, sizeof(curstate));

    cur = (struct my_tcp_opt *)&curstate;
    old = (struct my_tcp_opt *)&(con->ss);

    printf("compute_offset:  cur snd_nxt:  %u  snd_una:  %u\n"
	   "                 old snd_nxt:  %u  snd_una:  %u\n",
	   cur->snd_nxt, cur->snd_una,
	   old->snd_nxt, old->snd_una);

    return (cur->snd_una - old->snd_una - con->headerlen);
}


/*
 * Take over a remote connection
 *
 * returns 0 on quasi-success, -1 on utter failure
 */

int
takeover(struct remcon *con)
{
    struct con_info *newcon = NULL;
    int remfd = -1, servsock = -1;
    char reqbuf[512]; /* XXX - size me dynamically */
    int rc, len;
    
    DPRINTF(DEBUG_DEATH, "Taking over remote connection %s %d\n",
	    inet_ntoa(con->remhost.sin_addr),
	    ntohs(con->remhost.sin_port));

    /* See if we think we already have this connection! */
    
    /* Allocate a socket for them.
     * Eventually, mark it nonblocking and throw them in the event
     * loop
     */

    remfd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (remfd == -1) {
	perror("takeover: remote socket");
	goto error;
    }

    servsock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (servsock == -1) {
	perror("takeover: socket for local part");
	goto error;
    }
    
    if (connect(servsock, (struct sockaddr *)&localhost_sock,
		sizeof(localhost_sock)) == -1)
    {
	perror("takeover: could not connect to local process");
	goto error;
    }
    DPRINTF(DEBUG_DEATH, "Successfully connected fd %d to the server\n",
	    servsock);

    /* Restore the connection state */
    if (setsockstate(remfd, &(con->ss), sizeof(struct sockstate))) {
	perror("could not set socket state");
	goto error;
    }

    con->remhost.sin_family = PF_INET;

    /* And start it connecting */
    if (connect(remfd, (struct sockaddr *)&(con->remhost),
		sizeof(con->remhost)) < 0) {
	perror("could not connect to remote host");
	goto error;
    }

    DPRINTF(DEBUG_DEATH, "Connect done for client migrate fd %d\n", remfd);

    /* XXX:  This should be moved to a processing step from the
     * event loop so we can mark the socket nonblocking.
     * Also, we need to throw it into header stripping mode
     * so we can remove the headers from the connection,
     * and then put it into normal sending mode.
     *
     */

    con->offset = compute_offset(con, remfd);

    printf("Starting range: %d\n", con->offset);

    len = sprintf(reqbuf,
		  "GET %s HTTP/1.0\n"
		  "Host: localhost\n"
		  "Connection: close\n"
		  "Accept-Encoding: gzip\n"
		  "Accept-Language: en\n"
		  "Accept-Charset: iso-8859-1,*,utf-8\n"
		  "Range: bytes=%d-9999999999999\n\n",
		  con->url, con->offset);

    rc = write(servsock, reqbuf, len);
    if (rc != len) {
	perror("could not write entire string out");
    }

    newcon = add_connection(remfd);
    if (!newcon) {
	perror("add_connection");
	goto error;
    }
    /* not really;  it's PENDING_MIGRATE, but */
    newcon->con_seqno = con->con_seqno + 1;
    newcon->state = PENDING_FILTER_HEADERS;
    bcopy(&con->remhost, &newcon->remhost, sizeof(con->remhost));
    newcon->url = con->url;
    con->url = NULL;
    newcon->serv_fd = servsock;

    getsockstate(remfd, &newcon->ss, sizeof(struct sockstate));
    newcon->headerlen = -con->offset;

    printf("Adding local connection with key %d and seqno %d\n",
	   remcon_to_key((struct remcon *)newcon), newcon->con_seqno);
    if (hash_insert(all_cons, remcon_to_key((struct remcon *)newcon),
		    (struct remcon *)newcon) == -1) {
	perror("hash_insert failed");
    }

    printf("Takeover complete!\n");

    /*
     * Note the errors in this function:
     *  - It should just do the connect, not the write;
     *  - it should throw it into PENDING_MIGRATE, and
     *    then let an interior function get the TCP sequence number,
     *    figure out the file offset,
     *    send the range request, filter out the headers in
     *    the response, and be happy.
     */
    
    return 0;

 error:
    printf("eek, an error, don't do that!\n");
    if (newcon) rem_connection(newcon);
    if (remfd != -1) close(remfd);
    if (servsock != -1) close(servsock);
    return -1;
}

/*
 * A remote server died.  Figure out if we have to take over any of
 * its connections, and process them appropriately if so.
 */

void
server_died(struct server *remserver)
{
    struct remcon *con, *nextcon;
    
    DPRINTF(DEBUG_DEATH, "Server %s died\n",
	    inet_ntoa(remserver->sockaddr.sin_addr));
    DPRINTF(DEBUG_DEATH, "Shucks!\n");

    /* Iterate through the list of connections and see which
     * of them we need to take over */

    con = remserver->con_list;

    while (1) {
	if (con == NULL) break;
	nextcon = (struct remcon *)con->head.next;
	remove_remcon(remserver, con);
	if (takeoverp(con)) {
	    takeover(con);
	}
	free_remcon(con);
	
	con = nextcon;
    }
    DPRINTF(DEBUG_DEATH, "Server death handling complete\n");
}

void
kill_localcon(struct con_info *foundcon)
{
    /* XXX:  Ensure this does not block!! */
    close(foundcon->cli_fd);
    close(foundcon->serv_fd);
    rem_connection(foundcon);
}


void
handle_announce(int sock)
{
    char inbuf[2048];
    struct sockaddr_in announce_remote_sin;
    socklen_t remlen;
    struct wedge_header *hdr;
    struct onecon *newcon;
    struct remcon tmpremcon, *foundcon;
    int ncons, key, i, rc;

    struct server *remserver;

    remlen = sizeof(announce_remote_sin);
    DPRINTF(DEBUG_ANNOUNCE, "Got an announcement\n");

    rc = recvfrom(sock, inbuf, sizeof(inbuf), 0,
	     (struct sockaddr *)&announce_remote_sin, &remlen);
    if (rc <= 0) {
	perror("recvfrom was odd");
	return;
    }
    hdr = (struct wedge_header *)inbuf;
    ncons = ntohl(hdr->numcons);
    DPRINTF(DEBUG_ANNOUNCE, "From server %s with %d connections\n",
	    inet_ntoa(announce_remote_sin.sin_addr),
	    ncons);

    if (ntohl(hdr->cmd) == CMD_DEATH) {
	struct in_addr ia;
	ia.s_addr = hdr->remhost;
	
	remserver = find_server(ia);
	if (!remserver) {
	    DPRINTF(DEBUG_ANNOUNCE, "Unknown server death %s\n",
		    inet_ntoa(ia));
	} else {
	    server_died(remserver);
	}
	return;
    }
    /* Find the server in our list of servers, or bail out */
    remserver = find_server(announce_remote_sin.sin_addr);
    if (!remserver) {
	DPRINTF(DEBUG_ANNOUNCE, "Unknown server %s\n",
		inet_ntoa(announce_remote_sin.sin_addr));
	return;
    }
    
    /* That could have taken a bit of time - update now */
    gettimeofday(&now, NULL);

    /* Now deal with the connections */
    for (i = 0; i < ncons; i++) {
	struct in_addr a;
	newcon = (struct onecon *)((inbuf + sizeof(struct wedge_header) +
				    sizeof(struct onecon) *i));
	a.s_addr = newcon->remhost;
	tmpremcon.remhost.sin_addr.s_addr = newcon->remhost;
	tmpremcon.remhost.sin_port = newcon->remport;
	tmpremcon.remhost.sin_family = AF_INET;
	tmpremcon.url = newcon->url;
	tmpremcon.ss = newcon->ss;
	tmpremcon.headerlen = ntohl(newcon->headerlen);
	tmpremcon.con_seqno = ntohs(newcon->con_seqno);
	key = remcon_to_key(&tmpremcon);

	foundcon = hash_find(all_cons, key);
	if (!foundcon) {
	    /* Case 1:  It's a new connection on a remote machine */
	    DPRINTF(DEBUG_ANNOUNCE, "NEW CON with key %d\n", key);
	    add_remcon(remserver, &tmpremcon);
	} else {
	    if (foundcon->head.type == TYPE_LOCAL) {
		/* It's one of our connections */
		DPRINTF(DEBUG_ANNOUNCE,
			"Connection steal: their seq %d ours %d\n",
			tmpremcon.con_seqno, foundcon->con_seqno);
		if (((struct con_info *)foundcon)->state == PENDING_MIGRATE) {
		    /* Case 2:  We tried to grab the connection, but someone
		     * else got it first.  Cancel the migration and let
		     * them have it.
		     */
		    kill_localcon((struct con_info *)foundcon);
		    add_remcon(remserver, &tmpremcon);
		} else {
		    /* Another server has taken over one of our connections,
		     * and we haven't yet noticed it.
		     * If their connection sequence number is higher than
		     * ours, it's a valid takeover, and we'll relinquish
		     * the connection.  Otherwise, we keep ahold of it,
		     * since they're likely just out of date and will
		     * fix themselves when they receieve our next update.
		     */
		    if (tmpremcon.con_seqno > foundcon->con_seqno) {
			kill_localcon((struct con_info *)foundcon);
			add_remcon(remserver, &tmpremcon);
		    } else {
			DPRINTF(DEBUG_ANNOUNCE,
				"      but remote version is too old\n");
		    }
		}
	    } else {
		if (foundcon->server == remserver) {
		    /* It's an existing connection for that host.
		     * Refresh it and be done */
		    DPRINTF(DEBUG_ANNOUNCE, "Refreshing con %d\n", key);
		    refresh_remcon(remserver, foundcon);
		} else {
		    /* The connection moved to a new host.  Update it. */
		    DPRINTF(DEBUG_ANNOUNCE,
			    "Moving con %d from one server to another\n", key);
		    remove_remcon(foundcon->server, foundcon);
		    insert_remcon(remserver, foundcon);
		}
	    }
	}
    }
}

/* Something has happened on the inbound request socket */

void
handle_request(int sock)
{
    struct sockaddr_in request_remote_sin;
    int remlen;
    int datasock, servsock;
    char *remtmp;
    struct con_info *clicon;

#ifdef DEBUG
    struct timeval t1, t2;

    if (debug & DEBUG_PERF) 
	gettimeofday(&t1, NULL);
#endif
    
    remlen = sizeof(request_remote_sin);
    DPRINTF(DEBUG_TCP, "About to accept\n");

    datasock = accept(sock, (struct sockaddr *)&request_remote_sin, &remlen);
    if (datasock == -1) {
	perror("error in handle_request accept");
	return;
    }
    
    remtmp = inet_ntoa(request_remote_sin.sin_addr);
    DPRINTF(DEBUG_TCP, "Accepted connection from %s\n", remtmp);

    servsock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

    /* Note:  Assumption here is no long block on calling
     * connect() to a local server.  Beware of this assumption
     * if we become concerned about performance - make the connect
     * nonblocking instead and add a state to handle it.
     */

    if (connect(servsock, (struct sockaddr *)&localhost_sock,
		sizeof(localhost_sock)) == -1)
    {
	perror("could not connect to local process");
	close(datasock);
	return;
    }
    clicon = add_connection(datasock);
    if (!clicon) {
	fprintf(stderr, "could not add a connection.  sigh.\n");
	close(datasock);
	close(servsock);
	return;
    }
    clicon->con_seqno = 0;
    clicon->state = PENDING_GET;
    bcopy(&request_remote_sin, &(clicon->remhost), sizeof(request_remote_sin));
    clicon->serv_fd = servsock;
    printf("Adding local connection with key %d\n",
	   remcon_to_key((struct remcon *)clicon));
    if (hash_insert(all_cons, remcon_to_key((struct remcon *)clicon),
		    (struct remcon *)clicon) == -1) {
	perror("hash_insert failed");
    }
#ifdef DEBUG
    if (debug & DEBUG_PERF) {
	gettimeofday(&t2, NULL);
	DPRINTF(DEBUG_PERF, "%ld microseconds to connect\n",
		timediff(&t2, &t1));
    }
#endif
}

/*
 * Graft data from the from_fd to the to_fd.
 * return -1 on any problems, 0 on success.
 */

int
graft(int from_fd, int to_fd)
{
    char buf[2048];
    int rc, cli_rc;
    
    rc = read(from_fd, buf, sizeof(buf));
    if (rc == 0) {
	fprintf(stderr, "graft:  EOF on read of fd %d\n", from_fd);
	return -1;
    }
    if (rc <= 0) {
	fprintf(stderr, "read of fd %d failed", from_fd);
	perror("");
	return -1;
    }

    cli_rc = write(to_fd, buf, rc);
    if (cli_rc == -1) {
	fprintf(stderr, "could not write to client on fd %d", to_fd);
	perror("");
	return -1;
    }
    if (cli_rc == 0) {
	fprintf(stderr, "wrote no data to client on fd %d", to_fd);
	perror("");
	return -1;
    }

    if (cli_rc != rc) {
	fprintf(stderr, "Could not write all data to client.  feh!\n");
	fprintf(stderr, "(THIS IS VERY BAD!  I DON'T WANT TO BUFFER!\n\n");
	return -1;
    }

    /* All is happy - wrote a block of data out to to the client */
    return 0;
}

/* We haven't read the client's GET request, so we don't know
 * what file they're asking for.  Read some data, parse it a bit,
 * and then set their graft bit so they go directly to the server
 * from then on.
 */

int
find_get_and_graft(struct con_info *con)
{
    char buf[2049];
    int rc, cli_rc;
    char *getloc, *endloc = NULL;
    int url_len = 0;

    int from_fd, to_fd;

    from_fd = con->cli_fd;
    to_fd = con->serv_fd;
    
    rc = read(from_fd, buf, sizeof(buf)-1);
    if (rc <= 0) {
	return -1;
    }
    buf[sizeof(buf)] = 0;

    /* The world's laziest HTTP parsing
     * This will only do for testing, not for a real
     * implementation, but hey, I'm a starving student under a deadline. */
    DPRINTF(DEBUG_HTTP, "Looking for GET location\n"); fflush(stdout);
    if ((getloc = strstr(buf, "GET")) != NULL) {
	DPRINTF(DEBUG_HTTP, "Found GET\n"); fflush(stdout);
	getloc += 3;
	while (isspace(*getloc) && (*getloc)) getloc++;
	if (*getloc != 0) {
	    endloc = getloc;
	    while (!isspace(*endloc)) {
		url_len++;
		endloc++;
	    }
	}
    }

    if (url_len != 0) {
	con->url = malloc(url_len + 1);
	if (!con->url) {
	    perror("could not allocate space for URL storage");
	    return -1;
	}
#if 0
	DPRINTF(DEBUG_HTTP, "Copying URL\n"); fflush(stdout);
#endif
	con->url[url_len] = 0;
	strncpy(con->url, getloc, url_len);
	con->state = PENDING_COUNT_HEADERS;
	if (getsockstate(con->cli_fd, &(con->ss),
			 sizeof(struct sockstate)) != 0) {
	    perror("could not get socket state\n");
	    printf("(XXX:  I don't know what to do here!)\n");
	}
    
	DPRINTF(DEBUG_HTTP, "URL:  %s\n", con->url);
    }

    cli_rc = write(to_fd, buf, rc);
    if (cli_rc == -1) {
	perror("could not write to client");
	return -1;
    }
    if (cli_rc == 0) {
	perror("wrote no data to client");
	return -1;
    }

    if (cli_rc != rc) {
	fprintf(stderr, "Could not write all data to client.  feh!\n");
	fprintf(stderr, "(THIS IS VERY BAD!  I DON'T WANT TO BUFFER!\n\n");
	return -1;
    }

    /* All is happy - wrote a block of data out to to the client */
    return 0;
}

/*
 * The return connection:  Strip the HTTP headers going from the
 * server to the client, and then start the connection going
 * as normal.
 */

int
strip_headers(struct con_info *con)
{
    char buf[2049];
    int rc;
    char *getloc;

    int from_fd, to_fd;

    from_fd = con->serv_fd;
    to_fd = con->cli_fd;
    
    rc = read(from_fd, buf, sizeof(buf)-1);
    if (rc <= 0) {
	return -1;
    }
    buf[sizeof(buf)] = 0;

    /* Look for '\n\n'.  Don't sweat it if it happens on a buf
     * boundary, because - that's right - I'm a LAZY PIG.
     * ... but deal with that eventually.
     * XXX
     */

    DPRINTF(DEBUG_HTTP, "Stripping server HTTP headers\n");
    getloc = strstr(buf, "\r\n\r\n");
    if (!getloc) { getloc = strstr(buf, "\n\r\n\r"); }
    if (getloc != NULL) {
	DPRINTF(DEBUG_HTTP, "Found end of headers\n");
	getloc += 4;
	if (write(to_fd, getloc, rc-(int)(getloc-buf)) <= 0) {
	    return -1;
	}
	con->state = READ_GET;
    } else {
	DPRINTF(DEBUG_HTTP, "No HTTP headers found in buf\n");
	return 0;
    }
    return 0;
}

int
count_headers(struct con_info *con)
{
    char buf[2049];
    int rc;
    char *getloc;

    int from_fd, to_fd;

    from_fd = con->serv_fd;
    to_fd = con->cli_fd;
    
    rc = read(from_fd, buf, sizeof(buf)-1);
    if (rc <= 0) {
	return -1;
    }
    buf[sizeof(buf)] = 0;

    /* XXX Same '\n\n' problem as above.  Please fix eventually. */

    DPRINTF(DEBUG_HTTP, "Counting server HTTP headers\n");
    getloc = strstr(buf, "\r\n\r\n");
    if (!getloc) { getloc = strstr(buf, "\n\r\n\r"); }
    if (getloc != NULL) {
	DPRINTF(DEBUG_HTTP, "Found end of headers\n");
	getloc += 4;
	con->headerlen += (getloc-buf);
	con->state = READ_GET;
	printf("Headerlen: %d\n", con->headerlen);
    } else {
	con->headerlen += rc;
    }
    if (write(to_fd, buf, rc) != rc) {
	perror("count_headers: could not write buf back");
    }
    return 0;
}

/* A client file descriptor is readable.  Deal with it.
 * return 0 on success, -1 on failure.
 */

int
client_data_available(struct con_info *con)
{
    switch(con->state) {
    case PENDING_CON:
    case PENDING_MIGRATE:
	printf("We don't deal with PENDING_CON or PENDING_MIGRATE yet\n");
	break;
    case PENDING_GET:
	/* We're looking for the GET request.
	 * XXX:  Assume it's in the first 1024 bytes... this could fail.
	 */
	if (find_get_and_graft(con) == -1) {
	    printf("closing client and server connections\n");
	    close(con->cli_fd);
	    close(con->serv_fd);
	    rem_connection(con);
	    return -1;
	}
	break;
    case READ_GET:
	if (graft(con->cli_fd, con->serv_fd) == -1) {
	    printf("closing client and server connections\n");
	    close(con->cli_fd);
	    close(con->serv_fd);
	    rem_connection(con);
	    return -1;
	}
	break;
    }
    return 0;
}

int
server_data_available(struct con_info *con)
{
    switch(con->state) {
    case PENDING_FILTER_HEADERS:
	if (strip_headers(con) == -1) {
	    printf("closing client and server connections\n");
	    close(con->cli_fd);
	    close(con->serv_fd);
	    rem_connection(con);
	    return -1;
	}
	break;
    case PENDING_COUNT_HEADERS:
	if (count_headers(con) == -1) {
	    printf("closing client and server connections\n");
	    close(con->cli_fd);
	    close(con->serv_fd);
	    rem_connection(con);
	    return -1;
	}
	break;
    default:
	if (graft(con->serv_fd, con->cli_fd) == -1) {
	    printf("closing client and server connections\n");
	    close(con->cli_fd);
	    close(con->serv_fd);
	    rem_connection(con);
	    return -1;
	}
    }
    return 0;
}

void
wedge_loop(int local_port, int remote_port)
{
    int request_sock, announce_sock;
    struct sockaddr_in request_local_sin, request_remote_sin;
    struct sockaddr_in announce_local_sin, announce_remote_sin;
    
    fd_set readfds, writefds;
    int nfds;
    struct timeval timeout, timestart, timesave;
    long t_us;

    /* Create all of our sockets */
    request_sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (request_sock < 0) {
	perror("request TCP socket");
	exit(-1);
    }

    announce_sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (announce_sock < 0) {
	perror("announce UDP socket");
	exit(-1);
    }

    bzero(&request_local_sin, sizeof(request_local_sin));
    bzero(&request_remote_sin, sizeof(request_remote_sin));
    bzero(&announce_local_sin, sizeof(announce_local_sin));
    bzero(&announce_remote_sin, sizeof(announce_remote_sin));

    request_local_sin.sin_port = htons(local_port);
    request_local_sin.sin_family = AF_INET;
    request_local_sin.sin_addr.s_addr = INADDR_ANY;
#ifndef NO_SIN_LEN
    request_local_sin.sin_len = sizeof(request_local_sin);
#endif

    if (bind(request_sock,
	     (struct sockaddr *)&request_local_sin,
	     sizeof(request_local_sin)) == -1)
    {
	perror("binding request_sock");
	exit(-2);
    }

    announce_local_sin.sin_port = htons(ANNOUNCE_PORT);
    announce_local_sin.sin_family = AF_INET;
    announce_local_sin.sin_addr.s_addr = INADDR_ANY;

    if (bind(announce_sock,
	     (struct sockaddr *)&announce_local_sin,
	     sizeof(announce_local_sin)) == -1)
    {
	perror("binding announce_sock");
	exit(-3);
    }

    if (listen(request_sock, 16) == -1) {
	perror("listen on request_sock");
	exit(-4);
    }

    /*
     * Okay, now handle requests
     */
    
    timeout.tv_sec = UPDATE_SEC;
    timeout.tv_usec = UPDATE_USEC;

    while (1)
    {
	int rc;
	struct con_info *con, *nextcon;

	/* XXX:  Note:  I'm assuming here that the server is always
	 * readable/writable, so we don't need to worry about blocking
	 * on it.  That may prove to be a false assumption;  we'll see.
	 *
	 * There's some room for optimization here, but it seems silly.
	 * The real optimization is to bundle the wedge's announcement
	 * functionality into your webserver, and use its state management
	 * and socket management for everything.  This is the
	 * "portable and slow" version.
	 */

	gettimeofday(&timestart, NULL);
	nfds = request_sock;
	if (announce_sock > nfds) {
	    nfds = announce_sock;
	}

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	
	FD_SET(announce_sock, &readfds);
	FD_SET(request_sock, &readfds);


	for (con = cons;  con != NULL; con = list_to_con(con->head.next)) {
	    FD_SET(con->cli_fd, &readfds);
	    FD_SET(con->cli_fd, &writefds);
	    FD_SET(con->serv_fd, &readfds);
	    FD_SET(con->serv_fd, &writefds);

	    if (con->cli_fd > nfds) {
		nfds = con->cli_fd;
	    }

	    if (con->serv_fd > nfds) {
		nfds = con->serv_fd;
	    }
	}

	nfds++;

	timesave = timeout;

	/* XXX:  This is not the right select logic;  We spin on
	 * the select when clients are writable but the server
	 * has no data.  The logic for this select should be:
	 *
	 * - If server has no data, don't wait on client writable.
	 *   just wait on client read, server read
	 * - If server has data, but client isn't writable,
	 *   wait only on client write
	 *
	 * This logic will take two selects per write, but since
	 * the majority of our overhead is going in & out of
	 * a user application, it shouldn't matter much anyway,
	 * and we can get a nice pipeline of requests going.
	 *
	 * But for now, spin away!
	 */

	rc = select(nfds, &readfds, &writefds, NULL, &timeout);
	gettimeofday(&now, NULL);

#ifdef DEBUG
	if (0 & debug & DEBUG_REMOTE)
	    print_allcons();
#endif

	/* No matter what, let's check what's going on */
	t_us = timesave.tv_sec * 1000000 + timesave.tv_usec;
	t_us -= timediff(&now, &timestart);
	if (t_us < 10000) {
	    if (debug & DEBUG_REMOTE) print_allcons();
	    periodic_update(announce_sock);
	    timeout.tv_sec = UPDATE_SEC;
	    timeout.tv_usec = UPDATE_USEC;
	} else {
	    timeout.tv_sec = t_us / 1000000;
	    timeout.tv_usec = t_us % 1000000;
	}

	check_connection_cache(); /* Expire old connections */
	
	if (rc == 0)
	    continue;

	if (rc == -1) {
	    perror("error in select.  Bailing");
	    exit(-1);
	}

	if (FD_ISSET(announce_sock, &readfds)) {
	    DPRINTF(DEBUG_ANNOUNCE, "got an announcement.\n");
	    handle_announce(announce_sock);
	    /* We might have tweaked the set of FDs */
	    continue;
	}

	if (FD_ISSET(request_sock, &readfds)) {
	    DPRINTF(DEBUG_TCP, "Inbound tcp connection request\n");
	    handle_request(request_sock);
	    /* And ditto here */
	    continue;
	}

	con = cons;
	while (1) {
	    if (con == NULL) {
		break;
	    }
	    nextcon = list_to_con(con->head.next);

	    if (FD_ISSET(con->cli_fd, &readfds)) {
		DPRINTF(DEBUG_TCP, "Select OK for client FD %d write\n",
			con->cli_fd);
		/* We can always write to the server */
		if (client_data_available(con) == -1) {
		    con = nextcon;
		    continue;
		}
	    }
#if 0
	    /* Warning:  This is amazingly verbose */
	    if (FD_ISSET(con->serv_fd, &readfds) &&
		!FD_ISSET(con->cli_fd, &writefds)) {
		DPRINTF(DEBUG_TCP, "Server has data, client can't accept\n");
	    }
#endif
	    if (FD_ISSET(con->serv_fd, &readfds) &&
		FD_ISSET(con->cli_fd,  &writefds)) {
		/* Server has data and client is writable */
		DPRINTF(DEBUG_TCP, "Select returns OK for server FD %d\n",
			con->serv_fd);
		if (server_data_available(con) == -1) {
		    con = nextcon;
		    continue;
		}
	    }
	    /* Otherwise, let it buffer up in our socket buffers */
	    con = nextcon;
	}

	/* Done handling the available requests */
    }
}

/*
 * set up the cached copy of the localhost socket
 */

void
setup_localhost()
{
    bzero(&localhost_sock, sizeof(localhost_sock));
    localhost_sock.sin_family = AF_INET;
#ifndef NO_SIN_LEN
    localhost_sock.sin_len = sizeof(localhost_sock);
#endif
    localhost_sock.sin_port = htons(remote_port);
    inet_aton("127.0.0.1", &(localhost_sock.sin_addr));
}

void
usage()
{
    fprintf(stderr,
	    "usage:  wedge  [-h] [-d <debug>] [-p port] [-r remote port]\n"
	    );
}

void
help()
{
    usage();
    fprintf(stderr,
	    "        -h  ................. help (this message)\n"
	    "        -d <debug level> .... enable debugging, if compiled\n"
	    "        -p port ............. local port to bind to\n"
	    "        -r remote port ...... remote port to splice to\n"
	    );
}

int
main(int argc, char **argv)
{
    extern char *optarg;
    extern int optind;
    int ch;
    char *server_file = DEFAULT_SERVER_FILE_NAME;
    
    while ((ch = getopt(argc, argv, "hd:p:r:s:")) != -1)
	switch (ch) {
	case 'd':
#ifdef DEBUG
	    debug = atoi(optarg);
#else
	    fprintf(stderr, "not compiled with debugging support\n");
	    exit(-1);
#endif
	    break;
	case 's':
	    server_file = optarg;
	    break;
	case 'p':
	    local_port = atoi(optarg);
	    break;
	case 'r':
	    remote_port = atoi(optarg);
	    break;
	case 'h':
	    help();
	    exit(0);
	default:
	    usage();
	    exit(-1);
	}
    argc -= optind;
    argv += optind;
    
#ifdef DEBUG
    if (!debug)
#endif
	daemon(0,0);

    /* Set up the localhost_sock */
    setup_localhost();

    all_cons = hash_create(DEFAULT_HASHBUCKETS, NULL);
    if (!all_cons) {
	perror("hash_create remote connections table");
	exit(-1);
    }

    read_servers(server_file);

    wedge_loop(local_port, remote_port);
    
    exit(0);
}
