#ifndef _WEDGE_H_
#define _WEDGE_H_
/*
 * The protocol for letting two wedges share a list of open connections
 *
 * Wedges handle current connections in soft-state form.
 * - Should the originating wedge supply an "estimated timeout"
 *   for how long the remote wedges should keep things around,
 *   or should it be a fixed timeout/refresh period?
 *
 */

#include <sys/types.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>

/*
 * The socket state
 */

struct sockstate
{
  char ss_data[512];
};


/* Laziness for portability */
#ifdef __linux__
#define NO_SIN_LEN 1
#endif

#define UPDATE_SEC 5
#define UPDATE_USEC 0

#define CONNECTION_CACHE_TIMEOUT_USEC 10000000 /* Ten seconds */

#define ANNOUNCE_PORT 9939
#define MAXCONNS 256
#define DEFAULT_HASHBUCKETS 1009

#define DEFAULT_LOCAL_TCP_PORT 8080  /* Where we bind to */
#define DEFAULT_REMOTE_PORT      80  /* Where we relay to */

#define DEFAULT_SERVER_FILE_NAME "servers"

/* States for the con_info struct */

#define PENDING_CON     0	/* Not even connected */
#define PENDING_MIGRATE 1	/* Trying to migrate it */
#define PENDING_GET     2       /* Haven't received a URL yet */
#define PENDING_FILTER_HEADERS 3       /* Filtering headers back to client */
#define PENDING_COUNT_HEADERS 4
#define READ_GET	5	/* Did we see a GET request yet? */

/* Types for listels */
#define TYPE_LOCAL 0   /* It's a struct con_info */
#define TYPE_REMOTE 1  /* It's a struct remcon */

/* Commands for wedge announcements */
#define CMD_ANNOUNCE 0
#define CMD_DEATH    1

/*
 * What all of my list-addressable elements look like
 */
struct listel {
    int type;
    struct listel *prev, *next;
    int key;
    /* More stuff goes here */
};

/*
 * The local connections table
 */

struct con_info {
    struct listel head;
    struct sockaddr_in remhost; /* Snapshot of the client socket */
    u_int16_t con_seqno; /* monotonically increasing as takeovers happen */
    int cli_fd;			/* Remote client */
    int serv_fd;			/* FD to the local server */
    int state;			/* Current state */
    int headerlen;		/* Size of HTTP headers on initial req. */
    struct sockstate ss;
    char *url;
};

/*
 * The remote connections table
 *
 * Remote connections are announced on a periodic basis by their
 * server machine (every UPDATE_SEC/UPDATE_USEC).  We keep them
 * around and flush them out of our cache every
 * CONNECTION_CACHE_TIMEOUT_SEC/USEC.  We assume that
 * UPDATE_SEC and TIMEOUT are short relative to the average connection
 * duration, so that we don't have a large number of extra connections
 * in the cache that no longer exist on the servers.
 *
 * When a server dies, we need to iterate through all of its
 * connections.  Therefore, we maintain a connection list
 * on a per-remote-server basis.  Because the number of
 * remote servers is small, we maintain them in a linked
 * list.
 *
 * We store the per-server remote connections in a linked list
 * which is also addressed by a hash table.  When a connection
 * announcement comes in, we check for it in the hash table
 * of connections.  This hash table must index:
 *
 *  - Connections we're attempting to migrate
 *  - Connections belonging to other machines
 *
 * If the connection is not found, we insert it at the end of the
 * connection list for the announcing machine.
 * If it is found in the announcer's list, we move it to the
 * end of the list and update its expiration time.
 * If it is in the the list for a different machine, we delete it
 * from the other machine's list and insert it anew in the
 * announcer's machine.
 * If it is in the list connections we're attempting to migrate,
 * then we know that someone else has successfully taken it over,
 * so we cancel our migration, and add it to the announcer's
 * connection list.
 *
 * Occasionally (some fraction of connection cache timeout?),
 * we run through the lists and determine which of the connections
 * are eligible for timeouts, and remove them from the cache
 * and hash.
 *
 * Things to go in the hash:
 *   - Remote machine, port
 *   - ISN
 *   - URL
 *
 * Should I just transmit the connection hash with its announcement?
 * Nah.  We can do that later if we think it matters.
 */

struct remcon {
    struct listel head;
    struct sockaddr_in remhost;
    u_int16_t con_seqno; /* monotonically increasing as takeovers happen */
    struct timeval expire_time;
    struct server *server;
    struct sockstate ss;
    char *url;
    int headerlen;
    int offset; /* Current offset into URL */
};

struct hashtype; /* Declared in hash.h */

/*
 * Table of remote machines
 */
struct server {
    struct listel head;
    struct sockaddr_in sockaddr;
    struct remcon *con_list;
    struct remcon *con_tail;
};

/*
 * The protocol we use for sharing data between wedges
 */

struct onecon {
    u_int32_t remhost;   /* Network byte order */
    u_int16_t remport;
    u_int16_t con_seqno; /* monotonically increasing as takeovers happen */
    struct sockstate ss;
    int32_t headerlen; /* Need to sync;  network byte order */
    char url[25];       /* This could be problematic, but... */
};

struct wedge_header {
    int version;         /* Uhhh.  For the same of completeness? :) */
    int cmd;             /* Either 'announce' or 'death' */
    u_int32_t remhost;   /* If 'death', the remote host.  Otherwise ignore */
    int numcons;         /* Number of 'struct onecons' in this packet */
    /* XXX - A serious implementation would have a checksum. :) */
    /* Followed by many struct onecons */
};

#ifndef max
#define max(a, b) (a > b ? a : b)
#endif

#endif /* _WEDGE_H_ */
