/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: latency_handler.cc,v 1.2 2002/09/04 18:41:46 jsalz Exp $
 *
 * A handler which shapes incoming and/or outgoing traffic by limiting it to
 * a certain number of bytes per timeslice.
 *
 */

#include "config.h"

#include "latency_handler.hh"

DEFINE_HANDLER(latency, latency_handler, AF_INET, SOCK_STREAM);

latency_handler::latency_handler(init_context& ctxt, bool plumb) :
    flow_handler(ctxt, plumb),
    rtimer(this, &latency_handler::rfire),
    wtimer(this, &latency_handler::wfire)
{
    double adelay;

    get_arg("rdelay", adelay, 0.1, 0.0, 1e6);
    rdelay = (long long)(adelay * 1e6 + 0.5);
    get_arg("wdelay", adelay, 0.1, 0.0, 1e6);
    wdelay = (long long)(adelay * 1e6 + 0.5);

    ts_debug_1("rdelay is %d us", rdelay);
    ts_debug_1("wdelay is %d us", wdelay);
}

int latency_handler::write(data d) {
    time now = timer::now();

    wpkts.push_back(pkt(now + wdelay, d));
    if (!wtimer.is_armed())
	wtimer.arm(now + wdelay, timer::FROM_EPOCH);

    return 0;
}

bool latency_handler::avail(flow_handler *from, data d) {
    time now = timer::now();

    rpkts.push_back(pkt(now + rdelay, d));
    if (!rtimer.is_armed())
	rtimer.arm(now + rdelay, timer::FROM_EPOCH);

    return true;
}

bool latency_handler::may_exit() {
    return flow_handler::may_exit() && wpkts.empty();
}

void latency_handler::may_write(flow_handler *from, bool may) {
    flow_handler::may_write(from, may);
    if (may == true)
	wfire(wtimer);
}

void latency_handler::may_avail(bool may) {
    flow_handler::may_avail(may);
    if (may == true)
	rfire(rtimer);
}

void latency_handler::wfire(my_timer& t) {
    list<pkt>::iterator i;
    time now = timer::now();

    for (i = wpkts.begin();
	 i != wpkts.end() && i->first <= now && may_write_now();
	 ++i)
    {
	downstream[0]->write(i->second);
    }
    wpkts.erase(wpkts.begin(), i);

    if (!wpkts.empty() && may_write_now())
	t.arm(wpkts.begin()->first, timer::FROM_EPOCH);
}

void latency_handler::rfire(my_timer& t) {
    list<pkt>::iterator i;
    time now = timer::now();

    for (i = rpkts.begin();
	 i != rpkts.end() && i->first <= now && may_avail_now();
	 ++i)
    {
	upstream->avail(this, i->second);
    }
    rpkts.erase(rpkts.begin(), i);

    if (!rpkts.empty() && may_avail_now())
	t.arm(rpkts.begin()->first, timer::FROM_EPOCH);
}
