/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: log_handler.cc,v 1.5 2002/09/02 18:38:16 jsalz Exp $
 *
 * A handler which logs events to stderr or a file.
 *
 */

#include "config.h"

#include "log_handler.hh"

DEFINE_HANDLER(log, log_handler, AF_INET, SOCK_STREAM);
HANDLER_USAGE(log_handler,

"Logs network events.\n"
"\n"
"  -readable   Make output human-readable (default)\n"
"  -cstring    Output data as C-strings\n"
"  -hex        Output data as hex strings\n"
"\n"
"  -file=F     Output to file (use stdout if not specified)\n"
"\n"
"  -write      Log only outgoing bytes\n"
"  -read       Log only incoming bytes\n"
"  -all        Log lots of stuff\n"

	      );

int log_handler::count = 0;
log_handler::logmap log_handler::logs;

log_handler::log_handler(const init_context& ctxt, bool plumb = true) :
    flow_handler(ctxt, plumb), id(++count)
{
    outname = get_arg("file");
    if (outname.length()) {
	logmap::const_iterator i = logs.find(outname);
	if (i == logs.end()) {
	    outfile = new ofstream(outname.c_str(), ios::ate);
	    if (!outfile->is_open()) {
		delete outfile;
		outfile = 0;
		outname.erase();
		ts_error("Unable to open file %s for writing; using stdout",
			 outname.c_str());
	    } else {
		logs.insert(logpair(outname, this));
	    }
	}
    }

    int modecount = 0;
    mode = DEFAULT;
    if (get_flag("readable")) ++modecount, mode = READABLE;
    if (get_flag("cstring")) ++modecount, mode = CSTRING;
    if (get_flag("hex")) ++modecount, mode = HEX;
    if (modecount > 1)
	ts_error("More than one mode specified");

    if (get_flag("write") || get_flag("read")) {
	log_write = get_flag("write");
	log_avail = get_flag("read");
    } else {
	log_write = log_avail = true;
    }
    if (get_flag("all")) {
	log_write = log_avail = log_all = true;
    }

    if (outfile)
	out = outfile;
    else
	out = &cout;

    log("Opened");
}

log_handler::~log_handler()
{
    log("Destroyed");

    if (outfile) {
	logs.erase(outname);
	if (logs.find(outname) == logs.end()) {
	    ts_debug_1("Done with %s; closing", outname.c_str());
	    delete outfile;
	}
    }
}

int log_handler::write(data d) {
    if (log_write)
	log("Write: " + to_string(d));
    return flow_handler::write(d);
}

bool log_handler::avail(flow_handler *from, data d) {
    if (log_avail)
	log("Avail: " + to_string(d));
    return flow_handler::avail(from, d);
}

int log_handler::shutdown(bool r, bool w) {
    if (log_all)
	log(string("Shutdown: read=") + (r ? "t" : "f") + "; write=" + (w ? "t" : "f"));
    return flow_handler::shutdown(r, w);
}
    

void log_handler::log(string str) {
    (*out) << to_string(id) << "> " << str << endl;
}

string log_handler::to_string(int i) {
    char buf[32];
    sprintf(buf, "%d", i);
    return buf;
}

string log_handler::to_string(data d) {
    switch (mode) {
      case READABLE:
	{
	    string out = "[";
	    int dots = 0;

	    string readable;

	    for (unsigned int i = 0; i < d.length(); ++i) {
		unsigned char ch = *(d.bits() + i);
		if (ch >= ' ' && ch <= '~') {
		    readable += ch;
		} else if (ch == '\n') {
		    readable += "\\n";
		} else if (ch == '\r') {
		    readable += "\\r";
		} else {
		    if (readable.length()) {
			if (readable.length() >= 4) {
			    dots = 0;
			    out += readable;
			}
			readable.erase();
		    }
		    if (dots < 3) {
			out += ".";
			++dots;
		    }
		}
	    }
	    out += readable;
	    out += "] - " + to_string(d.length());
	    return out;
	}

      case CSTRING:
	{
	    string out = "\"";
	    for (unsigned int i = 0; i < d.length(); ++i) {
		unsigned char ch = *(d.bits() + i);
		if (ch >= ' ' && ch <= '~')
		    out += ch;
		else {
		    char buf[8];
		    sprintf(buf, "\\x%02X", ch);
		    out += buf;
		}
	    }
	    out += "\"";
	    return out;
	}

      case HEX:
	{
	    string out = "x\"";
	    for (unsigned int i = 0; i < d.length(); ++i) {
		char byte[3];
		sprintf(byte, "%02X", (unsigned char)*(d.bits() + i));
		out += byte;
	    }
	    out += "\"";
	    return out;
	}
    }

    return "UNKNOWN MODE";
}
