/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: shape_handler.cc,v 1.5 2002/09/02 18:38:16 jsalz Exp $
 *
 * A handler which shapes incoming and/or outgoing traffic by limiting it to
 * a certain number of bytes per timeslice.
 *
 */

#include "config.h"

#include "shape_handler.hh"

DEFINE_HANDLER(shape, shape_handler, AF_INET, SOCK_STREAM);

shape_handler::shape_handler(init_context& ctxt, bool plumb) :
    flow_handler(ctxt, plumb),
    rtimer(this, &shape_handler::rfire),
    wtimer(this, &shape_handler::wfire)
{
    double aslice;
    get_arg("slice", aslice, 1.0, 0.0, 1e6);
    slice = (long long)(aslice * 1e6 + 0.5);

    get_arg("imeter", rmeter, 1024, 0, INT_MAX);
    get_arg("ometer", wmeter, 1024, 0, INT_MAX);

    ts_debug_1("slice is %lld us", slice);
    ts_debug_1("imeter is %d bytes", rmeter);
    ts_debug_1("ometer is %d bytes", wmeter);
}

bool shape_handler::avail(flow_handler *from, data d) {
    if (d.length() == 0 || rmeter == 0)
	return upstream->avail(this, d);

    // Arm timer if not already armed
    if (!rtimer.is_armed())
	rtimer.arm(slice);

    unsigned int can_avail = rmeter - rbytes;

    if (can_avail >= d.length()) {
	rbytes += d.length();
	return upstream->avail(this, d);
    }

    rbytes = rmeter;
    rbuf.append(d.bits() + can_avail, d.length() - can_avail);

    if (can_avail == 0)
	return 1;
    else
	return upstream->avail(this, data(d.bits(), can_avail));
}

void shape_handler::rfire(my_timer& t) {
    rbytes = 0;
    if (rbuf.length() == 0)
	return;

    ts_debug_1("Arming timer for %d", slice);
    rtimer.arm(slice);

    if (rmeter > (int)rbuf.length()) {
	rbytes = rbuf.length();
	upstream->avail(this, rbuf);
	rbuf.erase();
    } else {
	rbytes = rmeter;
	upstream->avail(this, data(rbuf.data(), rmeter));
	rbuf.erase(0, rmeter);
    }
}

// Nearly an exact copy of the above...
int shape_handler::write(data d) {
    if (d.length() == 0 || wmeter == 0)
	return downstream[0]->write(d);

    // Arm timer if not already armed
    if (!wtimer.is_armed())
	wtimer.arm(slice);

    unsigned int can_write = wmeter - wbytes;

    if (can_write >= d.length()) {
	wbytes += d.length();
	return downstream[0]->write(d);
    }

    wbytes = wmeter;
    wbuf.append(d.bits() + can_write, d.length() - can_write);

    if (can_write == 0)
	return 0;
    else
	return downstream[0]->write(data(d.bits(), can_write));
}

void shape_handler::wfire(my_timer& t) {
    wbytes = 0;
    if (wbuf.length() == 0)
	return;

    ts_debug_1("Arming timer for %d", slice);
    wtimer.arm(slice);

    if (wmeter > (int)wbuf.length()) {
	wbytes = wbuf.length();
	downstream[0]->write(wbuf);
	wbuf.erase();
    } else {
	wbytes = wmeter;
	downstream[0]->write(data(wbuf.data(), wmeter));
	wbuf.erase(0, wmeter);
    }
}

