/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: socks_handler.cc,v 1.8 2002/09/04 16:16:43 jsalz Exp $
 *
 * A simple SOCKS flow handler.
 *
 */

#include "config.h"

#include "socks_handler.hh"

DEFINE_HANDLER(socks, socks_handler, AF_INET, SOCK_STREAM);
HANDLER_USAGE(socks_handler,
"A transparent SOCKS proxy handler.\n"
"Supports only unauthenicated connections.\n"
"\n"
"  -host   IP address of proxy (default 127.0.01)\n"
"  -port   Proxy port (default 1080)\n"
	      );

socks_handler::socks_handler(init_context& ctxt, bool plumb) :
    buf_handler(ctxt, plumb)
{
    state = NOTHING;

    string host = get_arg("host");
    string port = get_arg("port");

    if (host.length() == 0)
	host = "127.0.0.1";
    if (port.length() == 0)
	port = "1080";
    
    proxy = address::aton(host, port);
    if (!proxy)
	ts_error("Invalid host/port");
}

int socks_handler::bind(address a) {
    ts_error("SOCKS proxy handler does not support bind");
    return -EINVAL;
}
int socks_handler::listen(int backlog) {
    ts_error("SOCKS proxy handler does not support listen");
    return -EINVAL;
}
acceptret socks_handler::accept() {
    ts_error("SOCKS proxy handler does not support accept");
    return acceptret();
}

bool socks_handler::avail(flow_handler *h, data d) {
    if (state == ESTABLISHED)
	return upstream->avail(this, d);

    if (d.length() == 0) {
	// EOF
	upstream->avail(this, d);
	return true;
    }
	
    inbuf.append(d.bits(), d.length());

    switch (state) {
      case WAITING_METHOD:
	{
	    if (inbuf.length() < 2)
		return true;

	    ts_debug_1("Got version/method");
	    if (inbuf[0] != 5) {
		ts_error("Invalid SOCKS version");
		reset();
		return false;
	    }
	    if (inbuf[1] != 0) {
		ts_error("Invalid SOCKS method");
		reset();
		return false;
	    }
		
	    inbuf = string(inbuf, 2);

	    // Send the SOCKS request
	    string request("\x05\x01\x00\x01", 4);

	    assert(peer.addrlen() == sizeof(sockaddr_in));
	    const sockaddr_in *addr = (const sockaddr_in *)(peer.addr());
	    request.append((const char *)&addr->sin_addr.s_addr, 4);
	    request.append((const char *)&addr->sin_port, 2);
	    bufwrite(request);

	    state = WAITING_REPLY;
	    return true;
	}

      case WAITING_REPLY:
	{
	    if (inbuf.length() < 4)
		return true;

	    ts_debug_1("Got reply");
	    if (inbuf[0] != 5) {
		ts_error("Invalid SOCKS version");
		reset();
		return false;
	    }

	    if (inbuf[1] != 0) {
		ts_error("SOCKS error #%d", (int)inbuf[1]);
		reset();
		return false;
	    }

	    if (inbuf[3] != 1) {
		ts_error("No IP address given");
		reset();
		return false;
	    }

	    inbuf = string(inbuf, 4);

	    state = WAITING_REPLY_2;
	    return true;
	}

      case WAITING_REPLY_2:
	{
	    if (inbuf.length() < 6)
		return true;

	    state = ESTABLISHED;

	    ts_debug_1("Established! (inbuf length is %d)", inbuf.length());
	    upstream->connected(this, true);

	    if (inbuf.length() > 6)
		upstream->avail(this, data(inbuf.data() + 6, inbuf.length() - 6));

	    inbuf.erase();
	    return true;
	}

      default:
	ts_error("Unexpected state %d", state);
    }

    return true;
}

int socks_handler::connect(address a) {
    peer = a;
    int ret = downstream[0]->connect(proxy);

    if (ret == 0) {
	ts_debug_1("Connecting to %s...", proxy.c_str());
	state = CONNECTING;
    }

    return ret;
}

void socks_handler::connected(flow_handler *from, bool success) {
    ts_debug_1("Connected; success=%d", success);
    if (state == CONNECTING && success) {
	ts_debug_1("Connected; sending version/method");
	static const string version("\x05\x01\x00", 3);
	state = WAITING_METHOD;
	bufwrite(version);
    } else {
	ts_debug_1("Unable to connect");
	reset();
    }
}

int socks_handler::write(data d) {
    if (state != ESTABLISHED) {
	ts_debug_1("Broken pipe!");
	return -EPIPE;
    }

    return bufwrite(d);
}
