/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: flow_handler.hh,v 1.28 2002/10/08 19:03:33 snoeren Exp $
 *
 * The base class for all handlers, and the address and data classes used
 * by the handler interface.
 *
 */

#ifndef HH_HANDLER
#define HH_HANDLER

#include <cassert>
#include <climits>
#include <cmath>
#include <cfloat>
#include <string>
#include <vector>
#include <map>
#include <typeinfo>
#include <fstream>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <errno.h>
#include <netdb.h>

#include "tesla/tesla.h"
#include "tesla/serial.hh"

using namespace std;

// address: a string-like structure *with* the same copy/free semantics for the
// buffer as string has
class address {
    string bits;

public:
    address() {}
    address(const void *addr, socklen_t len) : bits((const char *)(addr), len) {}
    explicit address(const string& s) : bits(s) {}

    // Allow default copy and assignment constructors (which just copy the string)

    const sockaddr *addr() const { return (const sockaddr *)(bits.data()); }
    socklen_t addrlen() const { return bits.length(); }

    string as_string() const { return bits; }

    operator bool() const { return bits.length() != 0; }
    const char *c_str() const {
	static char tmpbuf[1024];
	const sockaddr_in *addr = (const sockaddr_in *)bits.data();
	const char *buf = inet_ntoa(addr->sin_addr);
	if (buf == 0) buf = "???";
	sprintf(tmpbuf, "%s:%d", buf, ntohs(addr->sin_port));
	return tmpbuf;
    }

    static address aton(string in, string port) {
	sockaddr_in addr;
	hostent *ent;

	addr.sin_family = AF_INET;
	addr.sin_port = htons(atoi(port.c_str()));

	if ((ent = gethostbyname(in.c_str())) == NULL) {
	    ts_debug_1("Invalid hostname \"%s\"", in.c_str());
	    return address();
	}
	memcpy(&addr.sin_addr.s_addr, ent->h_addr, ent->h_length);
	return address(&addr, sizeof addr);
    }
};
inline void freeze(oserial& o, const address& a) { o << a.as_string(); }
inline void unfreeze(iserial& i, address& a) {
    string s;
    i >> s;
    a = address(s);
}

// data: an *un-owned* data/length pair.  Unlike string, has absolutely no
// semantics regarding the copying or freeing of the data; this is essential
// a pair<const char*, unsigned int>.
class data
{
private:
    const char *dbits;
    unsigned int dlength;
    const sockaddr *daddr;
    const socklen_t daddrlen;
    const int derror;

public:
    data() : dbits(0), dlength(0), daddr(0), daddrlen(0), derror(0) {}
    data(const char *str) : dbits(str), dlength(strlen(str)), daddr(0), daddrlen(0), derror(0) {}
  data(const char *_dbits, unsigned int _dlength, int _derror = 0,
	 const sockaddr *_daddr = 0, socklen_t _daddrlen = 0) :
	dbits(_dbits), dlength(_dlength), daddr(_daddr), daddrlen(_daddrlen), derror(_derror) {}
    data(const string& s) : dbits(s.data()), dlength(s.length()), daddr(0), daddrlen(0), derror(0) {}

    // Copy constructor generalization
    data(const data& other, int start = 0, int count = -1) :
	dbits(other.dbits + start),
	dlength(count == -1 ? other.dlength - start : count),
	daddr(other.daddr),
	daddrlen(other.daddrlen),
	derror(other.derror)
    {
    }

    const char *bits() const { return dbits; }
    unsigned int length() const { return dlength; }
    const sockaddr *addr() const { return daddr; }
    socklen_t addrlen() const { return daddrlen; }
    int error() const { return derror; }

    operator string() const { return dlength == 0 ? string() : string(dbits, dlength); }
    const char & operator[](unsigned int index) const { return dbits[index]; }
};

class flow_handler;
class acceptret {
public:
    acceptret() : h(0) {}
    acceptret(flow_handler *_h, address _addr) : h(_h), addr(_addr) {}

    operator bool() const { return h; }

    flow_handler *const h;
    const address addr;
};

class flow_handler {
public:
    class info;
    static int __set_usage_info(info& i, string usage);

public:
    class init_context;
    typedef flow_handler *(*create_func)(init_context &);
    typedef map<string, string> dict;

    class info {
    private:
	string name;
	string clazz;
	string usage;
	int domain;
	int type;
	create_func cf;

	dict args;

	// Default copy constructor/assignment

    public:        
	info(const string& _name, const string& _clazz, const string& _usage,
	     int _domain, int _type,
	     create_func _cf) :
	    name(_name), clazz(_clazz), usage(_usage),
	    domain(_domain), type(_type), cf(_cf)
	{
	}

	string get_name() const { return name; }
	string get_clazz() const { return clazz; }
	string get_usage() const { return usage; }
	int get_domain() const { return domain; }
	int get_type() const { return type; }
	create_func get_cf() const { return cf; }
	flow_handler *create(init_context& ctxt) const {
	    return (*cf)(ctxt);
	}
	dict& get_args() { return args; }
	const dict& get_args() const { return args; }

	friend void flow_handler::__set_usage_info(info&, string usage);
    };

public:
    static const int PASS_FD = 99999;

    class init_context {
    private:
	const info& inf;
	flow_handler& upstream;
	int domain, type;
	int level;
	bool accepted;
	iserial *ser;
	
    public:
	init_context(const info& _inf, flow_handler& _upstream, int _domain, int _type,
		     int _level, bool _accepted = false) :
	    inf(_inf), upstream(_upstream), domain(_domain), type(_type),
	    level(_level), accepted(_accepted), ser(0) {
	}

	init_context(const info& _inf, iserial& _ser) :
	    inf(_inf), upstream(*(flow_handler*)0), domain(_inf.get_domain()), type(_inf.get_type()),
	    level(-1), accepted(false), ser(&_ser)
	{
	}

	const info& get_info() const { return inf; }
	int get_domain() const { return domain; }
	int get_type() const { return type; }
	int get_level() const { return level; }
	flow_handler *plumb(flow_handler& me, int domain, int type) const { return flow_handler::plumb(me, domain, type, level); }
	bool is_accepted() const { return accepted; }

	iserial *serial() const { return ser; }

	flow_handler& get_upstream() const { return upstream; }
    };

private:
    int domain;
    int type;
    int level;
    const info& inf;

    // Prohibit copying
    template<class T> flow_handler& operator=(T&);

    typedef map<string, const info*> infomap;
    typedef vector<const info*> infovec;
    static infovec handlers;
    static infomap *reg_handlers;
    static double nan;

protected:
    bool can_write;
    bool can_avail;

    flow_handler* upstream;
    vector<flow_handler*> downstream;

    flow_handler(const init_context& ctxt, bool plumb = true);

    static void register_handler(const info&);

    static int encode_pass_fd(int fd, string& out) {
	out.append((const char *)&fd, sizeof fd);
	return PASS_FD;
    }

    virtual void dump_children(string& out, int indent) {
	for (vector<flow_handler*>::iterator i = downstream.begin(); i != downstream.end(); ++i)
	    (*i)->dump(out, indent);
    }

    flow_handler *plumb(int domain, int type, int fd = -1);

    const dict& get_args() const { return inf.get_args(); }
    string get_arg(string key) const {
	string out;
	get_arg(key, out);
	return out;
    }

    enum { INVALID_INT = INT_MIN };
    bool get_arg(string key, string& val, string def = "") const;
    bool get_arg(string key, int& val, int def = INVALID_INT,
		 int min = INT_MIN, int max = INT_MAX) const;
    bool get_arg(string key, double& val, double def = nan,
		 double min = DBL_MIN, double max = DBL_MAX) const;
    bool get_flag(string key) const;

    bool may_write_now() { return can_write; }
    bool may_avail_now() { return can_avail; }

public:
    virtual ~flow_handler() {
	for (vector<flow_handler*>::iterator i = downstream.begin(); i != downstream.end(); ++i)
	    delete *i;
    }

    static void make_handlers();

    flow_handler& get_upstream() const { return *upstream; }
    void set_upstream(flow_handler *_upstream) { upstream = _upstream; }

    virtual void dump(string& out, int indent = 0) {
	char ch[64];

	for (int i = 0; i < indent; ++i) {
	    out.append("   ");
	}
	out.append(" - ").append(get_info().get_name());
	out.append("; me=");
	sprintf(ch, "%p", this);
	out.append(ch).append("; upstream=");
	sprintf(ch, "%p", &get_upstream());
	out.append(ch).append("\n");

	dump_children(out, indent + 1);
    }

    const info& get_info() const { return inf; }
    int get_domain() const { return domain; }
    int get_type() const { return type; }
    int get_level() const { return level; }

    virtual int connect(address a);
    virtual int bind(address a);
    virtual acceptret accept();
    virtual int close();
    virtual void accept_ready(flow_handler *from);
    virtual void connected(flow_handler *from, bool success);
    virtual bool avail(flow_handler *from, data d);
    virtual int write(data d);
    virtual int shutdown(bool r, bool w);
    virtual int listen(int backlog);
    virtual address getsockname();
    virtual address getpeername();
    virtual void may_write(flow_handler *from, bool may);
    virtual void may_avail(bool may);

    virtual bool may_exit();

    virtual bool save_state(oserial& out) const {
	return false;
    }

    virtual string getsockopt(int level, int optname, int maxlen);
    virtual int setsockopt(int level, int optname, string value);
    virtual int ioctl(string target, int optname, string value, string& out);

    bool messages_enabled() { return true; }
    void message(int msgname, string value);
    void message_fd(int msgname, int fd);

    static flow_handler *plumb(flow_handler& upstream, int domain, int type, int level = 0);

    friend void freeze(oserial& out, const flow_handler& h);
    friend void unfreeze(iserial& in, flow_handler*& h);
};

void freeze(oserial& out, const flow_handler& h);
void unfreeze(iserial& in, flow_handler*& h);



#define DECLARE_HANDLER \
public: \
    static info _info; \
    static flow_handler *_create(init_context&); \
    static int _inited;

#define DECLARE_DUMMY_HANDLER \
public: \
    static info _info;

#define DEFINE_HANDLER(name, clazz, domain, type) \
flow_handler *clazz::_create(init_context& ctxt) { return new clazz(ctxt); } \
flow_handler::info clazz::_info(#name, #clazz, "", domain, type, clazz::_create); \
int clazz::_inited = (ts_debug_2("Registering flow_handler %s", #clazz), register_handler(_info), 1)

#define HANDLER_USAGE(clazz, usage) static int __ts_##clazz##_usage = flow_handler::__set_usage_info(clazz::_info, usage);

#define DEFINE_DUMMY_HANDLER(clazz) \
flow_handler::info clazz::_info(#clazz, #clazz, -1, -1, 0);





#endif
