// -*- mode: C++; -*-

#ifndef NMSTL_H_SERIAL
#define NMSTL_H_SERIAL

#include <iostream>
#include <string>
#include <fstream>

inline string to_string(const int& i) {
    char buf[16];
    sprintf(buf, "%d", i);
    return buf;
}

class oserial {
    oserial(const oserial&);
    oserial& operator = (const oserial&);

    string err;
    bool my_good;

protected:
    oserial() : my_good(true) {}
    virtual ~oserial() {}

public:
    virtual void write(const void *bytes, unsigned int len) = 0;
    virtual bool good() const = 0;
    
    operator bool() const {
	return my_good && good();
    }

    string error() const {
	return err;
    }
    void set_error(const string& s) {
	my_good = false;
	err = s;
    }

    void write_type(const char *type) {
	unsigned int typelen = strlen(type);
	write((const char *)&typelen, sizeof typelen);
	write(type, typelen);
    }

    template<class T>
    inline oserial& operator << (const T& t) {
	freeze(*this, t);
	return *this;
    }
};

class oserialstream : public oserial {
    oserialstream(const oserialstream&);
    oserialstream& operator = (const oserialstream&);

    ostream& out;

public:
    void write(const void *bytes, unsigned int len) {
	out.write(static_cast<const char *>(bytes), len);
    }
    bool good() const { return out; }

public:
    oserialstream(ostream& out) : out(out) {}
};

class oserialstring : public oserial {
    oserialstring(const oserialstring&);
    oserialstring& operator = (const oserialstring&);

    string out;

public:
    oserialstring() {}

    void write(const void *bytes, unsigned int len) {
	out.append(static_cast<const char *>(bytes), len);
    }
    bool good() const { return true; }

    string str() { return out; }
};



#define NMSTL_INTEGER_FREEZE(T) \
inline void freeze(oserial& o, const T& t) { \
    unsigned int len = sizeof t; \
    o.write_type(#T); \
    o.write((const char *)&len, sizeof len); \
    o.write((const char *)&t, len); \
}
NMSTL_INTEGER_FREEZE(bool);
NMSTL_INTEGER_FREEZE(char);
NMSTL_INTEGER_FREEZE(unsigned char);
NMSTL_INTEGER_FREEZE(short);
NMSTL_INTEGER_FREEZE(unsigned short);
NMSTL_INTEGER_FREEZE(int);
NMSTL_INTEGER_FREEZE(unsigned int);
NMSTL_INTEGER_FREEZE(long);
NMSTL_INTEGER_FREEZE(unsigned long);
inline void freeze(oserial& o, const string& s) {
    unsigned int length = s.length();
    o.write_type("[string]"); \
    o.write((const char *)&length, sizeof length);
    o.write((const char *)s.data(), length);
}










class iserial {
    iserial(const iserial&);
    iserial& operator = (const iserial&);

    string err;
    bool my_good;

public:
    void check_type(const char *type) {
	unsigned int typelen;

	read((char *)&typelen, sizeof typelen);
	if (!*this || typelen > 8192) {
	    string err;
	    err = "Expecting ";
	    err += type;
	    err += "; unable to read type length";
	    set_error(err);
	    return;
	}

	char *typebuf = new char[typelen];
	read(typebuf, typelen);
	if (!*this) {
	    string err;
	    err = "Expecting ";
	    err += type;
	    err += "; unable to read type";
	    set_error(err);
	    delete[] typebuf;
	    return;
	}

	if (memcmp(type, typebuf, typelen) != 0) {
	    string err;
	    err += "Expecting type ";
	    err += type;
	    err += " but got ";
	    err += typebuf;
	    set_error(err);
	    delete[] typebuf;
	    return;
	}

	delete[] typebuf;
    }

    void check_typelen(const char *type, unsigned int length) {
	unsigned int len;
	read((char *)&len, sizeof len);
	if (!*this) {
	    string err;
	    err = "Deserialization error: expecting ";
	    err += type;
	    err += ", unable to read data length";
	    set_error(err);
	}

	if (len != length) {
	    string err;
	    err = "Deserialization error: ";
	    err += type;
	    err += ", expecting data length ";
	    err += to_string(length);
	    err += ", got ";
	    err += to_string(len);
	    set_error(err);
	}
    }

    iserial() : my_good(true) {}
    virtual ~iserial() {}

    operator bool() const {
	return my_good && good();
    }

    string error() { return err; }
    void set_error(const string& s) {
	my_good = false;
	err = s;
    }

    virtual void read(void *bytes, unsigned int len) = 0;
    virtual bool good() const = 0;

    template<class T>
    inline iserial& operator >> (T& t) {
	unfreeze(*this, t);
	return *this;
    }
};

class iserialstream : public iserial {
    iserialstream(const iserialstream&);
    iserialstream& operator = (const iserialstream&);

    istream& in;

public:
    void read(void *bytes, unsigned int len) {
	in.read(static_cast<char *>(bytes), len);
    }
    bool good() const { return in; }

public:
    iserialstream(istream& in) : in(in) {}
};

#define NMSTL_INTEGER_UNFREEZE(T) \
inline void unfreeze(iserial& i, T& t) { \
    i.check_type(#T); \
    i.check_typelen(#T, sizeof t); \
    i.read(&t, sizeof t); \
    if (!i) \
        i.set_error("Expecting " #T ", unable to read object"); \
}
NMSTL_INTEGER_UNFREEZE(bool);
NMSTL_INTEGER_UNFREEZE(char);
NMSTL_INTEGER_UNFREEZE(unsigned char);
NMSTL_INTEGER_UNFREEZE(short);
NMSTL_INTEGER_UNFREEZE(unsigned short);
NMSTL_INTEGER_UNFREEZE(int);
NMSTL_INTEGER_UNFREEZE(unsigned int);
NMSTL_INTEGER_UNFREEZE(long);
NMSTL_INTEGER_UNFREEZE(unsigned long);

inline void unfreeze(iserial& i, string& s)
{
    i.check_type("[string]");
    int length;
    i.read(&length, sizeof length);
    if (!i) {
	i.set_error("Expecting string, unable to read length");
	return;
    }

    char *buf = new char[length];
    i.read(buf, length);
    if (!i) {
	i.set_error("Expecting string, unable to read data");
	delete[] buf;
	return;
    }

    s = string(buf, length);
    delete[] buf;
};

#endif
