
#ifndef H_COMMON
#define H_COMMON

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <sys/socket.h>

// Debugging stuff

void __ts_debug_i(int level, char *file, int line, char *fmt, ...);
extern int __ts_disable_debug;

#define ts_debug(level, fmt, args...) __ts_debug_i(level, __FILE__, __LINE__, fmt, ##args)
#define ts_error(fmt, args...) ts_debug(-1, fmt, ##args)
#define ts_debug_1(fmt, args...) ts_debug(1, fmt, ##args)
#define ts_debug_2(fmt, args...) ts_debug(2, fmt, ##args)
#define ts_debug_3(fmt, args...) ts_debug(3, fmt, ##args)
#define ts_debug_4(fmt, args...) ts_debug(4, fmt, ##args)
#define ts_fatal(fmt, args...) (ts_error(fmt, ##args), exit(-1))

// SOL_TESLA is the pseudo-level applications can use to talk to TESLA
// via setsockopt

#define SOL_TESLA_IOCTL 818
#define SO_TESLA_HELLO 1
#define SO_TESLA_MESSAGES 2
#define SO_TESLA_INESSENTIAL 3
#define SO_TESLA_SAVE_STATE 4
#define SO_TESLA_RESTORE_STATE 5
#define SO_TESLA_FD_USAGE 6

#define TS_FD_IS_UNWRAPPED 0
#define TS_FD_IS_MASTER 1
#define TS_FD_IS_MSG 2
#define TS_FD_IS_WRAPPED 3

typedef struct {
    const char *target;
    int optname;
    const void *optval;
    socklen_t optlen;
    void *retval;
    socklen_t *retlen;
} ts_ioctl_s;

typedef void (*ts_message_handler)(int fd, const char *handler, int optname, void *msg, int msglen);

typedef struct {
    ts_message_handler handler;
    int signal;
} ts_message_handler_s;

// A ts_ioctl is just a setsockopt in disguise.  This avoids having to
// override ioctl (varargs are tricky) although we may change this in
// the future.

#define ts_ioctl(fd, target, optname, optval, optlen, retval, retlen) ({ \
    ts_ioctl_s io = { target, optname, optval, optlen, retval, retlen }; \
    setsockopt(fd, SOL_TESLA_IOCTL, 0, &io, sizeof io); \
})

#define ts_enabled() (ts_ioctl(-1, 0, SO_TESLA_HELLO, 0, 0, 0, 0) == 0)

#define ts_enable_messages(handler, signal) ({ \
    ts_message_handler_s h = { handler, signal }; \
    ts_ioctl(-1, 0, SO_TESLA_MESSAGES, &h, sizeof h, 0, 0); \
})

#define ts_set_inessential(v) ({ \
    int val = (v); \
    ts_ioctl(-1, 0, SO_TESLA_INESSENTIAL, &val, sizeof val, 0, 0); \
})

#define ts_restore_state(v) ({ ts_ioctl(-1, 0, SO_TESLA_RESTORE_STATE, (v), strlen(v), 0, 0); })
#define ts_save_state(v) ({ ts_ioctl(-1, 0, SO_TESLA_SAVE_STATE, (v), strlen(v), 0, 0); })

#define ts_fd_usage(fd) ts_ioctl(fd, 0, SO_TESLA_FD_USAGE, 0, 0, 0, 0)

#ifdef __cplusplus
};
#endif

#endif
