/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: timer.hh,v 1.3 2002/08/30 20:35:37 jsalz Exp $
 *
 * A class enabling handlers to receive timed events.
 *
 */

#ifndef HH_TIMER
#define HH_TIMER

#include <string>

using namespace std;

class timer {
    long long when;
    string desc;

    // No copying allowed
    timer(const timer&);
    timer& operator=(const timer&);

public:
    enum from { FROM_EPOCH, FROM_NOW };
    static const long long CLEAR;

    timer(long long when = CLEAR, from which_base = FROM_NOW, string desc = string());
    virtual ~timer();

    void arm(long long when, from which_base = FROM_NOW);
    bool is_armed() { return when != CLEAR; }

    bool armed() { return when != CLEAR; }
    unsigned long long time() { return when; }

    string get_desc() { return desc; }

    static long long now();
    static long long fire_all();

protected:
    virtual void fire() = 0;
};


template <class H, class T = int>
class method_timer : public timer {
public:
    typedef void(H::*M)(method_timer<H,T>&);

private:
    H* obj;
    M method;
    T arg;

public:
    method_timer(H* obj, M method, T arg = T()) : obj(obj), method(method), arg(arg) {}
    T& get_arg() { return arg; }
 
protected:
    virtual void fire() { (obj->*method)(*this); }
};


template <class T = int>
class function_timer : public timer {
public:
    typedef void(*M)(function_timer<T>&);

private:
    M function;
    T arg;

public:
    function_timer(M function, T arg = T()) : function(function), arg(arg) {}
    T& get_arg() { return arg; }
 
protected:
    virtual void fire() { (*function)(*this); }
};

#endif
