/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: default_handler.cc,v 1.14 2002/10/08 19:03:33 snoeren Exp $
 *
 * The default handler, which handles communication between the master
 * process and the network.
 *
 */

#include "config.h"

#include "default_handler.hh"
#include "tesla-internal.h"

static flow_handler::info default_info("default", "default", "", -1, -1, 0);

default_handler::default_handler(flow_handler *upstream, int _fd) :
    flow_handler( init_context(default_info, *upstream, -1, -1, INT_MAX), false )
{
    fd = _fd;

    fcntl(fd, F_SETFL, O_NONBLOCK);
    set_fd(fd);
    state = CONNECTED;
    set_ractive(true);
}

default_handler::default_handler(init_context& ctxt, int _fd) : flow_handler(ctxt, false) {
    fd = _fd;

    if (fd < 0) {
	state = NONE;
	ts_debug_2("socket %d %d", ctxt.get_domain(), ctxt.get_type());
	fd = socket(ctxt.get_domain(), ctxt.get_type(), 0);
	assert(fd >= 0);
	fcntl(fd, F_SETFL, O_NONBLOCK);
	set_fd(fd);
    } else {
	fcntl(fd, F_SETFL, O_NONBLOCK);
	set_fd(fd);
	state = CONNECTED;
	set_ractive(true);
    }
}

void default_handler::ravail() {
    static char buf[MAX_DGRAM_SIZE];
    static char addr[ADDRESS_SIZE];
    static socklen_t addrlen;

    assert(state == CONNECTED || state == LISTENING);

    if (state == CONNECTED) {
	ssize_t bytes;

	errno = 0;
	if (get_type() == SOCK_DGRAM) {
	    ts_debug_2("RAVAIL DGRAM");
	    addrlen = sizeof addr;
	    bytes = ::recvfrom(fd, &buf, MAX_DGRAM_SIZE, 0,
			       (sockaddr*)addr, &addrlen);
	} else
	    bytes = ::read(fd, buf, DATA_BUFFER_SIZE);

	// If we got an error, pass it up
	if (bytes < 0) {
	    switch (errno) {
	      case EAGAIN:
		ts_debug_2("System read would block");
		return;
	      default:
		ts_debug_1("System read returned %s", strerror(errno));
		break;
		return;
	    }
	}

	if (bytes == 0) {
	    set_ractive(false);
	    state = STOP_READING;
	    upstream->avail(this, data());
	}

	if (get_type() == SOCK_DGRAM)
	    upstream->avail(this, data(buf, bytes, errno,
				       (sockaddr*)addr, addrlen));
	else
	    upstream->avail(this, data(buf, bytes, errno));
    } else {
	/* Listening */
	state = LISTENING_AVAIL;
	upstream->accept_ready(this);
	set_ractive(false);
    }
}

void default_handler::wavail() {
  
  ts_debug_2("entering wavail(), state=%d",state);
    // Only time we care about wavail
    if (state == CONNECTING) {
	// We've successfully connected.
	set_wactive(false);

	int optval;
	socklen_t optlen = sizeof optval;
	if (::getsockopt(fd, SOL_SOCKET, SO_ERROR, &optval, &optlen) == 0 && optval == 0) {
	    ts_debug_2("Connect succeeded; returning B");
	    state = CONNECTED;
	    set_ractive(true);
	    upstream->connected(this, true);
	} else {
	    ts_debug_2("Connect failed; returning A");
	    state = NONE;
	    upstream->connected(this, false);
	}
    } else if (state == CONNECTED) {
	// Data buffered.  Try to flush it

	const char *p = buffer.data();
	int len = buffer.length();
	ts_debug_2("Trying to write buffered %d bytes", len);
	while (len > 0) {
	  ts_debug_2("Entering len loop");
	    int written = ::write(fd, p, len);
	    ts_debug_2("- system write call on %d returned %d (out of %d)", fd, written, len);
	    if (written == -1 && errno == EAGAIN) {
	      ts_debug_2("  - buffering remaining %d bytes", len);
	      //buffer = string(buffer, len, buffer.length());
	      buffer = string(p, len);
		return;
	    }
	    if (written <= 0) {
		write_shutdown = true;
		buffer = string();
		set_wactive(false);
		return;
	    }

	    p += written;
	    len -= written;
	}

	upstream->may_write(this, true);
	set_wactive(false);
	buffer.resize(0);
    }
    ts_debug_2("Exiting wavail");
}

bool default_handler::avail(flow_handler *from, data d) {
    ts_debug_2("In default_handler; upstream is %p", &get_upstream());
    assert(upstream);
    return upstream->avail(this, d);
}

int default_handler::connect(address a) {

    ts_debug_2("default_handler connecting to %s", a.c_str());
    if (::connect(fd, a.addr(), a.addrlen()) == 0 || errno == EINPROGRESS) {
	ts_debug_2(" - succeeded");
	state = CONNECTING;
	set_ractive(false);
	set_wactive(true);
	return 0;
    }
    ts_debug_2(" - failed %s", strerror(errno));
    return -errno;
}
 
int default_handler::bind(address a) {
    if (state != NONE)
	return -1;

    /* XXX */
    int on = 1;
    ::setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

    ts_debug_2("default_handler binding");
    if (::bind(fd, a.addr(), a.addrlen()) == 0) {
	ts_debug_2(" - succeeded");
	if (get_type() == SOCK_DGRAM) {
	    state = CONNECTED;
	    set_ractive(true);
	} else {
	    state = BOUND;
	}
	return 0;
    }

    ts_debug_2(" - failed");

    return -errno;
}

int default_handler::close() {
    state = CLOSED;
    int ret = ::close(fd);

    set_ractive(false);
    if(ret)
      return -errno;
    else
      return 0;
}
    
int
default_handler::write(data d) {
    int len = d.length();
    const char *p = d.bits();

    if (get_type() == SOCK_DGRAM) {
	const sockaddr_in *addr = (const sockaddr_in *)d.addr();
	socklen_t addrlen = d.addrlen();

	if (addrlen == 0)
	    addr = 0;

	int res = ::sendto(fd, d.bits(), d.length(), 0, (const sockaddr *)addr, addrlen);

	ts_debug_2("- system sendto call on %d returned %d", fd, res);
	if(res == -1)
	    return -errno;
	else
	    return 0;
    }
    
    if (buffer.size() != 0) {
	buffer.append(d.bits(), d.length());
	return 0;
    }

    while (len > 0) {
	int written = ::write(fd, p, len);

	ts_debug_2("- system write call on %d returned %d (out of %d)", fd, written, len);

	if (written == -1 && errno == EAGAIN) {
	    ts_debug_2("  - buffering remaining %d bytes", len);

	    // buffer the rest, select on write, and throttle upstream
	    buffer.append(p, len);
	    set_wactive(true);
	    upstream->may_write(this, false);
	    return 0;
	}
	if (written <= 0) {
	    ts_debug_2("  - closed for writing");

	    // closed for writing
	    return -errno;
	}

	ts_debug_2("  - wrote %d bytes successfully", written);
	
	p += written;
	len -= written;
    }
    
    return 0;
}
 
int default_handler::shutdown(bool r, bool w) {
  if (r || w) {
    int ret = ::shutdown(fd, r && w ? SHUT_RDWR : r ? SHUT_RD : SHUT_WR);
    if(ret)
      return -errno;
    else
      return 0;
  }

  return -EINVAL;
}

int default_handler::listen(int backlog) {
    // XXX!
    if (state != BOUND)
	return -1;

    if (::listen(fd, backlog) == 0) {
	state = LISTENING;
	ts_debug_2("State is LISTENING!");

	set_ractive(true);
	set_wactive(false);
	return 0;
    }
    return -errno;
}

address default_handler::getsockname() {
    sockaddr addr;
    socklen_t len = sizeof addr;
    int ret = ::getsockname(fd, &addr, &len);

    return ret == 0 ? address(&addr, len) : address();
}

address default_handler::getpeername() {
    sockaddr addr;
    socklen_t len = sizeof addr;
    int ret = ::getpeername(fd, &addr, &len);

    return ret == 0 ? address(&addr, len) : address();
}


acceptret default_handler::accept() {
    ts_debug_2("default_handler accepting on fd %d", fd);

    if (state != LISTENING_AVAIL) {
	ts_debug_2(" - nothing available!");
	return acceptret();
    }

    sockaddr addr;
    socklen_t addrlen = sizeof addr;

    int newfd = ::accept(fd, &addr, &addrlen);
    if (newfd < 0) {
	ts_debug_2(" - nope");
	return acceptret();
    }

    state = LISTENING;

    init_context ctxt(get_info(), *(flow_handler*)0, get_domain(), get_type(), get_level());
    default_handler *h = new default_handler(ctxt, newfd);

    set_ractive(true);
    set_wactive(false);

    ts_debug_2(" - yup! accepted %d", newfd);

    return acceptret(h, address(&addr, addrlen));
}

void default_handler::may_avail(bool may) {
    set_ractive(may);
}

string default_handler::getsockopt(int level, int optname, int maxlen) {
    assert(maxlen >= 0);

    void *buf = maxlen ? malloc(maxlen) : 0;
    socklen_t len = maxlen;
    int ret = ::getsockopt(fd, level, optname, buf, &len);

    if (maxlen && ret == 0) {
	string ret(static_cast<char *>(buf), len);
	free(buf);
	return ret;
    }

    free(buf);
    return string();
}

int default_handler::setsockopt(int level, int optname, string value) {
  int ret = ::setsockopt(fd, level, optname, value.data(), value.length()); 
  
  if(ret)
    return -errno;
  else
    return 0;
}

int default_handler::ioctl(string target, int optname, string value, string& out) {
    ts_debug_1("IOCTL!!!");
    if (target == "default") {
	char ch[16];
	sprintf(ch, "%d", optname);
	out += "IOCTL<";
	out += ch;
	out += ":";
	out += value;
	out += ">";
	return 0;
    }
    return -EINVAL;
}

void default_handler::dump(string& out, int indent = 0) {
  for (int i = 0; i < indent; ++i) {
    out.append("   ");
  }
  out.append(" - default_handler; state is %d");
#if 0
  out.append("; getsockname is ");
  out.append(getsockname().c_str());
  out.append("; getpeername is ");
  out.append(getpeername().c_str());
#endif
  out.append("\n");
}
