/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: flow_handler.cc,v 1.27 2002/10/08 19:03:33 snoeren Exp $
 *
 * The base class for all handlers.
 *
 */

#include "config.h"

#include "tesla/flow_handler.hh"

#include "default_handler.hh"
#include "top_handler.hh"

#include <iostream>
#include <vector>

#include "tesla/tesla.h"

using namespace std;

flow_handler::infovec flow_handler::handlers;
flow_handler::infomap *flow_handler::reg_handlers;
double flow_handler::nan = 0.0 * HUGE_VAL;

void flow_handler::register_handler(const info& inf)
{
    // Static initialization trick
    if (reg_handlers == 0) {
	ts_debug_2("Allocating handlers vector");
	reg_handlers = new infomap;
    }
    ts_debug_2("Registering flow_handler %s (domain=%d, type=%d)",
	       inf.get_name().c_str(), inf.get_domain(), inf.get_type());
    (*reg_handlers)[inf.get_name()] = &inf;
}

void flow_handler::make_handlers()
{
    const char *h1 = getenv("TS_HANDLERS");
    if (!h1)
	ts_fatal("No TS_HANDLERS environment variable");

    bool usage_only = getenv("TS_USAGE") != 0;
    if (usage_only && !*h1) {
	// Zero-length handlers; display available handlers
	if (!reg_handlers) {
	    fprintf(stderr, "(none)\n");
	    exit(0);
	}		

	for (infomap::const_iterator i = reg_handlers->begin();
	     i != reg_handlers->end();
	     ++i)
	{
	    if (i != reg_handlers->begin())
		fprintf(stderr, " ");
	    fprintf(stderr, "%s", i->first.c_str());
	}
	fprintf(stderr, "\n");
	exit(0);
    }


    char *h = strdup(h1);
    char *hh = h;
    char *tok;
    
    if (reg_handlers == 0) {
	ts_error("No handlers registered");
	return;
    }

    int count = 0;
    while ((tok = strtok(hh, ":")) != 0) {
	infomap::const_iterator i = reg_handlers->find(tok);
	if (i == reg_handlers->end()) {
	    ts_error("Unknown flow_handler %s", tok);
	} else {
	    char ch[32];
	    sprintf(ch, "TS__%d", count);
	    const char *arg = getenv(ch);

	    info *inf = new info(*(i->second));

	    if (arg) {
		// Read arguments
		while (*arg) {
		    string key;
		    string value;

		    // Read key
		    while (*arg && *arg != '=') {
			if (*arg == '\\') ++arg;
			if (*arg) key += *arg++;
		    }
		    if (*arg) ++arg; // Skip =

		    // Read value
		    while (*arg && *arg != ';') {
			if (*arg == '\\') ++arg;
			if (*arg) value += *arg++;
		    }
		    if (*arg) ++arg; // Skip ;

		    inf->get_args()[key] = value;
		}
	    }

	    ts_debug_2("Pushing flow_handler %s (%s) on stack", tok, inf->get_clazz().c_str());
	    if (usage_only) {
		string usage = inf->get_usage();
		if (usage.length())
		    fprintf(stderr, "*** Usage for %s:\n\n%s\n", inf->get_name().c_str(), usage.c_str());
		else
		    fprintf(stderr, "*** No usage available for %s\n", inf->get_name().c_str());
	    } else
		handlers.push_back(inf);
	}

	++count;
	hh = 0;
    }
    free(h);

    if (usage_only)
	exit(0);
}

flow_handler::info default_info("default", "default", "", -1, -1, 0);

flow_handler::flow_handler(const init_context& ctxt, bool plumb) : inf(ctxt.get_info())
{
    upstream = &ctxt.get_upstream();
    domain = ctxt.get_domain();
    type = ctxt.get_type();
    level = ctxt.get_level();

    // Turn things on by default
    can_write = can_avail = true;
    
    if (plumb) {
	if (!ctxt.serial() && !ctxt.is_accepted())
	    downstream.push_back(ctxt.plumb(*this, ctxt.get_domain(), ctxt.get_type()));
    }
}

flow_handler *flow_handler::plumb(int domain, int type, int fd)
{
    if (fd < 0)
	return plumb(*this, domain, type, get_level());

    init_context ctxt(default_info, *this, domain, type, handlers.size());
    return new default_handler(ctxt, fd);    
}

acceptret flow_handler::accept()
{
    assert(downstream.size() == 1);
    assert(inf.get_cf());
    acceptret h = downstream[0]->accept();
    if (!h)
	return acceptret();

    init_context ctxt(inf, *(flow_handler*)0, domain, type, level, true);
    flow_handler *s = inf.create(ctxt);
    s->downstream.push_back(h.h);
    h.h->upstream = s;
    
    return acceptret(s, h.addr);
}

flow_handler *flow_handler::plumb(flow_handler& upstream, int domain, int type, int level)
{
    if (handlers.size() == 0) return 0;

    ts_debug_2("plumb, upstream=%p, domain=%d, type=%d, level=%d",
	       &upstream, domain, type, level);

    for (unsigned int i = level; i < handlers.size(); ++i) {
	ts_debug_2("  - i=%d of %d; d,t=%d,%d", i, handlers.size(), handlers[i]->get_domain(), handlers[i]->get_type());
	if (handlers[i]->get_domain() == domain &&
	    handlers[i]->get_type() == type)
	{
	    ts_debug_2("    - got it!");
	    init_context ctxt(*(handlers[i]), upstream, domain, type, i + 1);
	    return handlers[i]->create(ctxt);
	}
    }

    if (level == 0) return 0;

    init_context ctxt(default_info, upstream, domain, type, handlers.size());
    return new default_handler(ctxt);
}

int flow_handler::connect(address a) {
    assert(downstream.size() == 1);
    return downstream[0]->connect(a);
}
int flow_handler::bind(address a) {
    assert(downstream.size() == 1);
    return downstream[0]->bind(a);
}

acceptret flow_handler::accept();
int flow_handler::close() {
    assert(downstream.size() == 1);
    return downstream[0]->close();
}

void flow_handler::accept_ready(flow_handler *from) {
    assert(upstream && downstream.size() == 1 && from == downstream[0]);
    upstream->accept_ready(this);
}
void flow_handler::connected(flow_handler *from, bool success) {
    assert(upstream && downstream.size() == 1 && from == downstream[0]);
    upstream->connected(this, success);
}
bool flow_handler::avail(flow_handler *from, data d) {
    assert(upstream);
    assert(downstream.size() == 1);
    assert(from == downstream[0]);
    return upstream->avail(this, d);
}
int flow_handler::write(data d) {
    assert(downstream.size() == 1);
    return downstream[0]->write(d);
}
int flow_handler::shutdown(bool r, bool w) {
    assert(downstream.size() == 1);
    return downstream[0]->shutdown(r,w);
}
int flow_handler::listen(int backlog) {
    assert(downstream.size() == 1);
    return downstream[0]->listen(backlog);
}

address flow_handler::getsockname() {
    assert(downstream.size() == 1);
    return downstream[0]->getsockname();
}
address flow_handler::getpeername() {
    assert(downstream.size() == 1);
    return downstream[0]->getpeername();
}

string flow_handler::getsockopt(int level, int optname, int maxlen) {
    assert(downstream.size() == 1);
    return downstream[0]->getsockopt(level, optname, maxlen);
}
int flow_handler::setsockopt(int level, int optname, string value) {
    assert(downstream.size() == 1);
    return downstream[0]->setsockopt(level, optname, value);
}
int flow_handler::ioctl(string target, int optname, string value, string& out) {
    if (downstream.empty()) return -EINVAL;
    int err = 0;

    for (vector<flow_handler*>::const_iterator i = downstream.begin(); i != downstream.end(); ++i) {
	out = "";
	err = (*i)->ioctl(target, optname, value, out);
    }
    return err;
}

void flow_handler::may_write(flow_handler *from, bool may) {
    can_write = may;
    upstream->may_write(this, may);
}

void flow_handler::may_avail(bool may) {
    can_avail = may;
    for (vector<flow_handler*>::const_iterator i = downstream.begin(); i != downstream.end(); ++i)
	(*i)->may_avail(may);
}

bool flow_handler::may_exit() {
    bool all_may_exit = true;

    for (vector<flow_handler*>::const_iterator i = downstream.begin(); i != downstream.end(); ++i) {
	if (!(*i)->may_exit())
	    all_may_exit = false;
    }

    return all_may_exit;
}

extern int messages_enabled;

extern void send_message(unsigned int conn_id, string handler, int name, string value);
extern void send_message_fd(unsigned int conn_id, string handler, int name, int the_fd);

void flow_handler::message(int msgname, string value) {
    if (!messages_enabled()) return;

    flow_handler *p = this;
    while (p->upstream) p = p->upstream;

    top_handler *t = static_cast<top_handler *>(p);

    ::send_message(t->get_conn_id(), get_info().get_name(), msgname, value);
}

void flow_handler::message_fd(int msgname, int fd) {
    if (!messages_enabled()) return;

    flow_handler *p = this;
    while (p->upstream) p = p->upstream;

    top_handler *t = static_cast<top_handler *>(p);

    ::send_message_fd(t->get_conn_id(), get_info().get_name(), msgname, fd);
}

bool flow_handler::get_flag(string key) const
{
    int val;
    get_arg(key, val, 0);
    return val != 0;
}

bool flow_handler::get_arg(string key, string& val, string def) const
{
    dict::const_iterator i = inf.get_args().find(key);
    if (i == inf.get_args().end()) {
	val = def;
	return false;
    }
    val = i->second;
    return true;
}

bool flow_handler::get_arg(string key, int& val, int def, int min, int max) const
{
    string sval;
    if (!get_arg(key, sval)) {
	val = def;
	return false;
    }

    char *eptr;
    val = strtol(sval.c_str(), &eptr, 0);
    if (*eptr) {
	ts_error("Invalid argument for %s parameter", key.c_str());
	val = def;
    } else if (!(val >= min && val <= max)) {
	ts_error("Argument for %s parameter must be between %d and %d",
		 key.c_str(), min, max);
	val = def;
    }

    return true;
}

bool flow_handler::get_arg(string key, double& val, double def, double min, double max) const
{
    string sval;
    if (!get_arg(key, sval)) {
	val = def;
	return false;
    }

    char *eptr;
    val = strtod(sval.c_str(), &eptr);
    if (*eptr) {
	ts_error("Invalid argument for %s parameter", key.c_str());
	val = def;
    } else if (!(val >= min && val <= max)) {
	ts_error("Argument for %s parameter must be between %f and %f",
		 key.c_str(), min, max);
	val = def;
    }

    return true;
}

int flow_handler::__set_usage_info(info& i, string usage) {
    i.usage = usage;
}

#include <typeinfo>
void freeze(oserial& out, const flow_handler& h) {

    ts_debug_1("Freezing a %s", h.get_info().get_name().c_str());
    out << h.get_info().get_name();
    out << h.can_write << h.can_avail;

    if (!h.save_state(out)) {
	if (out) {
	    string err = "Unable to serialize handler of type ";
	    err += h.get_info().get_name();
	    out.set_error(err);
	}
    }
}

void unfreeze(iserial& in, flow_handler*& h) {
    ts_debug_1("Unfreezing a flow handler");
    string name;
    bool can_write, can_avail;

    in >> name;
    in >> can_write >> can_avail;

    ts_debug_1(" - it's a %s", name.c_str());

    if (name == "top_handler") {
	// Special case - create its downstream first
	flow_handler *next;
	in >> next;
	if (!in) return;

	h = new top_handler(-1, next->get_info().get_type(), 0, next);
	next->upstream = h;
	h->can_write = can_write;
	h->can_avail = can_avail;

	return;
    }	

    // Find the handler by name
    flow_handler::infomap::const_iterator i = flow_handler::reg_handlers->find(name);
    if (i == flow_handler::reg_handlers->end()) {
	ts_debug_1(" - unable to find it!");
	in.set_error(string("Unable to find handler named ") + name);
	return;
    }

    ts_debug_1(" - found it!  Instantiating...");
    const flow_handler::info &inf = *(i->second);
    
    flow_handler::init_context init(inf, in);
    h = inf.create(init);
    h->can_write = can_write;
    h->can_avail = can_avail;
}
