/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: libc-dgram.c,v 1.5 2002/09/04 16:59:08 snoeren Exp $
 *
 * Datagram-related libc overrides for the libtesla.so wrapper library.
 * #included by tesla.c.
 *
 */

int read(int fd, void *buf, size_t len) {
    if (tesla_enabled(0) && ts_fds[fd].wrapped &&
	((ts_fds[fd].type == SOCK_DGRAM) || ts_fds[fd].connecting ||
	ts_fds[fd].connect_status)) {
      // If connect failed, report that now
      if((errno = ts_fds[fd].connect_status))
	return -1;
      // We read an async socket that hasn't yet connected
      if(ts_fds[fd].connecting)
	ts_fatal("Read without select not yet supported");
      return recvfrom(fd, buf, len, 0, 0, 0);
    } else {
      return TS_CALL_LIBC(read, fd, buf, len);
    }

}

int recv(int fd, void *buf, size_t len, int flags) {
    if (tesla_enabled(0) && ts_fds[fd].wrapped &&
	((ts_fds[fd].type == SOCK_DGRAM) || ts_fds[fd].connecting ||
	 ts_fds[fd].connect_status)) {
      // If connect failed, report that now
      if((errno = ts_fds[fd].connect_status))
	return -1;
      // We read an async socket that hasn't yet connected
      if(ts_fds[fd].connecting)
	ts_fatal("Recv without select not yet supported");
      return recvfrom(fd, buf, len, flags, 0, 0);
    } else {
      return TS_CALL_LIBC(recv, fd, buf, len, flags);
    }
}


int
recvfrom(int fd, void *buf, size_t count, int flags,
	 struct sockaddr *from, socklen_t *fromlen)
{
    static ts_dgram_header dg;
    static char garbage[1024];
    int bytes;
    int fflags, ret = 0;

    if (tesla_enabled(0) && !ts_fds[fd].wrapped &&
	((ts_fds[fd].type == SOCK_DGRAM) || ts_fds[fd].connecting ||
	 ts_fds[fd].connect_status)) {

      // If connect failed, report that now
      if((errno = ts_fds[fd].connect_status))
	return -1;
      // We read an async socket that hasn't yet connected
      if(ts_fds[fd].connecting)
	ts_fatal("Recvfrom without select not yet supported");

    if (flags)
	ts_fatal("TESLA doesn't support recvfrom/sendto flags");

    bytes = TS_CALL_LIBC(read, fd, &dg, 1/*XXX: sizeof dg*/);
    if (bytes <= 0) return bytes;

    fflags = fcntl(fd, F_GETFL);
    fcntl(fd, F_SETFL, fflags & ~O_NONBLOCK);

    if (bytes != sizeof dg) {
	if (!read_fully(fd, ((char*)&dg) + bytes, sizeof dg - bytes)) {
	    ts_error("Incomplete header from dgram channel");
	    goto out;
	}
    }

    if (count > dg.length)
	count = dg.length;

    if (!read_fully(fd, buf, count)) {
	ts_error("Incomplete data from dgram channel");
	goto out;
    }
    ret = count;

    dg.length -= count;
    while (dg.length) {
	ts_debug_1("Dgram buffer too small - discarding %d bytes", dg.length);
	bytes = dg.length < sizeof garbage ? dg.length : sizeof garbage;
	if (!read_fully(fd, garbage, bytes)) {
	    ts_error("Incomplete data from dgram channel");
	    goto out;
	}
	dg.length -= bytes;
    }

    if (from && fromlen) {
	if (*fromlen > dg.addrlen)
	    *fromlen = dg.addrlen;
	memcpy(from, &dg.addr, *fromlen);
    }

  out:
    fcntl(fd, F_SETFL, fflags);
    return ret;

    } else {
      return TS_CALL_LIBC(recvfrom, fd, buf, count, flags, from, fromlen);
    }
}


int write(int fd, const void *buf, size_t len) {
    if (!tesla_enabled(0) || !ts_fds[fd].wrapped || ts_fds[fd].type != SOCK_DGRAM)
	return TS_CALL_LIBC(write, fd, buf, len);
    return sendto(fd, buf, len, 0, 0, 0);
}


int send(int fd, const void *buf, size_t len, int flags) {
    if (!tesla_enabled(0) || !ts_fds[fd].wrapped || ts_fds[fd].type != SOCK_DGRAM)
	return TS_CALL_LIBC(send, fd, buf, len, flags);
    return sendto(fd, buf, len, flags, 0, 0);
}


int
sendto(int fd, const void *buf, size_t count, int flags,
       const struct sockaddr *to, socklen_t tolen)
{
    static ts_dgram_header dg;
    int fflags, ret = 0;
    int bytes;

    if (!tesla_enabled(0) || !ts_fds[fd].wrapped || ts_fds[fd].type != SOCK_DGRAM)
	return TS_CALL_LIBC(sendto, fd, buf, count, flags, to, tolen);

    if (flags)
	ts_fatal("TESLA doesn't support recvfrom/sendto flags");

    dg.length = count;
    dg.addrlen = tolen;

    if (tolen > sizeof dg.addrlen)
	tolen = sizeof dg.addrlen;
    if (tolen != 0)
	memcpy(&dg.addr, to, tolen);

    bytes = TS_CALL_LIBC(write, fd, &dg, 1 /*XXX: sizeof dg*/);
    if (bytes <= 0)
	return bytes;

    fflags = fcntl(fd, F_GETFL);
    fcntl(fd, F_SETFL, fflags & ~O_NONBLOCK);

    if (bytes != sizeof dg) {
	if (!write_fully(fd, ((const char *)&dg) + bytes, sizeof dg - bytes)) {
	    ts_error("Unable to write_fully dgram packet header");
	    goto out;
	}
    }

    if (!write_fully(fd, buf, count))
	ts_error("Unable to write_fully dgram packet data");

    ret = count;

  out:
    fcntl(fd, F_SETFL, fflags);
    return ret;
}


/* XXX: The recv/sendmsg commands are much trickier */

int
recvmsg(int fd, struct msghdr *msg, int flags)
{
    if (!tesla_enabled(0) || !ts_fds[fd].wrapped)
	return TS_CALL_LIBC(recvmsg, fd, msg, flags);
    
    ts_fatal("TESLA doesn't yet support sendmsg/recvmsg");
    return -1;
}

int
sendmsg(int fd, const struct msghdr *msg, int flags)
{
    if (!tesla_enabled(0) || !ts_fds[fd].wrapped)
	return TS_CALL_LIBC(sendmsg, fd, msg, flags);

    ts_fatal("TESLA doesn't yet support sendmsg/recvmsg");
    return -1;
}

