/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: state.c,v 1.3 2002/10/02 17:37:19 snoeren Exp $
 *
 * Functions to save and restore state.
 *
 */

static int save_state(const char *file)
{
    int scratch;
    master_msg_t msg;
    unsigned int id;
    FILE *out;

    ts_debug_1("Saving state to %s", file);

    out = fopen(file, "w");
    if (!out)
	return 0;

#define WRITE(buf, len) do { if (len && fwrite(buf, len, 1, out) != 1) { return 0; } } while (0)
#define WRITE_INT(val) do { scratch = val; WRITE(&scratch, sizeof scratch); } while (0)

    ts_debug_1("Saving state");

    // Write format version number (1)
    WRITE_INT(1);

    // Write FD table
    WRITE(&ts_fds, sizeof ts_fds);

    fclose(out);

    // Tell master to serialize its state too
    msg.type = MSG_SAVE_STATE;
    strcpy(msg.body.save_state.filename, file);
    id = send_master(&msg);
    if (recv_master(&msg) != id)
	ts_fatal("Mis-sequenced response");
    if (msg.type != MSG_ACK) {
	ts_debug_1("Master was unable to serialize");
	errno = EINVAL;
	return 0;
    }

    ts_debug_1("Closing master");

    // Master's goose is cooked
    TS_CALL_LIBC(close, master_fd);
    master_fd = -1;

    return 1;
}

static int restore_state(const char *file)
{
    unsigned int id;
    int scratch;
    master_msg_t msg;
    int bytes;
    int i;
    int fd;

    if (is_suid)
	ts_fatal("Cannot restore state while SUID");

    ts_debug_1("Restoring state from %s", file);

    fd = open(file, O_RDONLY);
    if (fd < 0)
	return 0;

    // Read format version number
    bytes = read(fd, &scratch, sizeof scratch);
    if (bytes < sizeof scratch)
	ts_fatal("Unable to format version number");
    if (scratch != 1)
	ts_fatal("Wrong version number");

    // Read FD table
    if (read(fd, &ts_fds, sizeof ts_fds) != sizeof ts_fds)
	ts_fatal("Unable to read ts_fds");

    // Launch master
    enable(fd);
    close(fd);

    // Enable returns only if master is alive (after it has restored all
    // its state)
    ts_debug_1("Master is alive");

    // Retrieve wrapped descriptors from master.
    for (i = 0; i < FD_SETSIZE; ++i) {
	if (ts_fds[i].wrapped) {
	    ts_debug_1("Asking master for wrapped FD %d (conn id %d)", i, ts_fds[i].conn_id);
	    msg.type = MSG_GET_FD;
	    msg.conn_id = ts_fds[i].conn_id;

	    id = send_master(&msg);
	    if (recv_master(&msg) != id)
		ts_fatal("Mis-sequenced response");

	    if (msg.type == MSG_FH_FOLLOWS) {
		int newfd = msg.body.fh_pass.fh;

		ts_debug_1("Master has provided wrapped FD %d as %d", i, newfd);
		if (newfd != i) {
		    if (master_fd == i) {
			master_fd = TS_CALL_LIBC(dup, master_fd);
			TS_CALL_LIBC(close, i);
		    }

		    TS_CALL_LIBC(dup2, newfd, i);
		    TS_CALL_LIBC(close, newfd);
		}
	    } else {
		ts_debug_1("Master cannot provide wrapped FD %d; unwrapping it", i);
		ts_fds[i].wrapped = 0;
	    }
	}
    }

    ts_debug_1("TESLA enabled!");

    is_enabled = 1;
    __ts_disable_debug = 0;

    ts_debug_1("Done restoring state");
    return 1;
}
