/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: tesla-internal.h,v 1.22 2002/09/30 16:57:52 jsalz Exp $
 *
 * Macros and declarations used inside TESLA code.
 *
 */

#ifndef H_TESLA_INTERNAL
#define H_TESLA_INTERNAL

#include <stdio.h>
#include <sys/socket.h>

#define ADDRESS_SIZE 256
#define DATA_BUFFER_SIZE 8192
#define MAX_DGRAM_SIZE 65536
#define MAX_OPTVAL_SIZE 1024
/* assert(MAX_DGRAM_SIZE > DATA_BUFFER_SIZE) */

#ifdef __cplusplus
extern "C" {
#endif

// Structure to track information about a single socket.

typedef struct
{
    int wrapped;  // Is it wrapped?
    int domain;   // e.g., PF_INET
    int type;     // e.g., SOCK_DGRAM
    int conn_id;  // connection ID used when communicating with master
    int connecting;
    int connect_status;
} ts_fd_info;
extern ts_fd_info ts_fds[FD_SETSIZE];

typedef struct
{
    size_t    length;
    socklen_t addrlen;
    struct sockaddr addr;
} ts_dgram_header;

// Stuff about application<->master communication.

typedef enum {
    MSG_HELLO,
    MSG_SOCKET,
    MSG_CONNECT,
    MSG_FH_FOLLOWS,
    MSG_ACK,
    MSG_NAK,
    MSG_BIND,
    MSG_LISTEN,
    MSG_ACCEPT,
    MSG_GETSOCKOPT,
    MSG_SETSOCKOPT,
    MSG_GETPEERNAME,
    MSG_GETSOCKNAME,
    MSG_ASYNC_MESSAGES,
    MSG_FORK,
    MSG_INESSENTIAL,
    MSG_SAVE_STATE,
    MSG_GET_FD,

    MSG_TYPE_COUNT
} master_msg_type_t;

extern const char *__ts_msg_types[];
extern const int __ts_msg_type_count;

#define MSG_TYPE(i) ((i) >= 0 && (i) < __ts_msg_type_count ? \
                     __ts_msg_types[(i)] : "UNKNOWN")

typedef struct {
    char handler[32];
    unsigned int conn_id;
    int name;
    int data_length;
} ts_async_message_s;

typedef struct {
    size_t len;
    unsigned int id;
    unsigned int conn_id;
    pid_t pid;
    unsigned int merrno;
    master_msg_type_t type;

    union {
	struct {
	} hello;
	struct {
	    int domain, type, protocol;
	} socket;
	struct {
	    int fh;
	} fh_pass;
	struct {
	    int addrlen;
	    char addr[ADDRESS_SIZE]; /* XXX: Copout */
	} address;
	struct {
	    int level;
	    int optname;
	    int optlen;
	    char target[32];
	    char optval[1024]; /* XXX: Also a copout */
	} sockopt;
	struct {
	    int enabled;
	    int pass;
	} async_messages;
	struct {
	    int val;
	} inessential;
	struct {
	    char filename[1024];
	} save_state;
	struct {
	    int fd;
	} get_fd;
    } body;
} master_msg_t;

#define OFFSET(str, mem) (((char*)&(((str*)0)->mem)) - (char*)0)
#define OFFSET_FROM(str, mem1, mem2) (OFFSET(str, mem2) - OFFSET(str, mem1))
#define MASTER_MSG_HDRLEN OFFSET(master_msg_t, body)



// Macros to call libc functions directly.

extern void *__ts_libc;

#ifdef RTLD_NEXT
# define LIBC_NEXT RTLD_NEXT
#else
# define LIBC_NEXT LIBC_NAME
#endif

#define TS_CALL_LIBC(func, args...) \
({ \
    static typeof(func) *f = NULL; \
    if (!f) { f = dlsym(LIBC_NEXT, #func); if (!f) { fprintf(stderr, "dlsym for %s failed", #func); exit(1); } } \
    (*f)(args); \
})

// Variables used for debugging
extern int __ts_debug_level;
extern FILE *__ts_debug_file;
extern void __ts_debug_init();

#ifdef __cplusplus
};
#endif

#endif
