/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: tesla.c,v 1.68 2002/09/25 04:17:14 jsalz Exp $
 *
 * The source for the libtesla.so wrapper library.
 *
 */

#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <dlfcn.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <setjmp.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/uio.h>
#include <signal.h>

#include "tesla-internal.h"
#include "tesla.h"

// Path to the master application.  LIBDIR must be defined on the command line
#define TESLA_MASTER_PATH LIBDIR "/tesla/teslamaster"

/***
 *
 * Externally visible routines.
 *
 ***/

// The following libc functions are redefined by TESLA
int accept(int fd, struct sockaddr *addr, socklen_t *addrlen);
int bind(int fd, const struct sockaddr *addr, socklen_t addrlen);
int close(int fd);
int connect(int fd, const struct sockaddr *addr, socklen_t addrlen);
int listen(int fd, int backlog);
int socket(int domain, int type, int protocol);
int getsockopt(int s, int level, int optname, void *optval, socklen_t *optlen);
int setsockopt(int s, int level, int optname, const void *optval,
	       socklen_t optlen);
int dup(int fd);
int dup2(int fd, int newfd);
pid_t vfork();
int select(int n, fd_set *rfds, fd_set *wfds, fd_set *efds,
	   struct timeval *timeout);

// These are overloaded by TESLA only for SOCK_DGRAM sockets
int read(int fd, void *buf, size_t len);
int recv(int fd, void *buf, size_t len, int flags);
int recvfrom(int fd, void *buf, size_t count, int flags,
	     struct sockaddr *from, socklen_t *fromlen);
int write(int fd, const void *buf, size_t len);
int send(int fd, const void *buf, size_t len, int flags);
int sendto(int fd, const void *buf, size_t count, int flags,
	   const struct sockaddr *to, socklen_t tolen);

// These should be overloaded by TESLA in the future
int recvmsg(int fd, struct msghdr *msg, int flags);
int sendmsg(int fd, const struct msghdr *msg, int flags);

/***
 *
 * Utility routines.
 *
 ***/

// Receive a FD from the master process over master_fd.
static int get_fd_from_master();

// Read count bytes from a file descriptor.
static int read_fully(int fd, void *buf, size_t count);

// Write count bytes to a file descriptor.
static int write_fully(int fd, const void *buf, size_t count);

// Send a message to the master.
static unsigned int send_master(master_msg_t *msg);

// Send just a header to the master (the data length is len).
static unsigned int send_master_header(master_msg_t *msg, int len);

// Receive a message from the master.
static unsigned int recv_master(master_msg_t *msg);

// Clean up on application exit.
static void cleanup(void);

// Initialize TESLA.
static void init() __attribute__((constructor));

// Set the message handler used for async messages from handlers.
static int set_message_handler(ts_message_handler handler, int sig);

// The sigaction handler for async messages.
static void msg_sigaction(int signal, siginfo_t *info, void *arg);

// State stuff
static int save_state(const char *file);
static int restore_state(const char *file);

/***
 *
 * Internal state.
 *
 ***/

// Info about file descriptors
static ts_fd_info ts_fds[FD_SETSIZE];

// FD used to communicate with the master process - synchronous
// communication only
static int master_fd = -1;

// ID of the next message to send to the master process
static unsigned int next_msg_id = 1;

// FD used to receive asynchronous messages from the master process
static int msg_fd = -1;

// Callback used to deliver asynchronous messages to the application
static ts_message_handler msg_handler = 0;

// Signal used for O_ASYNC I/O handling on msg_fd
static int msg_signal = 0;

// If is_suid, require that the euid is require_uid before doing any
// wrapping
static int is_suid = 0;
static uid_t require_uid;

// Tesla enabled yet?
static int is_enabled = 0;

#include "utility.c"
#include "libc.c"
#include "libc-dgram.c"
#include "state.c"
